/*
 * @(#)RubberSelectionBox.cpp 1.00 11 July 2000
 *
 * Copyright (c) Pete Goodliffe 2000 (pete@cthree.org)
 *
 * This file is part of anthem - the TSE3 sequencer.
 *
 * This program is modifiable/redistributable under the terms of the GNU
 * General Public License.
 *
 * You should have recieved a copy of the GNU General Public License along
 * with this program; see the file COPYING. If not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 0211-1307, USA.
 */

#include "gadgets/RubberSelectionBox.h"

#include "misc/kde2-compat.h"
#include <qtimer.h>
#include <iostream>
#include <qpainter.h>
#include <qpen.h>
#include <qcursor.h>
#include <qstyle.h>

namespace
{
    const size_t TIMER_MSECS = 20;
}

RubberSelectionBox::RubberSelectionBox(QScrollView *parent, int border)
: parent(parent), _filter(0), _dragging(false),
  pointerInAutoScrollBorder(false), border(border), boxOffScreen(false)
{
}


RubberSelectionBox::~RubberSelectionBox()
{
}


void RubberSelectionBox::startAnchoredDrag(int x, int y, int width, int height,
                                           int private_word = 0,
                                           bool autoScroll,
                                           bool constrain,
                                           int minx, int miny,
                                           int minwidth, int minheight,
                                           int maxx, int maxy,
                                           int maxwidth, int maxheight)
{
    if (!_dragging)
    {
        this->x            = x;
        this->y            = y;
        this->width        = width;
        this->height       = height;
        this->private_word = private_word;
        this->anchored     = true;
        this->autoScroll   = autoScroll;
        this->constrain    = constrain;
        this->minx         = minx;
        this->miny         = miny;
        this->minwidth     = minwidth;
        this->minheight    = minheight;
        this->maxx         = maxx;
        this->maxy         = maxy;
        this->maxwidth     = maxwidth;
        this->maxheight    = maxheight;
        _dragging          = true;
        doconstrain_anchored();
        draw();
    }
}


void RubberSelectionBox::startDragBox(int x, int y,
                                      int private_word = 0,
                                      int boxx, int boxy,
                                      int boxwidth, int boxheight,
                                      bool autoScroll,
                                      bool constrain,
                                      int maxx, int maxy,
                                      int maxwidth, int maxheight)
{
    if (!_dragging)
    {
        this->x            = boxx;
        this->y            = boxy;
        this->width        = boxwidth;
        this->height       = boxheight;
        this->private_word = private_word;
        this->deltax       = x - boxx;
        this->deltay       = y - boxy;
        this->anchored     = false;
        this->autoScroll   = autoScroll;
        this->constrain    = constrain;
        this->maxx         = maxx;
        this->maxy         = maxy;
        this->maxwidth     = maxwidth;
        this->maxheight    = maxheight;
        _dragging          = true;
        doconstrain_box();
        draw();
    }
}


void RubberSelectionBox::stop(bool signal)
{
    if (_dragging)
    {
        draw();
        _dragging = false;
        if (signal)
        {
            emit dragBoxCompleted(x, y, width, height, private_word, anchored);
        }
    }
}


void RubberSelectionBox::draw(QPainter *p)
{
    if (_dragging && !boxOffScreen)
    {
        bool destroyPainter = false;
        if (!p)
        {
            p = new QPainter;
            p->begin(parent->viewport());
            p->setRasterOp(NotROP);
            p->setPen(QPen(QPen(color0, 1)));
            p->setBrush(NoBrush);
            destroyPainter = true;
        }

        QPoint pnt = parent->contentsToViewport(QPoint(display_x, display_y));
#if KDE_VERSION < 300
        parent->style().drawFocusRect(p, QRect(pnt.x(), pnt.y(),
                                      display_width, display_height),
                                      parent->colorGroup(),
                                      &parent->colorGroup().base());
#else
        parent->style().drawPrimitive(QStyle::PE_FocusRect, p,
                                      QRect(pnt.x(), pnt.y(),
                                          display_width, display_height),
                                      parent->colorGroup());
#endif

        if (destroyPainter)
        {
            p->end();
            delete p;
        }
    }
}


void RubberSelectionBox::handleMouseReleaseEvent(QMouseEvent *e)
{
    if (_dragging)
    {
        draw();
        _dragging = false;
        emit dragBoxCompleted(x, y, width, height, private_word, anchored);
    }
}


void RubberSelectionBox::handleMouseMoveEvent(QMouseEvent *e)
{
    if (_dragging)
    {
        draw();

        // Update the rubber box position

        if (anchored)
        {
            int lastx = x+width;
            int lasty = y+height;

            width  += e->x()-lastx;
            height += e->y()-lasty;

            doconstrain_anchored();
        }
        else
        {
            x -= (x + deltax) - e->x();
            y -= (y + deltay) - e->y();

            doconstrain_box();
        }

        // Perform auto scroll

        if (!pointerInAutoScrollBorder && autoScroll)
        {
            if (e->x() < parent->contentsX() + border)
            {
                int delta = parent->contentsX() + border- e->x();
                parent->setContentsPos(parent->contentsX() - delta,
                                       parent->contentsY());
                pointerInAutoScrollBorder = true;
            }
            else if (e->x()
                     > parent->contentsX() + parent->visibleWidth() - border)
            {
                int delta
                    = e->x()
                    - (parent->contentsX() + parent->visibleWidth() - border);
                parent->setContentsPos(parent->contentsX() + delta,
                                       parent->contentsY());
                pointerInAutoScrollBorder = true;
            }
            else if (e->y() < parent->contentsY() + border)
            {
                int delta = parent->contentsY() + border - e->y();
                parent->setContentsPos(parent->contentsX(),
                                       parent->contentsY() - delta);
                pointerInAutoScrollBorder = true;
            }
            else if (e->y()
                     > parent->contentsY() + parent->visibleHeight() - border)
            {
                int delta
                    = e->y()
                    - (parent->contentsY() + parent->visibleHeight() - border);
                parent->setContentsPos(parent->contentsX(),
                                       parent->contentsY() + delta);
                pointerInAutoScrollBorder = true;
            }

            if (pointerInAutoScrollBorder)
            {
                // If the mouse button is held still, we want auto scroll to
                // auto repeat, so we fire a single shot event off to perform
                // another update in a few milliseconds time.
                QTimer::singleShot(TIMER_MSECS, this,
                                   SLOT(slotAutoScrollTimer()));

                if (!boxOffScreen)
                {
                    // Whilst auto scrolling we need to remove the box or all
                    // sorts of redraw nasties occur. So we schedule a timer
                    // that will redraw the box after the window scroll has
                    // occurred.
                    boxOffScreen = true;
                    QTimer::singleShot(TIMER_MSECS, this,
                                       SLOT(slotBoxBackOnScreen()));
                }
            }
        }

        draw();
    }
}


void RubberSelectionBox::doconstrain_anchored()
{
    /*
     * The constrain function is slightly yucky because it has to be able
     * to cope with the box growing in any direction if it's got an
     * anchored corner.
     */
    if (constrain)
    {
        // Minimum x
        if (x > minx && x < minx+minwidth)
        {
            x     = minx;
            width = minwidth;
        }
        if (minwidth != 0 && x <= minx && x+width < minx+minwidth)
        {
            width = (minx+minwidth)-x;
        }
        if (minwidth != 0 && x >= minx+minwidth && x+width > minx)
        {
            width = minx-x;
        }

        // Minimum y
        if (y > miny && y < miny+minheight)
        {
            y      = miny;
            height = minheight;
        }
        if (minheight != 0 && y <= miny && y+height < miny+minheight)
        {
            height = (miny+minheight)-y;
        }
        if (minheight != 0 && y >= miny+minheight && y+height > miny)
        {
            height = miny-y;
        }

        // Maximum x
        if (x < maxx)
        {
            x = maxx;
        }
        if (x > maxx + maxwidth)
        {
            x = maxx + maxwidth;
        }
        if (x + width > maxx + maxwidth)
        {
            width = maxx + maxwidth - x;
        }
        if (x + width < maxx)
        {
            width = maxx - x;
        }

        // Maximum y
        if (y < maxy)
        {
            y = maxy;
        }
        if (y > maxy + maxheight)
        {
            y = maxy + maxheight;
        }
        if (y+height > maxy+maxheight)
        {
            height = maxy + maxheight - y;
        }
        if (y+height < maxy)
        {
            height = maxy - y;
        }
    }

    // Filter
    display_x      = x;
    display_y      = y;
    display_width  = width;
    display_height = height;
    if (_filter)
    {
        _filter->filter(display_x, display_y, display_width, display_height,
                        private_word, anchored);
    }
}


void RubberSelectionBox::doconstrain_box()
{
    if (constrain)
    {
        if (width > maxwidth)
        {
            width = maxwidth;
        }
        if (x + width > maxx + maxwidth)
        {
            x = maxx + maxwidth - width;
        }
        if (x < maxx)
        {
            x = maxx;
        }

        if (height > maxheight)
        {
            height = maxheight;
        }
        if (y + height > maxy + maxheight)
        {
            y = maxy + maxheight - height;
        }
        if (y < maxy)
        {
            y = maxy;
        }
    }

    // Filter
    display_x      = x;
    display_y      = y;
    display_width  = width;
    display_height = height;
    if (_filter)
    {
        _filter->filter(display_x, display_y, display_width, display_height,
                        private_word, anchored);
    }
}


void RubberSelectionBox::slotAutoScrollTimer()
{
    pointerInAutoScrollBorder = false;
    QMouseEvent e(QEvent::MouseMove,
                  parent->viewportToContents
                      (parent->viewport()->mapFromGlobal(QCursor::pos())),
                  QCursor::pos(), 0, 0);
    handleMouseMoveEvent(&e);
}


void RubberSelectionBox::slotBoxBackOnScreen()
{
    boxOffScreen = false;
    draw();
}

