/*
 * @(#)RubberSelectionBox.h 1.00 11 July 2000
 *
 * Copyright (c) Pete Goodliffe 2000 (pete@cthree.org)
 *
 * This file is part of anthem - the TSE3 sequencer.
 *
 * This program is modifiable/redistributable under the terms of the GNU
 * General Public License.
 *
 * You should have recieved a copy of the GNU General Public License along
 * with this program; see the file COPYING. If not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 0211-1307, USA.
 */

#ifndef ANTHEM_RUBBERSELBOX_H
#define ANTHEM_RUBBERSELBOX_H

#include <qscrollview.h>

/**
 * This is the interface for @ref RubberSelectionBox filters. You can
 * use filters to 'snap' the RubberSelectionBox's dashed selection box
 * to certain positions on the screen.
 *
 * You register a filter with the @ref RubberSelectionBox using
 * @ref RubberSelectionBox::filter.
 *
 * @short   RubberSelectionBox position filter
 * @author  Pete Goodliffe
 * @version 1.2
 * @see     RubberSelectionBox
 */
class RubberSelectionBoxFilter
{
    public:
        /**
         * This is called by the @ref RubberSelectionBox and allows you to
         * alter the display coordinates before it displays the drag box. Every
         * time it recieves a mouse move event, it will calculate the new
         * drag box coordinates, then pass them to this filter method, where
         * you can 'snap' them to a position you'd prefer. Following this
         * the box is draw on the @ref QScrollView.
         *
         * This will only filter what is displayed, the values returned by the
         * @ref RubberSelectionBox in the
         * @ref RubberSelectionBox::dragBoxCompleted signal are not filtered -
         * you may wish to filter them yourself before using them.
         */
        virtual void filter(int &x, int &y, int &width, int &height,
                            int private_word, bool anchored) = 0;
};

/**
 * This class provides the facility for a @ref QScrollView widget to have a
 * draggable 'rubber' selection box. This selection box is a flexible tool.
 *
 * @sect Features
 *
 * It operates in one of two modes:
 * @li Either a free-floating box roaming around the @ref QScrollView,
 * @li or a box with one corner anchored, the box size drags instead.
 *
 * Additionally it provides:
 * @li A minimum and maximum size constraint facility, so the dragged box
 *     can't go out of a certain range, or shrink smaller than a certain size.
 * @li An attachable filter that transforms the coordinates before the
 *     box is displayed on the screen. You can visually demonstrate 'snap to
 *     position' effects with this.
 * @li A 'private word' flag that you specify at drag start that is returned
 *     when the drag completes - you can use this instead of storing state
 *     in your @ref QScrollView class to describe what the drag was for.
 * @li The ability to 'auto scroll' the QScrollView as the pointer gets
 *     near the edge.
 *
 * @sect Using the RubberSelectionBox
 *
 * You will create one RubberSelectionBox object in your @ref QScrollView
 * class.
 *
 * In the @ref QScrollView's drawContents method you place a call to
 * @ref draw at the very beginning, and one at the very end. This will ensure
 * that your display is updated correctly:
 *
 * <pre>
 * void MyWidget::drawContents(QPainter *p,
 *                             int clipx, int clipy, int clipw, int cliph)
 * {
 *     rubberSelBox->draw(p);
 *
 *     // Place your normal drawing code here...
 *
 *     rubberSelBox->draw(p);
 * }
 * </pre>
 *
 * In the @ref QScrollView's contentsMouseReleaseEvent method, you should
 * call @ref handleMouseReleaseEvent.
 *
 * <pre>
 * void MyWidget::contentsMouseReleaseEvent(QMouseEvent *e)
 * {
 *      rubberSelBox->handleMouseReleaseEvent(e);
 * }
 * </pre>
 *
 * In the contentsMouseMoveEvent you write something like:
 *
 * <pre>
 * void MyWidget::contentsMouseMoveEvent(QMouseEvent *e)
 * {
 *     if (rubberSelBox->dragging())
 *     {
 *         rubberSelBox->handleMouseMoveEvent(e)
 *         return;
 *     }
 *
 *     // else if a drag has started, call one of the start methods
 * }
 * </pre>
 *
 * You may also want to catch keyboard events, and if Escape is pressed call
 * the @ref stop method (don't forget to set your focus policy).
 *
 * <pre>
 * void MyWidget::keyPressEvent(QKeyEvent *e)
 * {
 *     if (e->key() == Key_Escape)
 *     {
 *         rubberSelBox->stop();
 *     }
 *     else
 *     {
 *         e->ignore();
 *     }
 * }
 * </pre>
 *
 * @sect Performing a drag box operation
 *
 * If you recieve a drag event in your widget and want to start a rubber drag
 * box operation call either @ref startAnchoredDrag or @ref startDragBox.
 * With the above plumbing carried out, that's all there is to it - the mouse
 * will drag a dotted 'Rubber' box around the @ref QScrollView.
 *
 * When the drag operation completes the RubberSelectionBox emits the
 * @ref dragBoxCompleted signal.
 *
 * @short   Rubber selection box
 * @author  Pete Goodliffe
 * @version 1.2
 */
class RubberSelectionBox : public QObject
{
        Q_OBJECT;

    public:

        /**
         * Creates a rubber selection box object. You specify the
         * parent object to produce the drag in.
         *
         * You may also specify in @p autoScrollBorder the size of the
         * border around the edge of the window that is used to perform
         * autoScroll. For most uses, the default value is satisfactory.
         *
         * @param parent           Parent @ref QScrollView object
         * @param autoScrollBorder Auto scroll border area size in pixels
         */
        RubberSelectionBox(QScrollView *parent, int autoScrollBorder = 20);
        ~RubberSelectionBox();

        /**
         * Starts an 'anchored' drag box operation. If a drag is already taking
         * place, then nothing happens and the first drag continues.
         *
         * An anchored drag has the x and y corner position fixed, and
         * the other corner 'floats', following the pointer position.
         *
         * You may 'constrain' the drag operation, and force the drag box
         * to be within a specified rectangle.
         *
         * @param x            x anchored corner pos
         * @param y            y anchored corner pos
         * @param width        initial box width
         * @param height       initial box height
         * @param private_word a special value you can supply that will be
         *                     returned to you, you can use this for your own
         *                     context
         * @param autoScroll   Whether to perform automatic window scrolling
         * @param constrain    whether the rubber box can move anywhere in the
         *                     window (flase) or is constrained to the min/max
         *                     values supplied (true)
         * @see                stop
         */
        void startAnchoredDrag(int x, int y, int width, int height,
                               int private_word = 0,
                               bool autoScroll = true,
                               bool constrain = false,
                               int minx = 0, int miny = 0,
                               int minwidth = 0, int minheight = 0,
                               int maxx = 0, int maxy = 0,
                               int maxwidth = 0, int maxheight = 0);

        /**
         * Starts a 'free' drag box operation. If a drag is already taking
         * place, then nothing happens and the first drag continues.
         *
         * A free drag has a single fixed size drag box moving around the
         * widget, following the mouse pointer.
         *
         * You may 'constrain' the drag operation, and force the drag box
         * to be within a specified rectangle.
         *
         * @param x            x pointer position at drag start
         * @param y            y pointer position at drag start
         * @param private_word a special value you can supply that will be
         *                     returned to you, you can use this for your own
         *                     context
         * @param boxx         minimum x position of box
         * @param boxy         minimum y position of box
         * @param boxwidth     drag box width
         * @param boxheight    drag box height
         * @param autoScroll   Whether to perform automatic window scrolling
         * @param constrain    whether the rubber box can move anywhere in the
         *                     window (flase) or is constrained to the min/max
         *                     values supplied (true)
         * @see                stop
         */
        void startDragBox(int x, int y,
                          int private_word = 0,
                          int boxx = 0, int boxy = 0,
                          int boxwidth = 0, int boxheight = 0,
                          bool autoScroll = true,
                          bool constrain = false,
                          int maxx = 0, int maxy = 0,
                          int maxwidth = 0, int maxheight = 0);

        /**
         * Stops the current drag (you might call this if Qt::Key_Escape is
         * pressed, for example).
         *
         * If there is no drag occuring, then nothing happens.
         *
         * You can choose whether the @ref dragBoxCompleted signal is emitted
         * or not with the @p signal boolean parameter.
         *
         * @param signal Whether to send the @ref dragBoxCompleted signal
         * @see   startAnchoredDrag
         * @see   startDragBox
         */
        void stop(bool signal = false);

        /**
         * Attaches or removes a filter from the RubberSelectionBox.
         * If @p filter is zero, then any attached filter is removed.
         *
         * Filters allow you to alter how the rubber box is drawn on the
         * screen. It is a mechanism for making the box draw in a different
         * location. The RubberSelectionBox class will not think you have
         * actually moved the box, it will just draw it where you say
         * insteaed of where it was going to.
         */
        void filter(RubberSelectionBoxFilter *filter) { _filter = filter; }

        /**
         * Returns true if a drag is taking place.
         */
        bool dragging() const { return _dragging; }

        /**
         * The rubber selection box is drawn in XOR mode, so drawing it once
         * produces output, drawing it a second time removes this output.
         *
         * In your widget's drawContents method you should call this twice,
         * once at the very beginning, and once at the very end. The box will
         * only be drawn if a drag is in progress.
         *
         * If you specify 0 as the painter then the object will create
         * its own QPainter for the scroll view.
         */
        void draw(QPainter *p = 0);

        /**
         * Call this from your widget's contentsMouseReleaseEvent
         */
        void handleMouseReleaseEvent(QMouseEvent *e);

        /**
         * Call this from your widget's contentsMouseMoveEvent
         */
        void handleMouseMoveEvent(QMouseEvent *e);

    signals:

        /**
         * This signal is emitted when the drag operation finishes.
         *
         * The values returned will not have been filtered if you have
         * attached a @ref RubberSelectionBoxFilter - you may wish
         * to do this yourself.
         *
         * @param x            X position of selection box drop point
         * @param y            Y position of selection box drop point
         * @param width        Width of selection box drop point
         * @param height       Height of selection box drop point
         * @param private_word The private word specified when the drag started
         * @param anchored     True if the drag was anchored
         */
        void dragBoxCompleted(int x, int y, int width, int height,
                              int private_word, bool anchored);

    private slots:

        void slotAutoScrollTimer();
        void slotBoxBackOnScreen();

    private:

        /**
         * Looks at x, y, width, height and constrains them to the max
         * box value, for 'free' drag box mode.
         */
        void doconstrain_box();

        /**
         * Looks at x, y, width, height and constrains them to the min/max
         * values, for 'anchored' mode.
         */
        void doconstrain_anchored();

        QScrollView              *parent;
        RubberSelectionBoxFilter *_filter;
        bool                      anchored;
        int                       x, y, width, height, private_word;
        int                       display_x, display_y;
        int                       display_height, display_width;
        int                       deltax, deltay;
        bool                      _dragging;

        bool                      constrain;
        int                       minx, miny, minwidth, minheight;
        int                       maxx, maxy, maxwidth, maxheight;

        bool                      autoScroll;
        bool                      pointerInAutoScrollBorder;
        int                       border;
        bool                      boxOffScreen;
};

#endif
