/*
 * @(#)VoiceWidget.h 1.00 3 April 2001
 *
 * Copyright (c) Pete Goodliffe 2000 (pete@cthree.org)
 *
 * This file is part of anthem - the TSE3 sequencer.
 *
 * This program is modifiable/redistributable under the terms of the GNU
 * General Public License.
 *
 * You should have recieved a copy of the GNU General Public License along
 * with this program; see the file COPYING. If not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 0211-1307, USA.
 */

#ifndef ANTHEM_GADGETS_VOICE_H
#define ANTHEM_GADGETS_VOICE_H

#include <qwidget.h>
#include <qpopupmenu.h>

/******************************************************************************
 * QT/KDE forward declarations
 *****************************************************************************/
class QLabel;
class QPushButton;

class KListView;

/******************************************************************************
 * Anthem forward declarations
 *****************************************************************************/
class MidiParamsSpinBox;
class ChannelWidget;
class PortWidget;

/******************************************************************************
 * File forward declarations
 *****************************************************************************/
class VoiceByNumberPopup;
class VoiceByNamePopup;

/**
 * This widget provides a facility for chosing a voice value. It uses the
 * @ref TSE3::Ins::Destination and @ref TSE3::Ins::Instrument objects
 * to provide a menu of named voices.
 *
 * @short   MIDI voice choice widget
 * @author  Pete Goodliffe
 * @version 1.0
 */
class VoiceWidget : public QWidget
{
        Q_OBJECT

    public:

        /**
         * Creates a VoiceWidget which is initially set to the
         * given values. You can specify whether the 'any' (-1)
         * value is permitted or not.
         *
         * You can specify a channel/port which is used for the initial
         * name lookup.
         */
        VoiceWidget(bool any,
                    int program = -1, int bankLSB = -1, int bankMSB = -1,
                    int channel = 1, int port = -1,
                    QWidget *parent = 0, const char *name = 0);

        /**
         * Returns the currently selected program value.
         */
        int program();

        /**
         * Sets the current program value.
         */
        void setProgram(int p);

        /**
         * Returns the currently selected bankLSB value.
         */
        int bankLSB();

        /**
         * Sets the current bankLSB value.
         */
        void setBankLSB(int b);

        /**
         * Returns the currently selected bankMSB value.
         */
        int bankMSB();

        /**
         * Sets the current bankMSB value.
         */
        void setBankMSB(int b);

        /**
         * Sets all the voice parameters in one fell swoop.
         */
        void setValues(int program, int bankLSB, int bankMSB);

        /**
         * Converts a data set to a text name.
         */
        static QString toTextName(int program, int bankLSB, int bankMSB,
                                  int channel, int port);

        /**
         * Converts a data set to a text number set for display.
         */
        static QString toTextNumber(int program, int bankLSB, int bankMSB,
                                    int channel, int port);

    public slots:

        void slotVoiceSelected(int program, int bankLSB, int bankMSB);

    private:

        void updateWidgets();

        bool         any;
        QLabel      *name;
        QLabel      *number;
        QPushButton *byName;
        QPushButton *byNumber;

        int          _channel, _port;
        int          _program, _bankLSB, _bankMSB;

        VoiceByNumberPopup *numberPopup;
        VoiceByNamePopup   *namePopup;
};

/**
 * Internal widget used by the @ref VoiceWidget.
 *
 * @short   MIDI voice choice widget
 * @author  Pete Goodliffe
 * @version 1.0
 * @internal
 */
class VoiceByNumberPopup : public QPopupMenu
{
        Q_OBJECT

    public:

        VoiceByNumberPopup(QWidget *parent, const char *name = 0);

    signals:

        void selected(int prog, int bankLSB, int bankMSB);

    public slots:

        void slotSelect(int prog, int bankLSB, int bankMSB);

    private slots:

        void slotChanged();
        void slotChangedBank();

    private:

        MidiParamsSpinBox *program;
        MidiParamsSpinBox *bankLSB;
        MidiParamsSpinBox *bankMSB;
        MidiParamsSpinBox *bank;
        QLabel            *bankLabel;
};

/**
 * Internal widget used by the @ref VoiceWidget.
 *
 * @short   MIDI voice choice widget
 * @author  Pete Goodliffe
 * @version 1.0
 * @internal
 */
class VoiceByNamePopup : public QPopupMenu
{
        Q_OBJECT

    public:

        VoiceByNamePopup(QWidget *parent, const char *name = 0);

    signals:

        void selected(int prog, int bankLSB, int bankMSB);

    public slots:

        void slotDestination(int channel, int port);
        void slotSelect(int prog, int bankLSB, int bankMSB);

    private slots:

        void slotChanged();

    private:

        void updateList();

        ChannelWidget *channelWidget;
        PortWidget    *portWidget;
        KListView     *listView;

        int program, bankLSB, bankMSB;
        int channel, port;
};

#endif
