/*
 * @(#)Streambuf.cpp 1.00 13 November 2000
 *
 * Copyright (c) Pete Goodliffe 2000 (pete@cthree.org)
 *
 * This file is part of anthem - the TSE3 sequencer.
 *
 * This program is modifiable/redistributable under the terms of the GNU
 * General Public License.
 *
 * You should have recieved a copy of the GNU General Public License along
 * with this program; see the file COPYING. If not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 0211-1307, USA.
 */

#include "misc/Streambuf.h"

#include <qstring.h>
#include <qtextview.h>
#include <qtimer.h>

namespace
{
    const int TIMER_MSECS = 1000;
}

/*
 * These classes have been implemented based on information found at:
 *     http://www.inf.uni-konstanz.de/~kuehl/iostream/
 */

/******************************************************************************
 * QStringOStream class
 *****************************************************************************/

QStringOStream::QStringOStream(QString &str, int bsize)
: std::ostream(new QStringStreambuf(str, bsize))
{
}


/******************************************************************************
 * QStringOStreambuf class
 *****************************************************************************/

QStringStreambuf::QStringStreambuf(QString &s, int bsize)
: str(s)
{
    if (bsize)
    {
        char *ptr = new char[bsize];
        setp(ptr, ptr + bsize);
    }
    else
    {
        setp(0, 0);
    }
    setg(0, 0, 0);
}


QStringStreambuf::~QStringStreambuf()
{
    sync();
    delete[] pbase();
}


int QStringStreambuf::overflow(int c)
{
    put_buffer();
    if (c != EOF)
    {
        if (pbase() == epptr())
        {
            put_char(c);
        }
        else
        {
            sputc(c);
        }
    }
    return 0;
}


int QStringStreambuf::sync()
{
    put_buffer();
    return 0;
}


void QStringStreambuf::put_buffer(void)
{
    if (pbase() != pptr())
    {
        int   len    = (pptr() - pbase());
        char *buffer = new char[len + 1];
        strncpy(buffer, pbase(), len);
        buffer[len] = 0;
        str += buffer;
        setp(pbase(), epptr());
        delete [] buffer;
    }
}


void QStringStreambuf::put_char(int c)
{
    str += (char) c;
}


/******************************************************************************
 * QTextViewOStream class
 *****************************************************************************/

QTextViewOStream::QTextViewOStream(QTextView &tv, int bsize)
: std::ostream(new QTextViewStreambuf(tv, bsize))
{
}


/******************************************************************************
 * QTextViewOStreambuf class
 *****************************************************************************/

QTextViewStreambuf::QTextViewStreambuf(QTextView &t, int bsize)
: tv(t)
{
    timer = new QTimer(this);
    connect(timer, SIGNAL(timeout()), this, SLOT(slotScrollDown()));

    if (bsize)
    {
        char *ptr = new char[bsize];
        setp(ptr, ptr + bsize);
    }
    else
    {
        setp(0, 0);
    }
    setg(0, 0, 0);
}


QTextViewStreambuf::~QTextViewStreambuf()
{
    sync();
    delete[] pbase();
}


int QTextViewStreambuf::overflow(int c)
{
    put_buffer();
    if (c != EOF)
    {
        if (pbase() == epptr())
        {
            put_char(c);
        }
        else
        {
            sputc(c);
        }
    }
    return 0;
}


int QTextViewStreambuf::sync()
{
    put_buffer();
    return 0;
}


void QTextViewStreambuf::put_buffer(void)
{
    if (pbase() != pptr())
    {
        int   len    = (pptr() - pbase());
        char *buffer = new char[len + 1];
        strncpy(buffer, pbase(), len);
        buffer[len] = 0;
        tv.append(buffer);
        timer->start(TIMER_MSECS, true);
        setp(pbase(), epptr());
        delete [] buffer;
    }
}


void QTextViewStreambuf::put_char(int c)
{
    static QString str;
    if (c == '\n')
    {
        tv.append(str);
        timer->start(TIMER_MSECS, true);
        str = "";
    }
    else
    {
        str += QChar(c);
    }
}


void QTextViewStreambuf::slotScrollDown()
{
    tv.setContentsPos(tv.contentsX(), tv.contentsHeight());
}
