/*
 * @(#)PhraseEditor_List.cpp 1.00 22 September 2000
 *
 * Copyright (c) Pete Goodliffe 2000 (pete@cthree.org)
 *
 * This file is part of anthem - the TSE3 sequencer.
 *
 * This program is modifiable/redistributable under the terms of the GNU
 * General Public License.
 *
 * You should have recieved a copy of the GNU General Public License along
 * with this program; see the file COPYING. If not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 0211-1307, USA.
 */

#include "phraseeditors/ListEditor.h"

#include "tse3/kdeui/PortWidget.h"
#include "tse3/kdeui/ChannelWidget.h"
#include "tse3/kdeui/ClockWidget.h"
#include "gadgets/KeyWidgets.h"

#include "tse3/PhraseEdit.h"
#include "tse3/util/NoteNumber.h"

#include <kcombobox.h>
#include <kiconloader.h>
#include <klocale.h>
#include <ktoolbar.h>

#include <qlayout.h>
#include <qlabel.h>
#include <qslider.h>
#include <qspinbox.h>


/******************************************************************************
 * File local definitions
 *****************************************************************************/

namespace
{
    enum Columns
    {
        Column_Time,
        Column_Pixmap,
        Column_Type,
        Column_Channel,
        Column_Port,
        Column_Data1,
        Column_Data2,
        Column_Description
    };

    const char *TIME    = "Time";
    const char *TYPE    = "Type";
    const char *CHANNEL = "C";
    const char *PORT    = "P";
    const char *DATA1   = "Data 1";
    const char *DATA2   = "Data 2";
    const char *DESC    = "Description";

    const char *STATUS[] =
    {
        "Note off",
        "Note on",
        "Key pressure",
        "Control change",
        "Program change",
        "Channel pressure",
        "Pitch bend",
        "SysEx",
        "Unknown"
    };

    const char *commandName(TSE3::MidiCommand cmd)
    {
        if (cmd.status >= 8)
        {
            return STATUS[cmd.status-8];
        }
        else
        {
            return STATUS[8];
        }
    }

    QString data1String(TSE3::MidiCommand cmd)
    {
        if (cmd.isNote())
        {
            std::string note = TSE3::Util::numberToNote(cmd.data1);
            return QString(note.c_str());
        }
        else
        {
            return QString::number(cmd.data1);
        }
    }
}


/******************************************************************************
 * ListEditorItem class
 *****************************************************************************/

/**
 * The ListEditorItem is a QListViewItem widget that specifically
 * knows how to display MIDI event information.
 */
class ListEditorItem : public QListViewItem
{
    public:

        /**
         * Construct a ListEditorItem for the given event.
         */
        ListEditorItem(const TSE3::MidiEvent &e,
                       ListEditorImpl *parent, QListViewItem *after);

        virtual ~ListEditorItem();

        void setItemHeight(int h)
        {
            setHeight(h);
        }

    private:

        TSE3::MidiEvent event;
};


ListEditorItem::ListEditorItem(const TSE3::MidiEvent &event,
                               ListEditorImpl *parent,
                               QListViewItem *after)
: QListViewItem(parent, after,
                ClockWidget::toText(event.time),
                QString(),
                commandName(event.data),
                QString::number(event.data.channel+1),
                QString::number(event.data.port),
                data1String(event.data),
                QString::number(event.data.data2)),
  event(event)
{
    setPixmap(1, parent->midiPixmap(event.data.status-8));
    if (event.data.status == TSE3::MidiCommand_NoteOn)
    {
        QString s("Duration = ");
        s.append(ClockWidget::toText(event.offTime - event.time));
        setText(7, s);
    }
}


ListEditorItem::~ListEditorItem()
{
}


/******************************************************************************
 * ListEditorImpl class
 *****************************************************************************/

ListEditorImpl::ListEditorImpl(TSE3::PhraseEdit *phraseEdit, QWidget *parent)
: FilteredKListView(parent, "ListEditorImpl"), phraseEdit(phraseEdit)
{
    attachTo(phraseEdit);

    for (size_t n = 0; n < 8; n++)
    {
        QString s;
        s.sprintf("midi/midi0%X", n+8);
        midiIcons[n] = UserIcon(s);
    }

    addColumn(TIME);
    addColumn("");
    addColumn(TYPE);
    addColumn(CHANNEL);
    addColumn(PORT);
    addColumn(DATA1);
    addColumn(DATA2);
    addColumn(DESC);
    setAllColumnsShowFocus(true);
    setSorting(-1);
    setColumnAlignment(1, AlignHCenter);

    connect(this, SIGNAL(doubleClicked(QListViewItem*)),
            SLOT(slotDoubleClicked(QListViewItem*)));
    connect(this, SIGNAL(clicked(QListViewItem*,const QPoint &,int)),
            SLOT(slotClicked(QListViewItem*,const QPoint &,int)));
    connect(this, SIGNAL(wantToSelect(QListViewItem*,bool)),
            SLOT(slotWantToSelect(QListViewItem*,bool)));
    connect(this, SIGNAL(wantToClear(QListViewItem*)),
            SLOT(slotWantToClear(QListViewItem*)));

    // Initial set of data
    updateAllItems();

    // Set the type pixmap column's width
    int pixWidth = 0;
    for (size_t n = 0; n < 8; n++)
    {
        if (midiIcons[n].width() > pixWidth)
        {
            pixWidth = midiIcons[n].width();
        }
    }
    setColumnWidthMode(1, Manual);
    setColumnWidth(1, pixWidth + itemMargin()*2);

    // Set the type text column's width
    int typeWidth = 0;
    for (size_t n = 0; n < 8; n++)
    {
        QString s(STATUS[n]);
        int width = fontMetrics().width(s);
        if (width > typeWidth)
        {
            typeWidth = width;
        }
    }
    setColumnWidth(2, typeWidth + itemMargin()*2);
}


void ListEditorImpl::updateAllItems()
{
    clear();
    QListViewItem *last = 0;
    for (size_t n = 0; n < phraseEdit->size(); n++)
    {
        last = new ListEditorItem((*phraseEdit)[n], this, last);
        setSelectedWithoutSignal(last, (*phraseEdit)[n].data.selected);
    }
}


void ListEditorImpl::slotClicked(QListViewItem *e, const QPoint &, int column)
{
    lastClickedCol = column;
    emit event((*phraseEdit)[indexOf(e)]);
}


void ListEditorImpl::ListEditorImpl::slotDoubleClicked(QListViewItem *e)
{
    TSE3::MidiEvent ev  = (*phraseEdit)[indexOf(e)];
    TSE3::Clock     dur = (ev.data.status == TSE3::MidiCommand_NoteOn)
                        ? ev.offTime - ev.time
                        : TSE3::Clock(-1) ;

    enum EditorType
    {
        Default,
        Note,
        Data1,
        Data2
    }
    editorType = Default;

    if (lastClickedCol == 5)
    {
        if (ev.data.isNote())
        {
            editorType = Note;
        }
        else
        {
            editorType = Data1;
        }
    }
    else if (lastClickedCol == 6)
    {
        editorType = Data2;
    }

    switch (editorType)
    {
        case Note:
        {
            ListEditor_NoteEditor *editor
                = new ListEditor_NoteEditor(indexOf(e), ev.data.data1, this);
            connect(editor,
                   SIGNAL(selected(size_t,int)),
                   SLOT(slotChangeData1(size_t,int)));
            editor->show();
            break;
        }
        case Data1:
        {
            ListEditor_DataEditor *editor
                = new ListEditor_DataEditor(indexOf(e), ev.data.data1, this);
            connect(editor,
                   SIGNAL(selected(size_t,int)),
                   SLOT(slotChangeData1(size_t,int)));
            editor->show();
            break;
        }
        case Data2:
        {
            ListEditor_DataEditor *editor
                = new ListEditor_DataEditor(indexOf(e), ev.data.data2, this);
            connect(editor,
                   SIGNAL(selected(size_t,int)),
                   SLOT(slotChangeData2(size_t,int)));
            editor->show();
            break;
        }
        case Default:
        {
            ListEditor_TimeChannelEditor *editor
                = new ListEditor_TimeChannelEditor(indexOf(e), ev.time, dur,
                                                   ev.data.channel,
                                                   ev.data.port,
                                                   this);
            connect(editor,
                    SIGNAL(selected(size_t,TSE3::Clock,TSE3::Clock,
                                    size_t,size_t)),
                    SLOT(slotChangeEvent(size_t,TSE3::Clock,TSE3::Clock,
                                         size_t,size_t)));
            editor->show();
            break;
        }
    }
}


void ListEditorImpl::slotWantToSelect(QListViewItem *e, bool add)
{
    if (!add)
    {
        phraseEdit->clearSelection();
    }
    phraseEdit->select(indexOf(e));
}


void ListEditorImpl::slotWantToClear(QListViewItem *e)
{
    phraseEdit->deselect(indexOf(e));
}


void ListEditorImpl::slotChangeEvent(size_t index, TSE3::Clock time,
                                     TSE3::Clock duration,
                                     size_t channel, size_t port)
{
    TSE3::MidiEvent e = (*phraseEdit)[index];
    e.time         = time;
    e.offTime      = time + duration;
    e.data.channel = channel;
    e.data.port    = port;
    phraseEdit->erase(index);
    phraseEdit->insert(e);
}


void ListEditorImpl::slotChangeData1(size_t index, int data)
{
    TSE3::MidiEvent e = (*phraseEdit)[index];
    e.data.data1      = data;
    phraseEdit->erase(index);
    phraseEdit->insert(e);
}


void ListEditorImpl::slotChangeData2(size_t index, int data)
{
    TSE3::MidiEvent e = (*phraseEdit)[index];
    e.data.data2      = data;
    phraseEdit->erase(index);
    phraseEdit->insert(e);
}


void ListEditorImpl::PhraseEdit_Reset(TSE3::PhraseEdit *)
{
    updateAllItems();
}


void ListEditorImpl::PhraseEdit_Tidied(TSE3::PhraseEdit *)
{
    updateAllItems();
}


void ListEditorImpl::PhraseEdit_Inserted(TSE3::PhraseEdit *, size_t index)
{
    QListViewItem *toFollow = 0;
    if (index != 0)
    {
        toFollow = itemWithIndex(index-1);
    }
    QListViewItem *item
        = new ListEditorItem((*phraseEdit)[index], this, toFollow);
    setCurrentItem(item);

    phraseEdit->clearSelection();
    phraseEdit->select(index);
}


void ListEditorImpl::PhraseEdit_Erased(TSE3::PhraseEdit *, size_t index)
{
    delete itemWithIndex(index);
}


void ListEditorImpl::PhraseEdit_Selection(TSE3::PhraseEdit *, size_t index,
                                          bool selected)
{
    setSelectedWithoutSignal(itemWithIndex(index), selected);
}


/******************************************************************************
 * ListEditor class
 *****************************************************************************/

ListEditor::ListEditor(TSE3::PhraseEdit *pe, KToolBar *toolbar,
                       QWidget *parent)
: PhraseEditorBase(pe, toolbar, parent),
  phraseEdit(pe),
  lastTime(0), lastDuration(TSE3::Clock::PPQN),
  lastChannel(0), lastPort(0),
  lastNote(0x40), lastVelocity(0x100), lastOffVelocity(0x100),
  lastControl(0), lastControlData(0)
{
    toolbar->insertButton(UserIcon("midi/midi09"), Toolbar_AddNote, true,
                                   i18n("Add note"));
    toolbar->insertButton(UserIcon("midi/midi0A"), Toolbar_AddKeyPressure, true,
                                   i18n("Add key pressure"));
    toolbar->insertButton(UserIcon("midi/midi0B"), Toolbar_AddControl, true,
                                   i18n("Add control"));
    toolbar->insertButton(UserIcon("midi/midi0C"), Toolbar_AddProgram, true,
                                   i18n("Add program change"));
    toolbar->insertButton(UserIcon("midi/midi0D"), Toolbar_AddChannelPressure,
                                   true, i18n("Add channel pressure"));
    toolbar->insertButton(UserIcon("midi/midi0E"), Toolbar_AddPitch, true,
                                   i18n("Add pitch bend"));
    toolbar->insertButton(UserIcon("midi/midi0F"), Toolbar_AddSysEx, true,
                                   i18n("Add SysEx"));
    toolbar->insertButton(BarIcon("editdelete"), Toolbar_Delete, true,
                                   i18n("Delete"));

    connect(toolbar, SIGNAL(clicked(int)), SLOT(slotToolbarClicked(int)));

    QVBoxLayout *layout = new QVBoxLayout(this);
    impl = new ListEditorImpl(phraseEdit, this);
    layout->addWidget(impl);

    connect(impl, SIGNAL(event(TSE3::MidiEvent)),
            SLOT(slotEvent(TSE3::MidiEvent)));

    attachTo(phraseEdit);
    updateDeleteEnabled();
}


ListEditor::~ListEditor()
{
    toolbar->removeItem(Toolbar_AddNote);
    toolbar->removeItem(Toolbar_AddKeyPressure);
    toolbar->removeItem(Toolbar_AddControl);
    toolbar->removeItem(Toolbar_AddProgram);
    toolbar->removeItem(Toolbar_AddChannelPressure);
    toolbar->removeItem(Toolbar_AddPitch);
    toolbar->removeItem(Toolbar_AddSysEx);
    toolbar->removeItem(Toolbar_Delete);
}


void ListEditor::slotToolbarClicked(int item)
{
    switch (item)
    {
        case Toolbar_AddNote:
        {
            TSE3::MidiEvent e(TSE3::MidiCommand(TSE3::MidiCommand_NoteOn,
                                                lastChannel, lastPort,
                                                lastNote, lastVelocity),
                              lastTime,
                              lastOffVelocity, lastTime+lastDuration);
            phraseEdit->insert(e);
            break;
        }
        case Toolbar_AddKeyPressure:
        {
            TSE3::MidiEvent e(TSE3::MidiCommand(TSE3::MidiCommand_KeyPressure,
                                                lastChannel, lastPort,
                                                lastNote, lastVelocity),
                              lastTime);
            phraseEdit->insert(e);
            break;
        }
        case Toolbar_AddControl:
        {
            TSE3::MidiEvent e(TSE3::MidiCommand(TSE3::MidiCommand_ControlChange,
                                                lastChannel, lastPort,
                                                lastControl, lastControlData),
                              lastTime);
            phraseEdit->insert(e);
            break;
        }
        case Toolbar_AddProgram:
        {
            TSE3::MidiEvent e(TSE3::MidiCommand(TSE3::MidiCommand_ProgramChange,
                                                lastChannel, lastPort,
                                                lastProgram),
                              lastTime);
            phraseEdit->insert(e);
            break;
        }
        case Toolbar_AddChannelPressure:
        {
            TSE3::MidiEvent e(
                TSE3::MidiCommand(TSE3::MidiCommand_ChannelPressure,
                                  lastChannel, lastPort, lastVelocity),
                lastTime);
            phraseEdit->insert(e);
            break;
        }
        case Toolbar_AddPitch:
        {
            TSE3::MidiEvent e(TSE3::MidiCommand(TSE3::MidiCommand_PitchBend,
                                                lastChannel, lastPort,
                                                lastPitch & 0xff,
                                                lastPitch >> 8),
                              lastTime);
            phraseEdit->insert(e);
            break;
        }
        case Toolbar_Delete:
        {
            phraseEdit->eraseSelection();
            break;
        }
    }
}


void ListEditor::slotEvent(TSE3::MidiEvent e)
{
    lastTime    = e.time;
    lastChannel = e.data.channel;
    lastPort    = e.data.port;
    switch (e.data.status)
    {
        case TSE3::MidiCommand_NoteOff:
        {
            // This should NOT happen (TM)
            lastNote     = e.data.data1;
            lastVelocity = e.data.data2;
            break;
        }
        case TSE3::MidiCommand_NoteOn:
        case TSE3::MidiCommand_KeyPressure:
        {
            lastNote     = e.data.data1;
            lastVelocity = e.data.data2;
            lastDuration = e.offTime - e.time;
            lastOffVelocity = e.offData.data2;
            break;
        }
        case TSE3::MidiCommand_ControlChange:
        {
            lastControl     = e.data.data1;
            lastControlData = e.data.data2;
            break;
        }
        case TSE3::MidiCommand_ProgramChange:
        {
            lastProgram = e.data.data1;
            break;
        }
        case TSE3::MidiCommand_ChannelPressure:
        {
            lastVelocity = e.data.data1;
            break;
        }
        case TSE3::MidiCommand_PitchBend:
        {
            lastPitch = e.data.data1 + (e.data.data2<<8);
            break;
        }
        default:
        {
            break;
        }
    }
}


void ListEditor::PhraseEdit_Selection(TSE3::PhraseEdit *,
                                      size_t /*index*/,
                                      bool /*selected*/)
{
    updateDeleteEnabled();
}


void ListEditor::updateDeleteEnabled()
{
    toolbar->setItemEnabled(Toolbar_Delete, phraseEdit->selection());
}




/******************************************************************************
 * ListEditor_TimeChannelEditor class
 *****************************************************************************/

ListEditor_TimeChannelEditor::ListEditor_TimeChannelEditor
    (size_t index, TSE3::Clock t, TSE3::Clock d, size_t c, size_t p,
     QWidget *parent)
: KDialogBase(Plain, "Edit event", Ok|Cancel, Ok, parent, "TimeChannelEditor",
              true, true),
  index(index)
{
    QGridLayout *layout
        = new QGridLayout(plainPage(), 3, 4, KDialogBase::spacingHint());

    QLabel *label = new QLabel("Time", plainPage());
    layout->addWidget(label, 0, 0);
    time = new ClockWidget(plainPage(), false, t);
    layout->addMultiCellWidget(time, 0, 0, 1, 3);

    label = new QLabel("Duration", plainPage());
    layout->addWidget(label, 1, 0);
    duration = new ClockWidget(plainPage(), false, d);
    if (d == -1)
    {
        duration->setEnabled(false);
    }
    layout->addMultiCellWidget(duration, 1, 1, 1, 3);

    label = new QLabel("Channel/port", plainPage());
    layout->addWidget(label, 2, 0);
    channel = new ChannelWidget(false, false, false, c, plainPage());
    layout->addWidget(channel, 2, 1);
    label = new QLabel(":", plainPage());
    layout->addWidget(label, 2, 2);
    port = new PortWidget(true, false, false, false, p, plainPage());
    layout->addWidget(port, 2, 3);
}


void ListEditor_TimeChannelEditor::accept()
{
    emit selected(index, time->value(), duration->value(),
                  channel->value(), port->value());
    done(Accepted);
}


/******************************************************************************
 * ListEditor_NoteEditor class
 *****************************************************************************/

ListEditor_NoteEditor::ListEditor_NoteEditor(size_t index, int note,
                                             QWidget *parent)
: KDialogBase(Plain, "Edit note", Ok|Cancel, Ok, parent, "NoteEditor",
              true, true),
  index(index)
{
    QVBoxLayout *layout
        = new QVBoxLayout(plainPage(), KDialogBase::spacingHint());

    key = new KeyWidget(note, plainPage());
    layout->addWidget(key);
    slider = new QSlider(0, 127, 1, note, QSlider::Horizontal, plainPage());
    layout->addWidget(slider);

    connect(slider, SIGNAL(valueChanged(int)), key,    SLOT(setValue(int)));
    connect(key,    SIGNAL(valueChanged(int)), slider, SLOT(setValue(int)));
}


void ListEditor_NoteEditor::accept()
{
    emit selected(index, key->value());
    done(Accepted);
}


/******************************************************************************
 * ListEditor_DataEditor class
 *****************************************************************************/

ListEditor_DataEditor::ListEditor_DataEditor(size_t index, int data,
                                             QWidget *parent)
: KDialogBase(Plain, "Edit data", Ok|Cancel, Ok, parent, "DataEditor",
              true, true),
  index(index)
{
    QVBoxLayout *layout
        = new QVBoxLayout(plainPage(), KDialogBase::spacingHint());

    spinbox = new QSpinBox(0, 127, 1, plainPage());
    spinbox->setValue(data);
    layout->addWidget(spinbox);
    slider = new QSlider(0, 127, 1, data, QSlider::Horizontal, plainPage());
    layout->addWidget(slider);

    connect(slider,  SIGNAL(valueChanged(int)), spinbox, SLOT(setValue(int)));
    connect(spinbox, SIGNAL(valueChanged(int)), slider,  SLOT(setValue(int)));
}


void ListEditor_DataEditor::accept()
{
    emit selected(index, spinbox->value());
    done(Accepted);
}

