/*
 * @(#)PhraseEditor_ListEditor.h 1.00 22 September 2000
 *
 * Copyright (c) Pete Goodliffe 2000 (pete@cthree.org)
 *
 * This file is part of anthem - the TSE3 sequencer.
 *
 * This program is modifiable/redistributable under the terms of the GNU
 * General Public License.
 *
 * You should have recieved a copy of the GNU General Public License along
 * with this program; see the file COPYING. If not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 0211-1307, USA.
 */

#ifndef ANTHEM_LISTEDITOR_H
#define ANTHEM_LISTEDITOR_H

#include "phraseeditors/PhraseEditor.h"
#include "gadgets/FilteredKListView.h"

#include "tse3/Notifier.h"
#include "tse3/Midi.h"
#include "tse3/listen/PhraseEdit.h"

#include <kdialogbase.h>

class QSlider;
class QSpinBox;

class ClockWidget;
class ChannelWidget;
class PortWidget;;
class KeyWidget;

/**
 * Internal implementation widget for the ListEditor.
 *
 * @short   @ref TSE3::Phrase list editor implementation window
 * @author  Pete Goodliffe
 * @version 1.0
 * @internal
 */
class ListEditorImpl : public FilteredKListView,
                       public TSE3::Listener<TSE3::PhraseEditListener>
{
        Q_OBJECT;

    public:

        ListEditorImpl(TSE3::PhraseEdit *phraseEdit, QWidget *parent);

        const QPixmap &midiPixmap(size_t index) const
        {
            return midiIcons[index];
        }

        /**
         * @reimplemented
         */
        virtual void PhraseEdit_Reset(TSE3::PhraseEdit *);

        /**
         * @reimplemented
         */
        virtual void PhraseEdit_Tidied(TSE3::PhraseEdit *);

        /**
         * @reimplemented
         */
        virtual void PhraseEdit_Inserted(TSE3::PhraseEdit *, size_t index);

        /**
         * @reimplemented
         */
        virtual void PhraseEdit_Erased(TSE3::PhraseEdit *, size_t index);

        /**
         * @reimplemented
         */
        virtual void PhraseEdit_Selection(TSE3::PhraseEdit *, size_t index,
                                          bool selected);

    signals:

        /**
         * This signal is emitted whenever an event is clicked on.
         */
        void event(TSE3::MidiEvent);

    private slots:

        void slotClicked(QListViewItem *e, const QPoint &p, int column);
        void slotDoubleClicked(QListViewItem *e);
        void slotWantToSelect(QListViewItem *, bool add);
        void slotWantToClear(QListViewItem *);

        void slotChangeEvent(size_t index, TSE3::Clock time,
                             TSE3::Clock duration,
                             size_t channel, size_t port);
        void slotChangeData1(size_t index, int data1);
        void slotChangeData2(size_t index, int data2);

    private:

        void updateAllItems();

        TSE3::PhraseEdit *phraseEdit;
        int               lastClickedCol;

        QPixmap midiIcons[8];
};

/**
 * A @ref TSE3::Phrase editor that plugs into the @ref PhraseEditor window.
 *
 * It displays events in a time ordered list. It is the most basic editor
 * type, but shows the most information.
 *
 * You do not create these directly, but create them via the @ref PhraseEditor
 * window.
 *
 * @short   @ref TSE3::Phrase list editor window
 * @author  Pete Goodliffe
 * @version 1.0
 */
class ListEditor : public PhraseEditorBase,
                   public TSE3::Listener<TSE3::PhraseEditListener>
{
        Q_OBJECT

    public:

        ListEditor(TSE3::PhraseEdit *phraseEdit, KToolBar *toolbar,
                   QWidget *parent);
        virtual ~ListEditor();

        /**
         * @reimplemented
         */
        virtual void PhraseEdit_Selection(TSE3::PhraseEdit *,
                                          size_t /*index*/,
                                          bool /*selected*/);

    private slots:

        void slotToolbarClicked(int);
        void slotEvent(TSE3::MidiEvent);

    private:

        /**
         * Resizes the contents
         */
        void resizeContents();

        /**
         * Sets the status of the Toolbar_Delete button to reflect whether
         * or not any events are selected in the @ref TSE3::PhraseEdit.
         */
        void updateDeleteEnabled();

        enum Toolbar_Std { Toolbar_AddNote = PhraseEditor::Toolbar_NoButtons,
                           Toolbar_AddKeyPressure,
                           Toolbar_AddControl,         Toolbar_AddProgram,
                           Toolbar_AddChannelPressure, Toolbar_AddPitch,
                           Toolbar_AddSysEx,
                           Toolbar_Delete };

        // UI widgets
        ListEditorImpl *impl;

        TSE3::PhraseEdit *phraseEdit;

        // Current event state
        TSE3::Clock     lastTime;
        TSE3::Clock     lastDuration;
        int             lastChannel;
        int             lastPort;
        int             lastNote;
        int             lastVelocity;
        int             lastOffVelocity;
        int             lastControl;
        int             lastControlData;
        int             lastProgram;
        int             lastPitch;
};

/******************************************************************************
 * List data editors
 *****************************************************************************/

/**
 * The basic list editor dialogue to select information about an event.
 *
 * @short   @ref ListEditor editor dialogue
 * @author  Pete Goodliffe
 * @version 1.0
 */
class ListEditor_TimeChannelEditor : public KDialogBase
{
        Q_OBJECT

    public:

        /**
         * Creates a dialogue with the specified initial values. If the
         * dialogue is not editting a @ref TSE3::MidiCommand_NoteOn
         * @ref TSE3::MidiEvent then set @p duation to -1 and it will be
         * disabled.
         */
        ListEditor_TimeChannelEditor(size_t index, TSE3::Clock time,
                                     TSE3::Clock duration,
                                     size_t channel, size_t port,
                                     QWidget *parent);
    signals:

        /**
         * An item has been selected
         */
        void selected(size_t index, TSE3::Clock time, TSE3::Clock duration,
                      size_t channel, size_t port);

    public slots:

        void accept();

    private:

        size_t          index;
        ClockWidget    *time;
        ClockWidget    *duration;
        ChannelWidget  *channel;
        PortWidget     *port;
};

/**
 * Editor for selection of note values.
 *
 * @short   @ref ListEditor editor note selection dialogue
 * @author  Pete Goodliffe
 * @version 1.0
 */
class ListEditor_NoteEditor : public KDialogBase
{
        Q_OBJECT

    public:

        /**
         * Creates a dialogue with the specified initial value.
         */
        ListEditor_NoteEditor(size_t index, int note, QWidget *parent);
    signals:

        /**
         * An item has been selected
         */
        void selected(size_t index, int note);

    public slots:

        void accept();

    private:

        size_t             index;
        KeyWidget         *key;
        QSlider           *slider;
};

/**
 * Editor for selection of data values.
 *
 * @short   @ref ListEditor editor data selection dialogue
 * @author  Pete Goodliffe
 * @version 1.0
 */
class ListEditor_DataEditor : public KDialogBase
{
        Q_OBJECT

    public:

        /**
         * Creates a dialogue with the specified initial value.
         */
        ListEditor_DataEditor(size_t index, int note, QWidget *parent);
    signals:

        /**
         * An item has been selected
         */
        void selected(size_t index, int data);

    public slots:

        void accept();

    private:

        size_t             index;
        QSpinBox          *spinbox;
        QSlider           *slider;
};

#endif
