/*
 * @(#)PhraseEditor.cpp 1.00 8 September 2000
 *
 * Copyright (c) Pete Goodliffe 2000 (pete@cthree.org)
 *
 * This file is part of anthem - the TSE3 sequencer.
 *
 * This program is modifiable/redistributable under the terms of the GNU
 * General Public License.
 *
 * You should have recieved a copy of the GNU General Public License along
 * with this program; see the file COPYING. If not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 0211-1307, USA.
 */

#include "phraseeditors/PhraseEditor.h"

#include "Application.h"
#include "dialogues/NewPhrase.h"
#include "phraseeditors/ListEditor.h"
#include "phraseeditors/PianoRollEditor.h"

#include "misc/kde2-compat.h"
#include <qsplitter.h>
#include <qvbox.h>
#include <qlayout.h>
#include <kcombobox.h>
#include <kiconloader.h>
#include <klocale.h>
#include <kmessagebox.h>
#include <ktoolbar.h>

#include "tse3/PhraseEdit.h"
#include "tse3/PhraseList.h"
#include "tse3/Phrase.h"
#include "tse3/Transport.h"
#include "tse3/cmd/Phrase.h"
#include "tse3/cmd/CommandHistory.h"

/******************************************************************************
 * PhraseEditorBase class
 *****************************************************************************/

PhraseEditorBase::~PhraseEditorBase()
{
}


/******************************************************************************
 * Basic PhraseEditorFactory classes
 *****************************************************************************/

class ListEditorFactory : public PhraseEditorFactory
{
    public:
        virtual QString editorType() const { return "List"; }
        virtual PhraseEditorBase *createEditor(TSE3::PhraseEdit *phraseEdit,
                                               KToolBar         *toolbar,
                                               QWidget          *parent)
        {
            return new ListEditor(phraseEdit, toolbar, parent);
        }
};
class PianoRollEditorFactory : public PhraseEditorFactory
{
    public:
        virtual QString editorType() const { return "Piano roll"; }
        virtual PhraseEditorBase *createEditor(TSE3::PhraseEdit *phraseEdit,
                                               KToolBar         *toolbar,
                                               QWidget          *parent)
        {
            return new PianoRollEditor(phraseEdit, toolbar, parent);
        }
};
class DrumEditorFactory : public PhraseEditorFactory
{
    public:
        virtual QString editorType() const { return "Drum"; }
        virtual PhraseEditorBase *createEditor(TSE3::PhraseEdit *phraseEdit,
                                               KToolBar         *toolbar,
                                               QWidget          *parent)
        {
            std::cout << "Drum editor not implemented yet\n";
            return new ListEditor(phraseEdit, toolbar, parent);
        }
};
class StaveRollEditorFactory : public PhraseEditorFactory
{
    public:
        virtual QString editorType() const { return "Stave roll"; }
        virtual PhraseEditorBase *createEditor(TSE3::PhraseEdit *phraseEdit,
                                               KToolBar         *toolbar,
                                               QWidget          *parent)
        {
            std::cout << "Stave roll editor not implemented yet\n";
            return new ListEditor(phraseEdit, toolbar, parent);
        }
};


/******************************************************************************
 * PhraseEditorImpl class
 *****************************************************************************/

class PhraseEditorImpl
{
    public:

        // TSE3 objects
        TSE3::Song                *song;
        TSE3::Cmd::CommandHistory *history;
        TSE3::PhraseEdit          *phraseEdit;
        TSE3::Phrase              *originalPhrase;

        // UI widgets
        PhraseEditorBase          *impl;
        KComboBox                 *typeCombo;
        QVBoxLayout               *layout;
        NewPhraseWindow           *newPhraseWindow;

        // State
        int                        implType;
        static int                 lastImplType;
        int                        toolbarSelected;
        bool                       createdFromScratch;

        static std::vector<PhraseEditorFactory *> &factories()
        {
            static bool init = false;
            if (!init)
            {
                // This order is hardwired, as in the PhraseEditor::EditorType
                // enums
                _factories.push_back(new ListEditorFactory);
                _factories.push_back(new PianoRollEditorFactory);
                _factories.push_back(new DrumEditorFactory);
                _factories.push_back(new StaveRollEditorFactory);
                init = true;
            }
            return _factories;
        }

    private:

        // How we access different phrase editors
        static std::vector<PhraseEditorFactory *> _factories;
};

std::vector<PhraseEditorFactory *> PhraseEditorImpl::_factories;


/******************************************************************************
 * PhraseEditor class
 *****************************************************************************/

int PhraseEditorImpl::lastImplType = PhraseEditor::ListEditor;

PhraseEditor::PhraseEditor(QWidget                   *parent,
                           TSE3::Song                *s,
                           TSE3::Cmd::CommandHistory *h,
                           TSE3::Phrase              *p,
                           int                        et)
: KDialog(parent, "PhraseEditor window", false, WDestructiveClose),
  pimpl(new PhraseEditorImpl)
{
    pimpl->song               = s;
    pimpl->history            = h;
    pimpl->phraseEdit         = new TSE3::PhraseEdit();
    pimpl->originalPhrase     = p;
    pimpl->impl               = 0;
    pimpl->newPhraseWindow    = 0;
    pimpl->createdFromScratch = (p == 0);

    pimpl->phraseEdit->reset(p);

    TSE3::Listener<TSE3::TransportListener>::attachTo
        (Application::application()->transport());
    TSE3::Listener<TSE3::PhraseEditListener>::attachTo(pimpl->phraseEdit);
    if (p) TSE3::Listener<TSE3::PhraseListener>::attachTo(p);

    updateCaption();

    toolbar = new KToolBar(this, "PhraseEditor::toolbar", true);
    pimpl->layout = new QVBoxLayout(this);
    pimpl->layout->addWidget(toolbar);

    // Add the standard buttons
    pimpl->typeCombo = new KComboBox(toolbar);
    pimpl->typeCombo->setInsertionPolicy(QComboBox::NoInsertion);
    for (int n = 0; n < numEditors(); ++n)
    {
        pimpl->typeCombo->insertItem(editorType(n));
    }
    connect(pimpl->typeCombo, SIGNAL(activated(int)),
            SLOT(slotEditorType(int)));
    toolbar->insertWidget(Toolbar_Type, pimpl->typeCombo->sizeHint().width(),
                          pimpl->typeCombo);
    toolbar->insertLineSeparator();

    toolbar->insertButton(UserIcon("a_stop"), Toolbar_Stop,
                          true, i18n("Stop"));
    toolbar->insertButton(UserIcon("a_play"), Toolbar_Play,
                          true, i18n("Play"));
    toolbar->insertButton(UserIcon("tick"), Toolbar_Accept,
                          true, i18n("Accept edit"));
    toolbar->insertButton(UserIcon("cross"), Toolbar_Reject,
                          true, i18n("Reject edit and close"));
    toolbar->insertButton(BarIcon("reload"), Toolbar_Revert,
                          true, i18n("Revert to original"));
    toolbar->insertLineSeparator();
    connect(toolbar, SIGNAL(clicked(int)), SLOT(slotToolbar(int)));

    // Set up the toolbar
    pimpl->toolbarSelected = 0xffffffff; // invalid
    Transport_Status(Application::application()->transport(),
                     Application::application()->transport()->status());

    // Create the editor
    slotEditorType(et == -1 ? pimpl->lastImplType : pimpl->implType);
}


PhraseEditor::~PhraseEditor()
{
    delete pimpl->impl;
    delete pimpl->phraseEdit;
    delete pimpl;
}


int PhraseEditor::numEditors()
{
    return PhraseEditorImpl::factories().size();
}


QString PhraseEditor::editorType(int index)
{
    return PhraseEditorImpl::factories()[index]->editorType();
}


int PhraseEditor::addEditor(PhraseEditorFactory *factory)
{
    PhraseEditorImpl::factories().push_back(factory);
    return PhraseEditorImpl::factories().size()-1;
}


void PhraseEditor::updateCaption()
{
    QString string("Editing phrase");
    if (pimpl->originalPhrase)
    {
        string.append(" \"");
        string.append(pimpl->originalPhrase->title().c_str());
        string.append("\"");
    }
    else if (pimpl->createdFromScratch)
    {
        string.append(" [new]");
    }
    else
    {
        string.append(" [deleted]");
    }
    if (pimpl->phraseEdit->modified())
    {
        string.append(" [modified]");
    }
    setCaption(string);
}


void PhraseEditor::slotEditorType(int editorType)
{
    delete pimpl->impl;

    if (editorType < numEditors())
    {
        pimpl->impl
            = PhraseEditorImpl::factories()[editorType]
                ->createEditor(pimpl->phraseEdit, toolbar, this);
    }

    pimpl->layout->add(pimpl->impl);
    pimpl->impl->show();
    pimpl->lastImplType = pimpl->implType = editorType;
}


void PhraseEditor::slotToolbar(int item)
{
    switch (item)
    {
        case Toolbar_Play:
        {
            Application::application()->transport()->play(pimpl->phraseEdit, 0);
            break;
        }
        case Toolbar_Stop:
        {
            Application::application()->transport()->stop();
            if (!toolbar->isButtonOn(Toolbar_Stop))
            {
                // This is a bit nasty, but stop is the only icon we have
                // to watch out for being erroneuously deselected.
                toolbar->setButton(Toolbar_Stop, true);
            }
            break;
        }
        case Toolbar_Accept:
        {
            if (!pimpl->newPhraseWindow)
            {
                QString t = "";
                if (pimpl->originalPhrase)
                {
                    t.append(pimpl->originalPhrase->title().c_str());
                }
                pimpl->newPhraseWindow
                    = new NewPhraseWindow(this, pimpl->song->phraseList(), t,
                                          true, false, true);
                connect(pimpl->newPhraseWindow,
                        SIGNAL(newPhraseDone(bool,QString,bool,bool,bool)),
                        SLOT(slotNewPhraseDone(bool,QString,bool,bool,bool)));
                pimpl->newPhraseWindow->show();
            }
            break;
        }
        case Toolbar_Reject:
        {
            if (pimpl->phraseEdit->modified() && messageBoxReject())
            {
                return;
            }
            delete this;
            break;
        }
        case Toolbar_Revert:
        {
            if (pimpl->phraseEdit->modified())
            {
                if (KMessageBox::questionYesNo(this, "You have altered this phrase.\nAre you sure that you want to lose your changes and revert to the original phrase?") == KMessageBox::No)
                {
                    return;
                }
                pimpl->phraseEdit->reset(pimpl->originalPhrase);
            }
            break;
        }
    }
}


void PhraseEditor::Transport_Status(TSE3::Transport *, int newStatus)
{
    if (newStatus != pimpl->toolbarSelected)
    {
        // Reset last icon
        switch (pimpl->toolbarSelected)
        {
            case TSE3::Transport::Resting:
            {
                toolbar->setButton(Toolbar_Stop, false);
                break;
            }
            case TSE3::Transport::Playing:
            case TSE3::Transport::SynchroPlaying:
            {
                toolbar->setButton(Toolbar_Play, false);
                break;
            }
        }
        // Set new icon
        switch (newStatus)
        {
            case TSE3::Transport::Resting:
            {
                toolbar->setButton(Toolbar_Stop, true);
                break;
            }
            case TSE3::Transport::Playing:
            case TSE3::Transport::SynchroPlaying:
            {
                toolbar->setButton(Toolbar_Play, true);
                break;
            }
        }
        pimpl->toolbarSelected = newStatus;
    }
}


void PhraseEditor::slotNewPhraseDone(bool success, QString title,
                                     bool replacePhrase, bool insertPart,
                                     bool keepOpen)
{
    pimpl->newPhraseWindow->delayedDestruct();
    pimpl->newPhraseWindow = 0;

    if (success)
    {
        if (!pimpl->originalPhrase && !replacePhrase
            && pimpl->song->phraseList()->phrase(title.latin1()))
        {
            KMessageBox::sorry(this,
                               "A Phrase with that name already exists");
            return;
        }

        TSE3::Phrase *phrase = 0;
        TSE3::Phrase *existing
            = pimpl->originalPhrase ? pimpl->originalPhrase : 0;

        if (replacePhrase && existing)
        {
            TSE3::Cmd::Phrase_Replace *cmd
                = new TSE3::Cmd::Phrase_Replace(pimpl->originalPhrase,
                                                pimpl->phraseEdit,
                                                pimpl->song,
                                                title.latin1());
            cmd->execute();
            phrase = cmd->phrase();
            if (pimpl->history)
            {
                pimpl->history->add(cmd);
            }
            else
            {
                delete cmd;
            }
        }
        else
        {
            TSE3::Cmd::Phrase_Create *cmd
                = new TSE3::Cmd::Phrase_Create(pimpl->song->phraseList(),
                                               pimpl->phraseEdit,
                                               title.latin1());
            cmd->execute();
            phrase = cmd->phrase();
            if (pimpl->history)
            {
                pimpl->history->add(cmd);
            }
            else
            {
                delete cmd;
            }
        }

        if (!keepOpen)
        {
            delete this;
        }
        else
        {
            pimpl->phraseEdit->setModified(false);
            if (pimpl->originalPhrase)
            {
                // If we did a "replace phrase" operation, this might
                // have become invalid as the old Phrase moved out of the
                // PhraseList
                TSE3::Listener<TSE3::PhraseListener>
                    ::detachFrom(pimpl->originalPhrase);
            }
            TSE3::Listener<TSE3::PhraseListener>::attachTo(phrase);
            pimpl->originalPhrase = phrase;
            updateCaption();
        }
    }
}


void PhraseEditor::Notifier_Deleted(TSE3::Phrase *p)
{
    pimpl->originalPhrase = 0;
}


void PhraseEditor::Phrase_Reparented(TSE3::Phrase *p)
{
    TSE3::Listener<TSE3::PhraseListener>::detachFrom(p);
    pimpl->originalPhrase = 0;
    updateCaption();
}


void PhraseEditor::Phrase_TitleAltered(TSE3::Phrase *p)
{
    updateCaption();
}


void PhraseEditor::PhraseEdit_Modified(TSE3::PhraseEdit *, bool)
{
    updateCaption();
}


void PhraseEditor::closeEvent(QCloseEvent *e)
{
    if (!pimpl->phraseEdit->modified() || !messageBoxReject())
    {
        e->accept();
    }

}


bool PhraseEditor::messageBoxReject()
{
    return KMessageBox::questionYesNo(this, "You have altered this phrase.\nAre you sure that you want to lose your changes and close the editor?", QString::null, KGuiItem("Yes, close the editor"), KGuiItem("No, keep the editor open"))
        == KMessageBox::No;
}

