/*
 * @(#)PianoRollEditor.cpp 1.00 28 April 2002
 *
 * Copyright (c) Pete Goodliffe 2000 (pete@cthree.org)
 *
 * This file is part of anthem - the TSE3 sequencer.
 *
 * This program is modifiable/redistributable under the terms of the GNU
 * General Public License.
 *
 * You should have recieved a copy of the GNU General Public License along
 * with this program; see the file COPYING. If not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 0211-1307, USA.
 */

#include "phraseeditors/PianoRollEditor.h"

#include "tse3/kdeui/ClockWidget.h"

#include "tse3/PhraseEdit.h"
#include "tse3/util/NoteNumber.h"

#include <qwidget.h>
#include <qlayout.h>
#include <qpainter.h>
#include <ktoolbar.h>


/******************************************************************************
 * File local definitions
 *****************************************************************************/

class NoteDisplayWidget : public QWidget
{
    public:

        NoteDisplayWidget(TSE3::PhraseEdit *phraseEdit, QWidget *parent);

        virtual void drawContents
            (QPainter *p, int clipx, int clipy, int clipw, int cliph);

    private:

        enum
        {
            WIDTH_EXTRASPACE = 200
        };

        int clockToPixel(TSE3::Clock c) { return c * scaleNum / scaleDiv; }
        TSE3::Clock pixelToClock(int p) { return p * scaleDiv / scaleNum; }

        int noteToPixel(int n)          { return n * noteHeight; }
        int pixelToNote(int p)          { return p / noteHeight; }

        void updateSize();

        TSE3::PhraseEdit *phraseEdit;
        int               scaleNum;
        int               scaleDiv;
        int               noteHeight;
        bool              horizontalRules;
        bool              verticalRules;
};


NoteDisplayWidget::NoteDisplayWidget(TSE3::PhraseEdit *pe, QWidget *parent)
: QWidget(parent), phraseEdit(pe),
  scaleNum(1), scaleDiv(1),
  noteHeight(10),
  horizontalRules(true), verticalRules(true)
{
    updateSize();
}


void NoteDisplayWidget::drawContents
            (QPainter *p, int clipx, int clipy, int clipw, int cliph)
{
    TSE3::Clock start   = pixelToClock(clipx);
    TSE3::Clock end     = pixelToClock(clipx+clipw);
    size_t      minnote = pixelToNote(clipy);
    size_t      maxnote = pixelToNote(clipy+cliph);

    if (verticalRules)
    {
        TSE3::Clock min = start, max = end;
        // snap
        QPen pen(p->pen());
        pen.setColor(QColor(0x66, 0x66, 0x66));
        pen.setStyle(DotLine);
        p->setPen(pen);
        while (min <= max)
        {
            if (1)//!min%4)
            {
                int x = clockToPixel(min);
                p->drawLine(x, 0, x, height());
            }
            min += TSE3::Clock::PPQN*4;
        }
    }
}


void NoteDisplayWidget::updateSize()
{
    int width = clockToPixel(phraseEdit->lastClock()) + WIDTH_EXTRASPACE;
    setMinimumSize(width, 127*noteHeight);
    setMaximumSize(width, 127*noteHeight);
}


/******************************************************************************
 * PianoRollEditorImpl class
 *****************************************************************************/

PianoRollEditorImpl::PianoRollEditorImpl(TSE3::PhraseEdit *phraseEdit, QWidget *parent)
: QScrollView(parent, "PianoRollEditorImpl"), phraseEdit(phraseEdit)
{
    attachTo(phraseEdit);

    ndw = new NoteDisplayWidget(phraseEdit, this);
    addChild(ndw);
    setResizePolicy(QScrollView::AutoOneFit);
}


void PianoRollEditorImpl::PhraseEdit_Reset(TSE3::PhraseEdit *)
{
}


void PianoRollEditorImpl::PhraseEdit_Tidied(TSE3::PhraseEdit *)
{
}


void PianoRollEditorImpl::PhraseEdit_Inserted(TSE3::PhraseEdit *, size_t index)
{
}


void PianoRollEditorImpl::PhraseEdit_Erased(TSE3::PhraseEdit *, size_t index)
{
}


void PianoRollEditorImpl::PhraseEdit_Selection(TSE3::PhraseEdit *, size_t index,
                                          bool selected)
{
}


/******************************************************************************
 * PianoRollEditor class
 *****************************************************************************/

PianoRollEditor::PianoRollEditor(TSE3::PhraseEdit *pe, KToolBar *toolbar,
                       QWidget *parent)
: PhraseEditorBase(pe, toolbar, parent),
  phraseEdit(pe)
{
    connect(toolbar, SIGNAL(clicked(int)), SLOT(slotToolbarClicked(int)));

    QVBoxLayout *layout = new QVBoxLayout(this);
    impl = new PianoRollEditorImpl(phraseEdit, this);
    layout->addWidget(impl);

    connect(impl, SIGNAL(event(TSE3::MidiEvent)),
            SLOT(slotEvent(TSE3::MidiEvent)));

    attachTo(phraseEdit);
    updateDeleteEnabled();
}


PianoRollEditor::~PianoRollEditor()
{
}


void PianoRollEditor::slotToolbarClicked(int item)
{
    switch (item)
    {
        case Toolbar_Something:
        {
        }
        case Toolbar_Delete:
        {
            phraseEdit->eraseSelection();
            break;
        }
    }
}


void PianoRollEditor::PhraseEdit_Selection(TSE3::PhraseEdit *,
                                      size_t /*index*/,
                                      bool /*selected*/)
{
    updateDeleteEnabled();
}


void PianoRollEditor::updateDeleteEnabled()
{
    toolbar->setItemEnabled(Toolbar_Delete, phraseEdit->selection());
}


