/*
 * @(#)Instrument.cpp 1.00 11 Oct 2000
 *
 * Copyright (c) Pete Goodliffe 2000 (pete@cthree.org)
 *
 * This file is part of anthem - the TSE3 sequencer.
 *
 * This program is modifiable/redistributable under the terms of the GNU
 * General Public License.
 *
 * You should have recieved a copy of the GNU General Public License along
 * with this program; see the file COPYING. If not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 0211-1307, USA.
 */

#include "settings/Instrument.h"

#include "Application.h"
#include "dialogues/TSE3Progress.h"

#include "tse3/ins/Instrument.h"
#include "tse3/ins/Destination.h"
#include "tse3/MidiScheduler.h"

#include <kdialogbase.h>
#include <qlayout.h>
#include <qlistview.h>
#include <qgroupbox.h>
#include <qhbox.h>
#include <qpushbutton.h>
#include <kfiledialog.h>
#include <qcombobox.h>
#include <kglobalsettings.h>

#include "Application.h"

namespace
{
    const char *NONE_STR = "<None>";
}


/******************************************************************************
 * InstrumentWidget class
 *****************************************************************************/

InstrumentWidget::InstrumentWidget(QWidget *parent)
: SettingsWidget(new SettingsImpl<TSE3::Ins::DestinationListener>
      (Application::application()->destination()), parent, "InstrumentWidget")
{
    QHBoxLayout *layout = new QHBoxLayout(this, KDialogBase::spacingHint());

    QGroupBox *gb1 = new QGroupBox(1, Horizontal, "Destinations", this);
    QGroupBox *gb2 = new QGroupBox(1, Horizontal, "Instruments", this);
    layout->addWidget(gb1);
    layout->addWidget(gb2);

    destBox = new QListView(gb1);
    insBox  = new KListBox(gb2);
    combo   = new QComboBox(gb1);
    QHBox *hbox = new QHBox(gb2);
    QPushButton *add = new QPushButton("Add", hbox);
    QPushButton *del = new QPushButton("Remove", hbox);
    connect(add, SIGNAL(clicked()), this, SLOT(slotAddInstrument()));
    connect(del, SIGNAL(clicked()), this, SLOT(slotDeleteInstrument()));

    destBox->addColumn("Port");
    destBox->addColumn("Channel");
    destBox->addColumn("Instrument");
    destBox->setRootIsDecorated(true);
    destBox->setSorting(-1);
    destBox->setAllColumnsShowFocus(true);

    connect(destBox, SIGNAL(selectionChanged()), SLOT(slotSelectionChanged()));
    connect(combo,   SIGNAL(activated(int)),     SLOT(slotComboActivated(int)));

    attachTo(Application::application()->destination());
    updateDestinations();
    updateInstruments();
    updateComboEnabled();
    updateComboItems();
}


void InstrumentWidget::slotAddInstrument()
{
    // Add new instrument
    QString file = KFileDialog::getOpenFileName
        ("", "*.ins", this, "Select an instrument definition file");
    if (file)
    {
        InstrumentChoiceWindow *icw = new InstrumentChoiceWindow(file, this);
        connect(icw, SIGNAL(instrument(TSE3::Ins::Instrument*)),
                SLOT(slotInstrument(TSE3::Ins::Instrument*)));
        icw->show();
    }
    updateComboItems();
}


void InstrumentWidget::slotDeleteInstrument()
{
    // Delete instrument
    int n = 0;
    while (!insBox->isSelected(n) && n < insBox->numRows())
    {
        n++;
    }
    if (n < insBox->numRows())
    {
        TSE3::Ins::Destination *dest
            = Application::application()->destination();
        TSE3::Ins::Instrument *ins = dest->instrument(n);
        dest->removeInstrument(ins);
        delete ins;
    }
    updateComboItems();
}


void InstrumentWidget::slotInstrument(TSE3::Ins::Instrument *instrument)
{
    Application::application()->destination()->addInstrument(instrument);
}


void InstrumentWidget::slotSelectionChanged()
{
    updateComboEnabled();
    if (QListViewItem *lvi = destBox->selectedItem())
    {
        for (int n = 0; n < combo->count(); n++)
        {
            if (combo->text(n) == lvi->text(2))
            {
                combo->setCurrentItem(n);
                break;
            }
        }
    }
    else
    {
        combo->setCurrentItem(0); // i.e. "None"
    }
}


void InstrumentWidget::slotComboActivated(int item)
{
    if (QListViewItem *lvi = destBox->selectedItem())
    {
        lvi->setText(2, combo->text(item));
        if (!lvi->parent())
        {
            QListViewItem *child = lvi->firstChild();
            while (child)
            {
                child->setSelectable(!item);
                child = child->nextSibling();
            }
        }
    }
}


void InstrumentWidget::Destination_Altered(TSE3::Ins::Destination *,
                                           size_t /*channel*/,
                                           size_t /*port*/,
                                           TSE3::Ins::Instrument *)
{
    updateInstruments();
    updateComboItems();
}


void InstrumentWidget::Destination_InstrumentAdded(TSE3::Ins::Destination*,
                                                   TSE3::Ins::Instrument *)
{
    updateInstruments();
    updateComboItems();
}


void InstrumentWidget::Destination_InstrumentRemoved(TSE3::Ins::Destination *,
                                                     TSE3::Ins::Instrument *)
{
    updateInstruments();
    updateComboItems();
}


void InstrumentWidget::updateComboEnabled()
{
    combo->setEnabled(destBox->selectedItem() != 0);
}


void InstrumentWidget::updateComboItems()
{
    combo->clear();
    combo->insertItem(NONE_STR);
    TSE3::Ins::Destination *dest = Application::application()->destination();
    for (size_t n = 0; n < dest->numInstruments(); n++)
    {
        combo->insertItem(dest->instrument(n)->title().c_str());
    }
}


void InstrumentWidget::updateDestinations()
{
    destBox->clear();
    TSE3::Ins::Destination *dest = Application::application()->destination();
    TSE3::MidiScheduler    *sch  = Application::application()->scheduler();

    for (size_t port = 0; port < sch->numPorts(); port++)
    {
        QString sport, sportdest(NONE_STR);
        sport.sprintf("Port %d", port+1);
        if (dest->port(port))
        {
            sportdest.sprintf("%s", dest->port(port)->title().c_str());
        }
        QListViewItem *lvi = new QListViewItem(destBox, sport, "", sportdest);
        QListViewItem *last = 0;
        for (size_t channel = 0; channel < 16; channel++)
        {
            QString schannel, schanneldest;
            sport.sprintf("%d", port+1);
            if (dest->channel(channel, port))
            {
                schanneldest.sprintf
                    ("%s", dest->channel(channel, port)->title().c_str());
            }
            else
            {
                schanneldest.sprintf(NONE_STR);
            }
            schannel.sprintf("%d", channel+1);
            last = new QListViewItem(lvi, last, sport, schannel, schanneldest);
            last->setSelectable(dest->allChannels(port));
        }
        lvi->setOpen(dest->allChannels(port));
    }
}


void InstrumentWidget::updateInstruments()
{
    insBox->clear();
    TSE3::Ins::Destination *dest = Application::application()->destination();
    for (size_t n = 0; n < dest->numInstruments(); n++)
    {
        insBox->insertItem(dest->instrument(n)->title().c_str());
    }
}


void InstrumentWidget::setValues()
{
    TSE3::Ins::Destination *dest = Application::application()->destination();
    QListViewItem          *lvi  = destBox->firstChild();
    size_t                  port = 0;
    while (lvi)
    {
        const QString insName = lvi->text(2);
        if (insName == NONE_STR)
        {
            dest->setPort(port, 0);
            QListViewItem *ch_lvi  = lvi->firstChild();
            size_t         channel = 0;
            while (ch_lvi)
            {
                const QString insName = ch_lvi->text(2);
                dest->setChannel(channel, port,
                                 dest->instrument(insName.latin1()));
                ch_lvi = ch_lvi->nextSibling();
                channel++;
            }
        }
        else
        {
            dest->setPort(port, dest->instrument(insName.latin1()));
        }

        lvi = lvi->nextSibling();
        port++;
    }
}


void InstrumentWidget::updateWidgets()
{
    updateDestinations();
    updateComboEnabled();
}


/******************************************************************************
 * InstrumentChoiceWindow class
 *****************************************************************************/

InstrumentChoiceWindow::InstrumentChoiceWindow(QString file, QWidget *parent)
: KDialogBase(Plain, "Instruments", Close|Ok, Close, parent,
              "Instruments", false, true)
{
    QVBoxLayout *layout
        = new QVBoxLayout(plainPage(), KDialogBase::spacingHint());

    listBox = new KListBox(plainPage());
    layout->addWidget(listBox);

    TSE3ProgressDialog prog("Reading instrument definitions", QString::null);
    cif = new TSE3::Ins::CakewalkInstrumentFile(file.latin1());
    std::list<std::string>::const_iterator i = cif->instruments(&prog).begin();
    while (i != cif->instruments().end())
    {
        listBox->insertItem((*i).c_str());
        i++;
    }

    connect(listBox, SIGNAL(doubleClicked(QListBoxItem*)),
            SLOT(slotExecuted(QListBoxItem*)));
}


InstrumentChoiceWindow::~InstrumentChoiceWindow()
{
    delete cif;
}


void InstrumentChoiceWindow::accept()
{
    hide(); // We want to do this before producing the TSE3ProgressDialog

    int n = 0;
    while (!listBox->isSelected(n) && n < listBox->numRows())
    {
        n++;
    }
    if (n < listBox->numRows())
    {
        std::list<std::string>::const_iterator i = cif->instruments().begin();
        for (int m = 0; m < n; m++, i++);
        TSE3ProgressDialog prog("Reading instrument definition", QString::null);
        TSE3::Ins::Instrument *ins = cif->instrument(*i, &prog);
        if (ins) emit instrument(ins);
    }
    delayedDestruct();
}


void InstrumentChoiceWindow::slotClose()
{
    delayedDestruct();
}


void InstrumentChoiceWindow::slotExecuted(QListBoxItem *item)
{
    accept();
}


