/*
 * @(#)MidiFilter.cpp 1.00 1 June 2000
 *
 * Copyright (c) Pete Goodliffe 2000 (pete@cthree.org)
 *
 * This file is part of anthem - the TSE3 sequencer.
 *
 * This program is modifiable/redistributable under the terms of the GNU
 * General Public License.
 *
 * You should have recieved a copy of the GNU General Public License along
 * with this program; see the file COPYING. If not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 0211-1307, USA.
 */

#include "settings/MidiFilter.h"

#include "tse3/kdeui/ClockWidget.h"
#include "tse3/kdeui/PortWidget.h"
#include "tse3/kdeui/ChannelWidget.h"
#include "Application.h"

#include <qlabel.h>
#include <qlayout.h>
#include <qgroupbox.h>
#include <qcheckbox.h>
#include <qradiobutton.h>
#include <qspinbox.h>
#include <qbuttongroup.h>

#include <kdialogbase.h>

#include "tse3/MidiFilter.h"

/******************************************************************************
 * MidiFilterWidget class
 *****************************************************************************/

MidiFilterWidget::MidiFilterWidget(TSE3::MidiFilter *mf,
                                   QWidget *parent, const char *name)
: SettingsWidget(new SettingsImpl<TSE3::MidiFilterListener>(mf), parent, name),
  noPorts(3), mf(mf)
{
    QVBoxLayout *layout = new QVBoxLayout(this, KDialogBase::spacingHint());

    status = new QCheckBox("Status (MIDI output enabled)", this);
    layout->addWidget(status);

    QHBoxLayout *hlayout = new QHBoxLayout(layout);

    // Channel filters
    QGroupBox *groupBox = new QGroupBox(8, Horizontal, "Channel filter", this);
    hlayout->addWidget(groupBox);
    for (size_t n = 0; n < 16; n++)
    {
        QString s;
        s.sprintf("%i", n+1);
        channelFilter[n] = new QCheckBox(s, groupBox);
    }

    QGridLayout *gl = new QGridLayout(layout, 5, 4);

    // Destination
    QLabel *label = new QLabel("Channel", this);
    label->setAlignment(AlignRight|AlignVCenter);
    gl->addWidget(label, 0, 0);
    channel = new ChannelWidget(true, true, true, 0, this);
    gl->addWidget(channel, 0, 1);
    label = new QLabel("Port", this);
    label->setAlignment(AlignRight|AlignVCenter);
    gl->addWidget(label, 0, 2);
    port = new PortWidget(true, true, true, true, 0, this);
    gl->addWidget(port, 0, 3);

    // Offset
    label = new QLabel("Offset", this);
    label->setAlignment(AlignRight|AlignVCenter);
    gl->addWidget(label, 1, 0);
    offset = new ClockWidget(this, false);
    offset->setMaximumSize(offset->sizeHint());
    offset->setSpecialValueText("Off");
    gl->addWidget(offset, 1, 1);
    backwards = new QRadioButton("Backwards", this);
    gl->addWidget(backwards, 1, 2);
    forwards = new QRadioButton("Forwards", this);
    gl->addWidget(forwards, 1, 3);
    QButtonGroup *bg = new QButtonGroup();
    bg->insert(backwards);
    bg->insert(forwards);

    // Other timing
    label = new QLabel("Time scale", this);
    label->setAlignment(AlignRight|AlignVCenter);
    gl->addWidget(label, 2, 0);
    timeScale = new QSpinBox(0, 9999, 1, this);
    timeScale->setSuffix("%");
    gl->addWidget(timeScale, 2, 1);

    label = new QLabel("Quantise", this);
    label->setAlignment(AlignRight|AlignVCenter);
    gl->addWidget(label, 2, 2);
    quantise = new ClockWidget(this, false);
    quantise->setSpecialValueText("Off");
    gl->addWidget(quantise, 2, 3);

    label = new QLabel("Transpose", this);
    label->setAlignment(AlignRight|AlignVCenter);
    gl->addWidget(label, 3, 0);
    transpose = new QSpinBox(-127, 127, 1, this);
    transpose->setValue(0);
    transpose->setSuffix(" semi-tones");
    gl->addWidget(transpose, 3, 1);

    label = new QLabel("Velocity scale", this);
    label->setAlignment(AlignRight|AlignVCenter);
    gl->addWidget(label, 3, 2);
    velScale = new QSpinBox(0, 9999, 1, this);
    velScale->setValue(0);
    velScale->setSuffix("%");
    gl->addWidget(velScale, 3, 3);

    label = new QLabel("Min velocity", this);
    label->setAlignment(AlignRight|AlignVCenter);
    gl->addWidget(label, 4, 0);
    minVel = new QSpinBox(0, 127, 1, this);
    minVel->setSpecialValueText("Off");
    minVel->setWrapping(true);
    gl->addWidget(minVel, 4, 1);

    // Remember that I have fiddled the end values of this spin box
    // so that I can have some 'specal value text'
    label = new QLabel("Max velocity", this);
    label->setAlignment(AlignRight|AlignVCenter);
    gl->addWidget(label, 4, 2);
    maxVel = new QSpinBox(-1, 126, 1, this);
    maxVel->setSpecialValueText("Off");
    maxVel->setWrapping(true);
    gl->addWidget(maxVel, 4, 3);

    connect(offset, SIGNAL(valueChanged(int)), SLOT(handleOffsetChanged()));
    updateWidgets();
}


MidiFilterWidget::~MidiFilterWidget()
{
}


void MidiFilterWidget::setValues()
{
    setValues(*mf);
}


void MidiFilterWidget::updateWidgets()
{
    status->setChecked(mf->status());
    for (size_t c = 0; c < 16; c++)
    {
        channelFilter[c]->setChecked(mf->channelFilter(c));
    }
    channel->setValue(mf->channel());
    port->setValue(mf->port());
    offset->setValue(abs(mf->offset()));
    backwards->setChecked(mf->offset() < 0);
    forwards->setChecked(mf->offset() >= 0);
    timeScale->setValue(mf->timeScale());
    quantise->setValue(mf->quantise());
    transpose->setValue(mf->transpose());
    velScale->setValue(mf->velocityScale());
    minVel->setValue(mf->minVelocity());
    int maxvelval = mf->maxVelocity();
    if (maxvelval == 127) maxvelval = -1;
    maxVel->setValue(maxvelval);
    handleOffsetChanged();
}


void MidiFilterWidget::handleOffsetChanged()
{
    backwards->setEnabled(offset->value());
    forwards->setEnabled(offset->value());
}


void MidiFilterWidget::setValues(TSE3::MidiFilter &mf)
{
    mf.setStatus(status->isChecked());
    for (size_t c = 0; c < 16; c++)
    {
        mf.setChannelFilter(c, channelFilter[c]->isChecked());
    }
    mf.setChannel(channel->value());
    mf.setPort(port->value());
    TSE3::Clock offsetval = offset->value();
    if (backwards->isChecked()) offsetval = -offsetval;
    mf.setOffset(offsetval);
    mf.setTimeScale(timeScale->value());
    mf.setQuantise(quantise->value());
    mf.setTranspose(transpose->value());
    mf.setVelocityScale(velScale->value());
    mf.setMinVelocity(minVel->value());
    int maxvelval = maxVel->value();
    if (maxvelval == -1) maxvelval = 127;
    mf.setMaxVelocity(maxvelval);
}

