/*
 * @(#)PartView.h 1.00 15 November 2000
 *
 * Copyright (c) Pete Goodliffe 2000 (pete@cthree.org)
 *
 * This file is part of anthem - the TSE3 sequencer.
 *
 * This program is modifiable/redistributable under the terms of the GNU
 * General Public License.
 *
 * You should have recieved a copy of the GNU General Public License along
 * with this program; see the file COPYING. If not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 0211-1307, USA.
 */

#ifndef ANTHEM_SETTINGS_PARTVIEW_H
#define ANTHEM_SETTINGS_PARTVIEW_H

#include "settings/SettingsWidget.h"

#include "songwindow/SongWindow.h"
#include "songwindow/PartView.h"

class SongWindow;
class QCheckBox;
class KComboBox;

/**
 * This class provides a list of every @ref SongWindow in the currently
 * running application.
 *
 * It also stores a default set of PartView::ViewOptions that can be
 * saved with the application choices and has functionality to apply this
 * to every @ref PartView child.
 *
 * @ref SongWindows register with this class when they are created.
 *
 * This class is a singleton; there is only one PartViewList per application
 * instance.
 *
 * @short   PartView settings widget.
 * @author  Pete Goodliffe
 * @version 1.0
 */
class SongWindowList
{
    public:

        /**
         * Returns a pointer to the singleton PartViewList object.
         */
        static SongWindowList *songWindowList();

        /**
         * Sets the ViewOptions for every child @ref PartView,
         * and sets this as the stored default options.
         */
        void setPVViewOptions(const PartView::ViewOptions &options);

        /**
         * Sets the ViewOptions for every child @ref SongWindow,
         * and sets this as the stored default options.
         */
        void setSWViewOptions(const SongWindow::ViewOptions &options);

        /**
         * Sets the follow cursor value for every child @ref TimeLine,
         * and sets this as the stored default options.
         */
        void setFollowCursor(bool fc);

        /**
         * Returns the current default @PartView::ViewOptions.
         */
        const PartView::ViewOptions &pvViewOptions() const { return pvoptions; }

        /**
         * Returns the current default @PartView::ViewOptions.
         */
        const SongWindow::ViewOptions &swViewOptions() const
            { return swoptions; }

        /**
         * Returns the current default @ref TimeLine::followCursor setting.
         */
        bool followCursor() const { return _followCursor; }

        /**
         * Only the SongWindow class can access add and remove instances.
         */
        friend class SongWindow;

    private:

        /**
         * Registers a SongWindow
         */
        void add(SongWindow *songWindow);

        /**
         * Deregisters a SongWindow
         */
        void remove(SongWindow *songWindow);

    private:

        SongWindowList();

        PartView::ViewOptions   pvoptions;
        SongWindow::ViewOptions swoptions;
        bool                    _followCursor;
};

/**
 * This class provides a widget that alters a @ref PartView.
 *
 * @short   PartView settings widget.
 * @author  Pete Goodliffe
 * @version 1.0
 */
class PartViewWidget : public SettingsWidget
{
    public:
        /**
         * Specify the @ref TSE3::Part to be altered, and also the
         * @ref TSE3::Song it is in (needed to create the
         * @ref PhraseSelectWidget).
         */
        PartViewWidget(QWidget *parent, const char *name = 0);
        virtual ~PartViewWidget();
        virtual void setValues();
        virtual void updateWidgets();

    private:

        QCheckBox *verticalRules;
        QCheckBox *phraseName;
        QCheckBox *phraseEvents;
        QCheckBox *plainBoxes;
        QCheckBox *fillBoxes;
        KComboBox *colours;
        QCheckBox *followCursor;
        QCheckBox *lmbSetsClock;
        QCheckBox *rmbSetsClock;
        QCheckBox *dragSetsMarkers;
};

#endif
