/*
 * @(#)Plugins.cpp 1.00 1 June 2000
 *
 * Copyright (c) Pete Goodliffe 2000 (pete@cthree.org)
 *
 * This file is part of anthem - the TSE3 sequencer.
 *
 * This program is modifiable/redistributable under the terms of the GNU
 * General Public License.
 *
 * You should have recieved a copy of the GNU General Public License along
 * with this program; see the file COPYING. If not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 0211-1307, USA.
 */

#include "settings/Plugins.h"

#include "Application.h"

#include <qlayout.h>
#include <qvbox.h>
#include <qgroupbox.h>
#include <qpushbutton.h>
#include <qlistbox.h>

#include <kdialogbase.h>
#include <kiconloader.h>
#include <kfiledialog.h>
#include <kmessagebox.h>

#include <algorithm>

/******************************************************************************
 * PluginsWidget class
 *****************************************************************************/

PluginsWidget::PluginsWidget(QWidget *parent, const char *name)
: SettingsWidget(0, parent, name)
{
    QVBoxLayout *layout = new QVBoxLayout(this, KDialogBase::spacingHint(), -1);

    QGroupBox *gb1 = new QGroupBox(1, Horizontal, "Loaded plugins", this);
    QGroupBox *gb2 = new QGroupBox(1, Horizontal, "Directories to scan", this);
    layout->addWidget(gb1);
    layout->addWidget(gb2);

    // Loaded
    loaded = new QListBox(gb1);
    loaded->setSelectionMode(QListBox::NoSelection);

    // Directories
    directories = new QListBox(gb2);
    directories->setSelectionMode(QListBox::Single);
    QHBox *hbox = new QHBox(gb2);
    b_add    = new QPushButton("Add", hbox);
    b_delete = new QPushButton("Delete", hbox);
    b_up     = new QPushButton(hbox);
    b_down   = new QPushButton(hbox);
    b_up->setPixmap(BarIcon("up"));
    b_down->setPixmap(BarIcon("down"));

    connect(directories, SIGNAL(selectionChanged()),
            SLOT(slotSelectionChanged()));
    connect(b_add, SIGNAL(clicked()), SLOT(slotAdd()));
    connect(b_delete, SIGNAL(clicked()), SLOT(slotDelete()));
    connect(b_up, SIGNAL(clicked()), SLOT(slotUp()));
    connect(b_down, SIGNAL(clicked()), SLOT(slotDown()));

    updateWidgets();
    updateButtons();
}


PluginsWidget::~PluginsWidget()
{
}


void PluginsWidget::setValues()
{
    // Add new directories
    bool                asked     = false;
    bool                doloading = false;
    ApplicationPlugins *plugins   = Application::application()->plugins();

    for (unsigned int n = 0; n < directories->count(); n++)
    {
        QString dirname = directories->text(n);

        if (std::find(plugins->directories().begin(),
                      plugins->directories().end(),
                      dirname.latin1()) == plugins->directories().end())
        {
            if (!asked)
            {
                doloading = KMessageBox::questionYesNo
                    (this,
                     "Do you want to load the plugins in the "
                     "new directories now?") == KMessageBox::Yes;
            }
            plugins->addDirectory(dirname.latin1(), doloading);
        }
    }

    // Remove old ones
    bool removed_any = false;
    do
    {
        removed_any = false;
        ApplicationPlugins::plugin_dir_list::const_iterator i =
            plugins->directories().begin();
        while (i != plugins->directories().end() && !removed_any)
        {
            bool match = false;
            for (unsigned int n = 0;
                 n < directories->count() && !removed_any;
                 n++)
            {
                if (*i == directories->text(n).latin1())
                {
                    match = true;
                }
            }
            if (!match)
            {
                plugins->removeDirectory(*i);
                removed_any = true;
            }
            ++i;
        }
    }
    while (removed_any);
}


void PluginsWidget::updateWidgets()
{
    ApplicationPlugins *plugins = Application::application()->plugins();

    // Loaded
    {
        loaded->clear();
        ApplicationPlugins::plugin_loaded_list::const_iterator i
            = plugins->loaded().begin();
        while (i != plugins->loaded().end())
        {
            loaded->insertItem(i->name.c_str());
            ++i;
        }
    }

    // Directories
    {
        directories->clear();
        ApplicationPlugins::plugin_dir_list::const_iterator i
            = plugins->directories().begin();
        while (i != plugins->directories().end())
        {
            directories->insertItem(i->c_str());
            ++i;
        }
        selection = -1;
    }

}


void PluginsWidget::updateButtons()
{
    bool enabled = (selection != -1);
    b_delete->setEnabled(enabled);
    b_up->setEnabled(enabled);
    b_down->setEnabled(enabled);
}


void PluginsWidget::slotSelectionChanged()
{
    selection = -1;
    for (unsigned int n = 0; n < directories->count(); n++)
    {
        if (directories->isSelected(n)) selection = n;
    }
    updateButtons();
}


void PluginsWidget::slotAdd()
{
    QString dir
        = KFileDialog::getExistingDirectory(QString::null, this,
                                            "Select new plugin directory");
    if (dir != QString::null)
    {
        if (!directories->findItem(dir))
        {
            directories->insertItem(dir);
        }
    }
}


void PluginsWidget::slotDelete()
{
    directories->removeItem(selection);
    selection = -1;
    updateButtons();
}


void PluginsWidget::slotUp()
{
    if (selection != 0)
    {
        QListBoxItem *item = directories->item(selection);
        directories->setSelected(selection, false);
        directories->takeItem(item);
        directories->insertItem(item, selection-1);
        directories->setSelected(item, true);
    }
}


void PluginsWidget::slotDown()
{
    if (selection != directories->count()-1)
    {
        QListBoxItem *item = directories->item(selection);
        directories->setSelected(selection, false);
        directories->takeItem(item);
        directories->insertItem(item, selection+1);
        directories->setSelected(item, true);
    }
}

