/*
 * @(#)SettingsWidgets. 1.00 1 June 2000
 *
 * Copyright (c) Pete Goodliffe 2000 (pete@cthree.org)
 *
 * This file is part of anthem - the TSE3 sequencer.
 *
 * This program is modifiable/redistributable under the terms of the GNU
 * General Public License.
 *
 * You should have recieved a copy of the GNU General Public License along
 * with this program; see the file COPYING. If not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 0211-1307, USA.
 */

#ifndef ANTHEM_SETTINGSWIDGET_H
#define ANTHEM_SETTINGSWIDGET_H

#include <qframe.h>

#include "tse3/Notifier.h"

/**
 * Base class for SettingsImpl template classes.
 *
 * The methods in this class simply allow you to call a template class
 * (@ref SettingsImpl) without using templates.
 */
class AbstractSettingsImpl
{
    public:
        virtual ~AbstractSettingsImpl() = 0;
        virtual void preSetValues()     = 0;
        virtual void postSetValues()    = 0;
};

/**
 * Preforms the generic stuff for a SetttingsWidget that can't yet
 * be put into the SettingsWidget class because moc can't handle Q_OBJECT
 * inside a template class.
 *
 * The SettingsImpl will perform these tasks for you:
 *
 *   <li>On deleted being recieved, the SettingsWindow will delete itself
 *   <li>Implements an accept that dettaches from the Notifier before
 *       settings it's values.
 */
template <class interface_type>
class SettingsImpl : public AbstractSettingsImpl,
                     public TSE3::Listener<interface_type>
{
    public:

        typedef typename interface_type::notifier_type notifier_type;

        /**
         * Creates the SettingsImpl.
         *
         * If notifier is non-0 then it is attached to and the
         * SettingsWidget will track deletions.
         *
         * @see updateWidgets
         */
        SettingsImpl(notifier_type *n)
        : notifier(n)
        {
            if (notifier)
            {
                attachTo(notifier);
            }
        }

        /**
         * Copes with the notifier_type being deleted.
         *
         * Will be reintroduced when moc handles template classes
         */
        /*
        void Notifier_Deleted(notifier_type *src)
        {
            delete this;
        }
        */

    public:

        /**
         * This is called when the new settings have been 'OK'ed. It does the
         * following:
         *   <li>Detaches from the notifier (prevents event cascades, where
         *       out own modifications send us update events)
         *   <li>Sets "notifier = 0;"
         *   <li>Calls @ref setValues
         *   <li>Reattaches this widget again, setting notifier
         *
         * @sect Important note
         *
         * You may not always want to call this method: if you are providing
         * support for the @ref TSE3::Cmd classes this method may not
         * always be helpful, for instance the @ref TrackWindow needs to
         * create one @ref TSE3::Cmd::Command for its setting despite
         * comprising of several SettingsWidget objects. For this reason
         * the @ref TrackWindow does not use each SettingsWidget::accept
         * method, but performs the 'accepting' itself.
         */
        void preSetValues()
        {
            if (notifier)
            {
                detachFrom(notifier);
            }
        }

        void postSetValues()
        {
            if (notifier)
            {
                attachTo(notifier);
            }
        }

    private:

        notifier_type *notifier;
};

/**
 * This is a base class for widgets that alter the state of TSE3 objects.
 * It is essentially a blank frame that subclasses will populate, but it
 * also provides a generic interface for playing with settings.
 *
 * It's not as easy to implement as we'd like, since moc doesn't (yet)
 * support Q_OBJECT in template classes. Therefore the simple methods that
 * would be in this class have been split out into the SettingsWidgetTemplate
 * class.
 *
 * @short   Base class for TSE3 settings widgets
 * @author  Pete Goodliffe
 * @version 1.0
 */
class SettingsWidget : public QFrame
{
        Q_OBJECT

    public:

        /**
         * Creates the SettingsWidget. The @ref AbstractSettingsImpl is
         * now owned by the SettingsWidget, and will be deleted with it.
         *
         * @see updateWidgets
         */
        SettingsWidget(AbstractSettingsImpl *impl,
                       QWidget *parent, const char *name = 0);

        virtual ~SettingsWidget();

        /**
         * You must override this method to take the values in the
         * widgets and set them in the TSE3 object. The object is
         * guaranteed to still exist, and this object is no longer
         * attached to it, so no cascade of events will occur.
         */
        virtual void setValues() = 0;

        /**
         * This will cause all the widgets in the SettingsWindow to be updated.
         *
         * You should write a constructor that creates all widgets in the
         * QFrame, then calls this method to set their values.
         *
         * If any state change is detected by the widget, it will
         * call this method to cause updates.
         */
        virtual void updateWidgets() = 0;

    public slots:

        /**
         * @reimplemented
         */
        virtual void accept();

    private:

        AbstractSettingsImpl *impl;
};

#endif
