/*
 * @(#)SongWindow.h 1.00 22 May 2000
 *
 * Copyright (c) Pete Goodliffe 2000 (pete@cthree.org)
 *
 * This file is part of anthem - the TSE3 sequencer.
 *
 * This program is modifiable/redistributable under the terms of the GNU
 * General Public License.
 *
 * You should have recieved a copy of the GNU General Public License along
 * with this program; see the file COPYING. If not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 0211-1307, USA.
 */

#ifndef ANTHEM_SONGWINDOW_H
#define ANTHEM_SONGWINDOW_H

#include <kmainwindow.h>
#include <kurl.h>

#include "misc/Interfaces.h"

#include "tse3/util/Snap.h"
#include "tse3/Notifier.h"
#include "tse3/app/Modified.h"
#include "tse3/listen/Song.h"
#include "tse3/listen/Transport.h"
#include "tse3/listen/cmd/CommandHistory.h"
#include "tse3/listen/app/Record.h"
#include "tse3/listen/app/TrackSelection.h"
#include "tse3/listen/app/PartSelection.h"

/******************************************************************************
 * TSE3 forward declarations
 *****************************************************************************/
namespace TSE3
{
    class Song;
    class Part;
    class Track;
}

/******************************************************************************
 * KDE/Qt forward declarations
 *****************************************************************************/
class QSplitter;
class KToolBar;
class QPopupMenu;
class KRecentFilesAction;
class KURL;
class KRadioAction;
class KToggleAction;

/******************************************************************************
 * Anthem forward declataions
 *****************************************************************************/
class TrackList;
class PartView;
class TimeLine;
class SongInfoWindow;
class PhraseListWindow;
class ClockWindow;
class AboutWindow;
class NewPhraseWindow;
class PanicWindow;
class SnapWidget;
class PartInsertActionWidget;
class GotoWindow;
class ZoomWindow;
class MidiSchedulerTimeSlider;
class ClockDisplayWidget;
class FlagComboBox;

/**
 * The SongWindow is the main Anthem widget. It displays the TSE3 Song to the
 * user, provides a transport toolbar, and leads to the other editing windows.
 *
 * It contains a QSplitter which divides the window into two: on the left the
 * Track view showing the track parameters, on the right the Part view showing
 * each Part in each Track.
 *
 * The window allows access to the various Part and Phrase editors and to the
 * PhraseList window.
 *
 * The SongWindow is a collaboration of several different widgets, the
 * @ref TrackList, a @ref TimeLine, and the @ref PartView. A @ref ZoomWidget
 * is used to control zooming. The connections between these closely
 * connected windows/widgets is actually quite disgusting ;-)
 *
 * @short   Main Anthem Song window.
 * @author  Pete Goodliffe
 * @version 1.0
 */
class SongWindow : public KMainWindow,
                   public TSE3::Listener<TSE3::App::RecordListener>,
                   public TSE3::Listener<TSE3::TransportListener>,
                   public TSE3::Listener<TSE3::Cmd::CommandHistoryListener>,
                   public TSE3::Listener<TSE3::App::ModifiedListener>,
                   public TSE3::Listener<TSE3::App::TrackSelectionListener>,
                   public TSE3::Listener<TSE3::App::PartSelectionListener>,
                   public TSE3::Listener<TSE3::SongListener>,
                   public Zoomable
{
        Q_OBJECT

    public:

        /**
         * Creates a song window for the given song. Ownership of the Song
         * passes to the SongWindow object.
         *
         * If the song parameter is not specified, creates a new Song to work
         * with.
         *
         * The song should already be managed by the @ref Application object.
         * if it is not, then all hell can break loose. If you pass zero for
         * @p song then the newly created @ref TSE3::Song is inserted into the
         * @ref Application.
         *
         * You can pass an optional filename for this Song.
         *
         * @param song     @ref TSE3::Song to work on
         * @param filename Filename of this @ref TSE3::Song
         */
        SongWindow(TSE3::Song *song = 0,
                   const QString &filename = "", const KURL &url = "");

        /**
         * Dtor - the Song is deleted with the window.
         */
        ~SongWindow();

        /**
         * Returns the @ref TSE3::Song used by this window.
         */
        TSE3::Song *theSong() const { return song; }

        /**
         * Changes which @ref TSE3::Song this window views.
         *
         * In most situations you will want to open a new SongWindow for this
         * song rather than change the view.
         *
         * It is usedful, for example, when you load a new @ref TSE3::Song and
         * have an unused SongWindow.
         *
         * @param song     @ref TSE3::Song to work on
         * @param filename Filename of this @ref TSE3::Song
         * @param url      URL of file
         */
        void changeSong(TSE3::Song *song,
                        const QString &filename = "", const KURL &url = "");

        /**
         * Calls @ref createGUI() and them patches the undo and redo actions
         * so that they have popup menus. It's basically an evil hack.
         */
        void createGUIAndPatchActions(const QString &xmlfile = QString::null,
                                      bool _conserveMemory = TRUE);

        struct ViewOptions
        {
            ViewOptions() : rmbSetsClock(false), lmbSetsClock(true) {}
            bool rmbSetsClock;
            bool lmbSetsClock;
        };

        /**
         * Returns the current ViewOptions
         *
         * @see setViewOptions
         */
        const ViewOptions &viewOptions() const { return _viewOptions; }

        /**
         * Sets the view options
         */
        void setViewOptions(const ViewOptions &vo)
        {
            _viewOptions = vo;
        }

        /**
         * @reimplemented
         */
        virtual QSize sizeHint() const;

        /**
         * Returns the child @ref TrackList widget related to this SongWindow.
         */
        TrackList *childTrackList() const { return trackList; }

        /**
         * Returns the child @ref PartView widget related to this SongWindow.
         */
        PartView *childPartView() const { return partView; }

        /**
         * Returns the child @ref TimeLine widget related to this SongWindow.
         */
        TimeLine *childTimeLine() const { return timeLine; }

        /**
         * @reimplemented
         */
        virtual void Record_RecordingEnded(TSE3::App::Record *src,
                                           TSE3::Song        *song,
                                           TSE3::Track       *track);

        /**
         * @reimplemented
         */
        virtual void Transport_Status(TSE3::Transport *src, int status);

        /**
         * @reimplemented
         */
        virtual void CommandHistory_Undo(TSE3::Cmd::CommandHistory *src);

        /**
         * @reimplemented
         */
        virtual void CommandHistory_Redo(TSE3::Cmd::CommandHistory *src);

        /**
         * @reimplemented
         */
        virtual void Modified_Changed(TSE3::App::Modified *);

        /**
         * @reimplemented
         */
        virtual void TrackSelection_Selected
            (TSE3::App::TrackSelection *, TSE3::Track *, bool selected);

        /**
         * @reimplemented
         */
        virtual void PartSelection_Selected
            (TSE3::App::PartSelection *, TSE3::Part *, bool selected);

        /**
         * @reimplemented
         */
        virtual void Song_FromAltered(TSE3::Song *song, TSE3::Clock);

        /**
         * @reimplemented
         */
        virtual void Song_ToAltered(TSE3::Song *song, TSE3::Clock);

        /**
         * @reimplemented
         *
         * Sorts out the splitter size
         */
        virtual void show();

    public slots:

        /*
         * Overridden slots
         */

        void dropEvent(QDropEvent *);

    protected slots:

        /**
         * The @ref QHeader object in the @ref TrackList is only resized
         * on it's polish, which occurs after the SongWindow polish. So I
         * have to connect a signal to track it's size - I do this to make
         * the timeline the same height. The size of the @ref QHeader can
         * be found via the @ref TrackList's headerHeight() method.
         */
        void slotTrackListHeaderHeightSet();

        /**
         * Opens a zoom popup window.
         */
        void slotOpenZoom();

        /**
         * @reimplemented
         */
        int zoomHorizontal();

        /**
         * @reimplemented
         */
        void setZoomHorizontal(int pc);

        /**
         * @reimplemented
         */
        int zoomVertical();

        /**
         * @reimplemented
         */
        void setZoomVertical(int pc);

        /**
         * The slot is connected to the @ref PartView's zoom signals.
         */
        void zoomInHorizontal();

        /**
         * The slot is connected to the @ref PartView's zoom signals.
         */
        void zoomOutHorizontal();

        /**
         * The slot is connected to the @ref PartView's zoom signals.
         */
        void zoomInVertical();

        /**
         * The slot is connected to the @ref PartView's zoom signals.
         */
        void zoomOutVertical();

        /**
         * Called to open the @ref AboutWindow. The @ref AboutWindow is
         * only constructed when it's needed, and then cached for
         * later use.
         */
        void showAboutApplication();

        /**
         * This slot is connected to the @ref NewPhraseWindow and handles
         * the user's selection of a new @ref Phrase after recording.
         */
        void slotNewPhraseDone(bool success, QString title,
                               bool replacePhrase, bool insertPart);

        /**
         * The @ref PartView's pointer mode has changed - now update the
         * toolbar.
         */
        void slotPointerModeChanged();

        /**
         * About to show the undo menu - sets the items in it
         */
        void slotAboutToShowUndo();

        /**
         * About to show the redo menu - sets the items in it
         */
        void slotAboutToShowRedo();

        /**
         * The undo popup menu was activated
         */
        void slotUndoActivated(int id);

        /**
         * The redo popup menu was activated
         */
        void slotRedoActivated(int id);

        /**
         * Connected to the @ref SnapWidget.
         */
        void slotSetSnap(int snap);

        /**
         * Connected to the QClipboard.
         */
        void slotClipboardDataChanged();

        /**
         * Opens a @ref NewTrackWindow.
         */
        void slotInsertTrack();

        /**
         * Add @ref TSE3::Track operation, called from the RMB menu.
         */
        void slotInsertTrackRMB();

        /**
         * Properties operation from RMB.
         */
        void slotRMBProperties();

        /**
         * Attached to the PartView.
         *
         * Specifies the track index the menu was opened at, and also
         * any @ref TSE::Part the menu was opened over (zero for none).
         */
        void slotRmbMenuOpened(size_t track, TSE3::Part *part);

        /**
         * Attached to the TrackList.
         *
         * Specifies the track index the menu was opened at.
         */
        void slotRmbMenuOpened(size_t track);

        /**
         * Sets clock if settings permit.
         */
        void slotSetClock(TSE3::Clock, Qt::ButtonState);

        /**
         * The user has made a selection from the flagCombo.
         */
        void slotFlagComboActicated(int);

        /*
         * Menu/Toolbar slots (self explanatory)
         */
        void slotOpenNew();
        void slotOpen();
        void slotOpenRecent(const KURL &);
        void slotSave();
        void slotSaveAs();
        void slotMerge();
        void slotAppend();
        void slotRevert();
        void slotUndo();
        void slotRedo();
        void slotClearHistory();
        void slotCut();
        void slotCopy();
        void slotPaste();
        void slotSongInfoWindow();
        void slotPointerSelect();
        void slotPointerSnip();
        void slotPointerGlue();
        void slotPointerInsert();
        void slotTransportStop();
        void slotTransportPlay();
        void slotTransportRecord();
        void slotTransportRew();
        void slotTransportFF();
        void slotTransportSynchro();
        void slotTransportPunchIn();
        void slotFlagPrev();
        void slotFlagNext();
        void slotFlagAdd();
        void slotFlagList();
        void slotMarkerFrom();
        void slotMarkerTo();
        void slotMarkerBoth();
        void slotMarkerRepeat();
        void slotSortTracks();       // Opens a @ref TrackSortWindow.
        void slotShowTip();
        void slotKeyBindings();
        void slotConfigureToolbars();
        void slotSettingsWindow();
        void slotPhraseListWindow(); // Opens @ref PhraseListWindow
        void slotClockWindow();      // Opens @ref ClockWindow
        void slotKeyboardWindow();   // Opens @ref KeyboardWindow
        void slotPanicWindow();      // Opens @ref PanicWindow
        void slotGotoWindow();       // Opens @ref GotoWindow
        void slotVuWindow();         // Opens @ref VuWindow
        void slotPartExport();
        void slotPartPosition();
        void slotPartFromStart();
        void slotPartFromEnd();
        void slotPartToStart();
        void slotPartToEnd();
        void slotPartQuantise();
        void slotPartRemix();
        void slotPartPartition();
        void slotPartExtract();
        void slotPartSubdivide();
        void slotPartStretch();
        void slotGoStart();
        void slotGoEnd();
        void slotGoFrom();
        void slotGoTo();
        void slotSelectAllInTrack();
        void slotSelectAllInSelectedTracks();
        void slotToolbarToggled(bool);

        /**
         * The opened @ref SongInfoWindow has been closed.
         */
        void slotSongInfoWindowClosed();

    protected:

        /**
         * @reimplemented
         */
        void saveProperties(KConfig*);

        /**
         * @reimplemented
         */
        void readProperties(KConfig*);

        /**
         * @reimplemented
         */
        virtual bool queryClose();

        /**
         * @reimplemented
         */
        virtual bool queryExit();

    private:

        /**
         * Set the horizontal zoom in the SongWindow.
         */
        void setHorizontalZoom(int newValue);

        /**
         * Set the vertical zoom in the SongWindow.
         */
        void setVerticalZoom(int pixels);

        /**
         * Does an open for the specified file/url
         */
        bool doOpen(const QString &filename, const KURL &url);

        /**
         * Does a 'save as' operation and returns whether or not the
         * user followed it though, or cancelled it.
         */
        bool doSaveAs();

        /**
         * Does all that good file saving stuff.
         */
        void doSave(const QString &filename, const KURL &url,
                    int format = -1);

        /**
         * Sets the caption in a "filename (modified)" stylee.
         */
        void doSetCaption();

        /**
         * Puts up a prompt window saying that the song has been modified, and
         * then either does a save or accepts cancellation.
         *
         * Returns true if the user wants to trash this @ref TSE3::Song (it
         * may have been saved) or false if the operation has been cancelled.
         *
         * If the Song hasn't been modified, then this method returns true
         * immediately.
         */
        bool messageBoxSave();

        /**
         * Changes the state of the Cut and Copy actions according to the
         * PartSelection and TrackSelection states.
         */
        void updateCutCopy();

        // IDs for toolbar icons
        enum Toolbar_Buttons   { Toolbar_New, Toolbar_Open, Toolbar_Save,
                                 Toolbar_Cut, Toolbar_Copy, Toolbar_Paste,
                                 Toolbar_Undo, Toolbar_Redo, Toolbar_Help,
                                 Toolbar_Select, Toolbar_Snip,
                                 Toolbar_Glue, Toolbar_Insert,
                                 Toolbar_PhraseList,
                                 Transport_Stop, Transport_Play,
                                 Transport_Record, Transport_Rew, Transport_FF,
                                 Transport_Repeat, Transport_Time,
                                 Transport_Tempo, Transport_TimeSig,
                                 Transport_Synchro, Transport_PunchIn,
                                 Transport_Panic,
                                 Marker_From, Marker_To, Marker_Both,
                                 Marker_FromDisplay, Marker_ToDisplay,
                                 Marker_Repeat,
                                 Tempo_Half, Tempo_Unity, Tempo_Double,
                                 Tempo_Other,
                                 Flag_Combo,
                                 Flag_Prev, Flag_Next, Flag_Add, Flag_List };

        // Window status
        int         _zoomVertical;
        int         minZoomVertical, maxZoomVertical;
        QString     filename;
        KURL        fileurl;
        int         midiFileFormat;

        // RMB menu
        QPopupMenu *rmbMenu;
        bool        rmbMenuFromTrack; // true TrackList, false PartView
        size_t      rmbMenuTrack;
        TSE3::Part *rmbMenuPart;
        int         rmbPropertiesID;

        enum SelectionStatus
        {
            NothingSelected,
            TrackSelected,
            PartSelected
        };

        SelectionStatus selectionStatus;

        ViewOptions     _viewOptions;

        // TSE3 objects
        TSE3::Song                *song;
        TSE3::Cmd::CommandHistory *history;
        TSE3::Util::Snap           snap;
        TSE3::App::Modified        modified;
        TSE3::App::PartSelection  *partSelection;
        TSE3::App::TrackSelection *trackSelection;

        // UI widgets
        QSplitter               *splitter;
        TrackList               *trackList;
        PartView                *partView;
        TimeLine                *timeLine;

        SnapWidget              *snapWidget;
        PartInsertActionWidget  *piaWidget;
        QPopupMenu              *edit;
        KAction                 *undoAction;
        KAction                 *redoAction;
        KAction                 *clearHistoryAction;
        KAction                 *cutAction;
        KAction                 *copyAction;
        KAction                 *pasteAction;
        QPopupMenu              *undoMenu;
        QPopupMenu              *redoMenu;
        KRadioAction            *selectAction;
        KRadioAction            *snipAction;
        KRadioAction            *glueAction;
        KRadioAction            *insertAction;
        KToggleAction           *stopAction;
        KToggleAction           *playAction;
        KToggleAction           *recordAction;
        KAction                 *rewAction;
        KAction                 *ffAction;
        KRadioAction            *synchroAction;
        KRadioAction            *punchinAction;
        KRecentFilesAction      *recentFilesAction;
        KAction                 *propertiesAction;
        ClockDisplayWidget      *fromDisplay;
        ClockDisplayWidget      *toDisplay;
        MidiSchedulerTimeSlider *msTimeSlider;
        FlagComboBox            *flagCombo;
        QList<KAction>           toolbarActionList;
        KAction                 *bothMarkersAction;
        KAction                 *fromMarkerStartAction;
        KAction                 *fromMarkerEndAction;
        KAction                 *toMarkerStartAction;
        KAction                 *toMarkerEndAction;

        // Other windows
        SongInfoWindow    *songInfoWindow;
        PhraseListWindow  *phraseListWindow;
        AboutWindow       *aboutWindow;
        NewPhraseWindow   *newPhraseWindow;
        ClockWindow       *clockWindow;
        GotoWindow        *gotoWindow;
        ZoomWindow        *zoomWindow;
};

#endif
