/*
 * @(#)SongWindow_TrackList.cpp 1.00 22 May 2000
 *
 * Copyright (c) Pete Goodliffe 2000 (pete@cthree.org)
 *
 * This file is part of anthem - the TSE3 sequencer.
 *
 * This program is modifiable/redistributable under the terms of the GNU
 * General Public License.
 *
 * You should have recieved a copy of the GNU General Public License along
 * with this program; see the file COPYING. If not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 0211-1307, USA.
 */

#include "songwindow/TrackList.h"

#include "Application.h"
#include "gadgets/ColourPixmap.h"
#include "tse3/kdeui/PortWidget.h"
#include "gadgets/FilteredKListView.h"
#include "dialogues/Track.h"
#include "misc/Clipboard.h"

#include <kapp.h>
#include <klocale.h>
#include <qlistbox.h>
#include <qheader.h>
#include <qlayout.h>
#include <kiconloader.h>
#include <qpopupmenu.h>

#include "tse3/Song.h"
#include "tse3/Track.h"
#include "tse3/MidiFilter.h"
#include "tse3/MidiParams.h"
#include "tse3/DisplayParams.h"
#include "tse3/cmd/Song.h"
#include "tse3/cmd/CommandHistory.h"
#include "tse3/cmd/CommandGroup.h"
#include "tse3/app/TrackSelection.h"

enum TrackListColumns
{
    Column_Colour,
    Column_Title,
    Column_Status,
    Column_Solo,
    Column_Destination,
    Column_Params
};


/******************************************************************************
 * TrackListViewItem class
 *****************************************************************************/

/**
 * The TrackListViewItem is a QListViewItem widget that specifically
 * knows how to display @ref Track information and will keep itself
 * up to date with the @ref Track's details.
 */
class TrackListViewItem : public QListViewItem,
                          public TSE3::Listener<TSE3::TrackListener>,
                          public TSE3::Listener<TSE3::SongListener>,
                          public TSE3::Listener<TSE3::MidiFilterListener>,
                          public TSE3::Listener<TSE3::MidiParamsListener>
{
    public:

        /**
         * Construct a TrackListViewItem for the given @ref Track. You should
         * specify the index of this @ref Track which will be used as a
         * sorting key.
         */
        TrackListViewItem(TSE3::Track *track, QListView *parent,
                          QListViewItem *after);

        virtual ~TrackListViewItem();

        void setItemHeight(int h)
        {
            setHeight(h);
        }

        TSE3::Track *track() { return _track; }

        /**
         * @reimplemented
         */
        virtual void Track_TitleAltered(TSE3::Track *src);

        /**
         * @reimplemented
         */
        virtual void Track_Deleted(TSE3::Track *src);

        /**
         * @reimplemented
         */
        virtual void Track_DisplayParamsAltered(TSE3::Track *src);

        /**
         * @reimplemented
         */
        virtual void Song_SoloTrackAltered(TSE3::Song *src, int soloTrack);

        /**
         * @reimplemented
         */
        virtual void MidiFilter_Altered(TSE3::MidiFilter *src, int what);

        /**
         * @reimplemented
         */
        virtual void MidiParams_Altered(TSE3::MidiParams *src, int what);

        /**
         * @reimplemented
         */

    private:

        void updateColour();
        void updatePixmap();

        TSE3::Track *_track;
        TSE3::Song  *_song;
};


TrackListViewItem::TrackListViewItem(TSE3::Track *track, QListView *parent,
                                     QListViewItem *after)
: QListViewItem(parent,
                after,
                "", // colour
                track->title().c_str(),
                "", // status
                "", // solo
                PortWidget::toText(track->filter()->channel(),
                                   track->filter()->port()),
                TrackWindow::paramString(track)),
  _track(track), _song(track->parent())
{
    updateColour();
    updatePixmap();
    TSE3::Listener<TSE3::TrackListener>::attachTo(_track);
    TSE3::Listener<TSE3::MidiFilterListener>::attachTo(_track->filter());
    TSE3::Listener<TSE3::MidiParamsListener>::attachTo(_track->params());
    TSE3::Listener<TSE3::SongListener>::attachTo(_song);
}


TrackListViewItem::~TrackListViewItem()
{
}


void TrackListViewItem::Track_TitleAltered(TSE3::Track *)
{
    setText(Column_Title, _track->title().c_str());
    repaint();
}


void TrackListViewItem::Song_SoloTrackAltered(TSE3::Song *src, int soloTrack)
{
    updatePixmap();
}


void TrackListViewItem::Track_Deleted(TSE3::Track *src)
{
    delete this;
}


void TrackListViewItem::Track_DisplayParamsAltered(TSE3::Track *src)
{
    updateColour();
}


void TrackListViewItem::MidiFilter_Altered(TSE3::MidiFilter *, int)
{
    updatePixmap();
    setText(Column_Destination,
            PortWidget::toText(_track->filter()->channel(),
                               _track->filter()->port()));
}


void TrackListViewItem::MidiParams_Altered(TSE3::MidiParams *, int)
{
    setText(Column_Params, TrackWindow::paramString(_track));
}


void TrackListViewItem::updateColour()
{
    setPixmap(Column_Colour, ColourPixmap::newPixmap(*_track->displayParams()));
}


void TrackListViewItem::updatePixmap()
{
    if (_track->filter()->status())
    {
        setPixmap(Column_Status, QPixmap(UserIcon("speaker")));
    }
    else
    {
        setPixmap(Column_Status, QPixmap(UserIcon("speakeroff")));
    }
    if (_song->soloTrack() == int(_song->index(_track)))
    {
        setPixmap(Column_Solo, QPixmap(UserIcon("solo")));
    }
    else
    {
        setPixmap(Column_Solo, QPixmap(UserIcon("solooff")));
    }
}


/******************************************************************************
 * TrackList class
 *****************************************************************************/

namespace
{
    const char *COLOUR      = "";             // It's smaller with no text
    const char *TITLE       = "Title";
    const char *STATUS      = "";             // Ditto
    const char *SOLO        = "";             // Ditto
    const char *DESTINATION = "Destination";
    const char *PARAMETERS  = "Parameters";
}


TrackList::TrackList(TSE3::Song *song, QWidget *parent,
                     TSE3::App::TrackSelection *ts)
: QWidget(parent, "TrackList"), song(song), trackSelection(ts), rmbMenu(0)
{
    TSE3::Listener<TSE3::App::TrackSelectionListener>::attachTo(trackSelection);
    TSE3::Listener<TSE3::SongListener>::attachTo(song);

    QVBoxLayout *layout = new QVBoxLayout(this);
    listView = new FilteredKListView(this, "TrackListInternal");
    layout->addWidget(listView);
    listView->addColumn(COLOUR);
    listView->addColumn(TITLE);
    listView->addColumn(STATUS);
    listView->addColumn(SOLO);
    listView->addColumn(DESTINATION);
    listView->addColumn(PARAMETERS);
    listView->setVScrollBarMode(QScrollView::AlwaysOff);
    listView->setHScrollBarMode(QScrollView::AlwaysOn);
    listView->setAllColumnsShowFocus(true);
    listView->setSorting(-1);

    listView->setColumnWidth(Column_Title, kapp->fontMetrics().maxWidth()*10);

    connect(listView, SIGNAL(doubleClicked(QListViewItem*)),
            SLOT(slotTrackDoubleClicked(QListViewItem*)));
    connect(listView, SIGNAL(clicked(QListViewItem*,const QPoint&,int)),
            SLOT(slotClicked(QListViewItem*, const QPoint&,int)));
    connect(listView, SIGNAL(contentsMoving(int,int)),
            SIGNAL(contentsMoving(int,int)));
    connect(listView, SIGNAL(wantToSelect(QListViewItem*,bool)),
            SLOT(slotWantsToSelect(QListViewItem*,bool)));
    connect(listView, SIGNAL(wantToClear(QListViewItem*)),
            SLOT(slotWantsToClear(QListViewItem*)));
    connect(listView,
            SIGNAL(rightButtonPressed(QListViewItem*,const QPoint&,int)),
            SLOT(slotRMB(QListViewItem*,const QPoint&,int)));

    updateTracks();
}


void TrackList::setRMBMenu(QPopupMenu *menu)
{
    rmbMenu = menu;
}


void TrackList::updateTracks()
{
    QListViewItem *last = 0;
    for (size_t track = 0; track < song->size(); track++)
    {
        last = new TrackListViewItem((*song)[track], listView, last);
    }
}


void TrackList::changeSong(TSE3::Song *s)
{
    // 1. Out with the old
    TSE3::Listener<TSE3::SongListener>::detachFrom(song);
    listView->clear();

    // 2. In with the new
    song = s;
    TSE3::Listener<TSE3::SongListener>::attachTo(song);
    updateTracks();
}


void TrackList::polish()
{
    // Do the 'polishing'
    QWidget::polish();

    // Now the header's height has been set tell the SongWindow
    listView->header()->polish();
    emit headerHeightSet();
}


int TrackList::headerHeight() const
{
    QSize size = listView->header()->sizeHint();
    return size.height();
}


int TrackList::itemHeight() const
{
    if (listView->firstChild())
        return listView->firstChild()->height();
    std::cerr << "No items in TrackList in TrackList::itemHeight\n";
    return 20;
}


void TrackList::resizeEvent(QResizeEvent *e)
{
    listView->resize(width(), height());
}


void TrackList::slotTrackDoubleClicked(QListViewItem *trackItem)
{
    size_t index  = listView->indexOf(trackItem);
    int    column = lastClickCol;

    switch (column)
    {
        case Column_Colour:
            column = TrackWindow::DisplayTab;
            break;
        case Column_Title:
        case Column_Status:
        case Column_Solo:
            column = TrackWindow::TrackTab;
            break;
        case Column_Destination:
            column = TrackWindow::FilterTab;
            break;
        case Column_Params:
            column = TrackWindow::ParamsTab;
            break;
    }

    new TrackWindow(this, (*song)[index],
                    Application::application()->history(song),
                    TrackWindow::Tabs(column));
}


void TrackList::move(int x, int y)
{
    bool b = this->signalsBlocked();
    blockSignals(true);
    listView->setContentsPos(listView->contentsX(), y);
    blockSignals(b);
}


void TrackList::setItemHeight(int height)
{
    QListViewItem *item = listView->firstChild();
    while (item)
    {
        static_cast<TrackListViewItem*>(item)->setItemHeight(height);
        item = item->nextSibling();
    }
}


void TrackList::slotClicked(QListViewItem *item, const QPoint &p, int column)
{
    lastClickCol = column;
    TSE3::Track * const track = static_cast<TrackListViewItem*>(item)->track();

    if (column == Column_Status)
    {
        track->filter()->setStatus(!track->filter()->status());
    }
    if (column == Column_Solo)
    {
        int index = song->index(track);
        song->setSoloTrack(song->soloTrack() == index ? -1 : index);
    }
}


void TrackList::slotWantsToSelect(QListViewItem *item, bool add)
{
    TSE3::Track * const track = static_cast<TrackListViewItem*>(item)->track();
    trackSelection->select(track, add);
}


void TrackList::slotWantsToClear(QListViewItem *item)
{
    TSE3::Track * const track = static_cast<TrackListViewItem*>(item)->track();
    trackSelection->deselect(track);
}


void TrackList::slotRMB(QListViewItem *item, const QPoint &p, int)
{
    if (rmbMenu)
    {
        TSE3::Track * const track
            = static_cast<TrackListViewItem*>(item)->track();
        size_t index = song->index(track);
        emit rmbMenuOpened(index);
        rmbMenu->popup(p);
    }
}


int TrackList::preferredShowWidth() const
{
    int width = 0;
    for (size_t n = 0; n <= Column_Solo; n++)
    {
        width += listView->columnWidth(n);
    }
    return width;
}


void TrackList::slotCut()
{
    toClipboard(trackSelection);

    TSE3::Cmd::CommandGroup *command = new TSE3::Cmd::CommandGroup();

    TSE3::App::TrackSelection::iterator_t i = trackSelection->begin();
    while (i != trackSelection->end())
    {
        command->add(new TSE3::Cmd::Song_RemoveTrack(*i));
        i++;
    }
    command->execute();
    Application::application()->history(song)->add(command);
}


void TrackList::slotCopy()
{
    toClipboard(trackSelection);
}


void TrackList::slotPaste()
{
    std::cout << "TrackList::slotPaste\n";
}


void TrackList::Song_TrackInserted(TSE3::Song *song, TSE3::Track *track)
{
    size_t index = song->index(track);

    QListViewItem *lastItem = 0;
    if (index)
    {
        lastItem = listView->firstChild();
        --index;
        while (index)
        {
            lastItem = lastItem->nextSibling();
            --index;
        }
    }

    new TrackListViewItem(track, listView, lastItem);
}


void TrackList::Song_TrackRemoved(TSE3::Song *song, TSE3::Track *track,
                                  size_t index)
{
    QListViewItem *item = listView->firstChild();
    while (index)
    {
        item = item->nextSibling();
        --index;
    }
    TSE3::Track * const t = static_cast<TrackListViewItem*>(item)->track();

    if (track == t)
    {
        delete item;
    }
    else
    {
        std::cerr << "TrackList inconsitency at line " << __LINE__ << std::endl;
    }
}


void TrackList::TrackSelection_Selected(TSE3::App::TrackSelection *,
                                        TSE3::Track               *track,
                                        bool                       selected)
{
    if (track->parent())
        listView->setSelectedWithoutSignal(listView->itemWithIndex
                                               (track->parent()->index(track)),
                                           selected);
}
