/*
 * @(#)SongWindow_TrackList.h 1.00 22 May 2000
 *
 * Copyright (c) Pete Goodliffe 2000 (pete@cthree.org)
 *
 * This file is part of anthem - the TSE3 sequencer.
 *
 * This program is modifiable/redistributable under the terms of the GNU
 * General Public License.
 *
 * You should have recieved a copy of the GNU General Public License along
 * with this program; see the file COPYING. If not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 0211-1307, USA.
 */

#ifndef ANTHEM_SONGWINDOW_TRACKLIST_H
#define ANTHEM_SONGWINDOW_TRACKLIST_H

#include <qwidget.h>
#include <klistview.h>

#include "tse3/Notifier.h"
#include "tse3/listen/Song.h"
#include "tse3/listen/app/TrackSelection.h"

/******************************************************************************
 * TSE3 forward declarations
 *****************************************************************************/
namespace TSE3
{
    class Song;
    class Track;
    class Part;
}

/******************************************************************************
 * KDE/Qt forward declarations
 *****************************************************************************/
class QListViewItem;
class QPopupMenu;

/******************************************************************************
 * Anthem forward declarations
 *****************************************************************************/
class FilteredKListView;

/**
 * This widget displays the list of @ref Tracks in the Song. It is held within
 * the main @ref SongWindow.
 *
 * There can optionally be a @ref Track selected.
 *
 * @short   List of Tracks.
 * @author  Pete Goodliffe
 * @version 1.0
 */
class TrackList : public QWidget,
                  public TSE3::Listener<TSE3::SongListener>,
                  public TSE3::Listener<TSE3::App::TrackSelectionListener>
{
        Q_OBJECT;

    public:

        TrackList(TSE3::Song *song, QWidget *parent,
                  TSE3::App::TrackSelection *ts);

        /**
         * Changes which @ref TSE3::Song this window views.
         *
         * This can only be called safely by the @ref SongWindow.
         */
        void changeSong(TSE3::Song *song);

        /**
         * Overridden so I can notice when the @ref QHeader's height is set.
         */
        virtual void polish();

        /**
         * Sets the popup menu to show when the right menu button is
         * closed. Currently, the SongWindow passes the 'Edit' menu here.
         *
         * You can pass zero which disables this menu (and is the default
         * value).
         */
        void setRMBMenu(QPopupMenu *menu);

        /**
         * This method sets the height of each list item and the
         * list view header. It should only be called by the @ref SongWindow
         * on reciept of a zoom event.
         */
        void setItemHeight(int height);

        /**
         * Returns the height of the @ref QHeader that is at the top of this
         * TrackList. Qt guarantees this will never change.
         */
        int headerHeight() const;

        /**
         * Returns the current height of each of the items in the list.
         */
        int itemHeight() const;

        /**
         * Returns the width of the first few columns that the TrackList
         * would like to have displayed.
         */
        int preferredShowWidth() const;

        /**
         * @reimplemented
         */
        virtual void Song_TrackInserted(TSE3::Song *, TSE3::Track *);

        /**
         * @reimplemented
         */
        virtual void Song_TrackRemoved(TSE3::Song *, TSE3::Track *, size_t);

        /**
         * @reimplemented
         */
        virtual void TrackSelection_Selected
            (TSE3::App::TrackSelection *, TSE3::Track *, bool selected);

    protected:

        virtual void resizeEvent(QResizeEvent *event);

    public slots:

        /**
         * The @ref PartView has been scrolled, move this view with it.
         */
        void move(int x, int y);

        /**
         * Perform a cut operation
         */
        void slotCut();

        /**
         * Perform a copy operation
         */
        void slotCopy();

        /**
         * Perform a paste operation
         */
        void slotPaste();

    signals:

        /**
         * Contents moving signal is emitted when the list view is
         * scrolled. It is connected to the @ref PartList's move slot.
         */
        void contentsMoving(int x, int y);

        /**
         * Emitted by polish, when the header height is set, so the
         * @ref SongWindow can set the @ref TimeLine's height to match.
         */
        void headerHeightSet();

         /**
          * Emitted when the RMB menu is opened.
          *
          * @param track The index of the track RMB is clicked over
          */
         void rmbMenuOpened(size_t track);

    protected slots:

        /*
         * These slots receive signals from the list view.
         */
        void slotTrackDoubleClicked(QListViewItem *trackItem);
        void slotClicked(QListViewItem *item, const QPoint &p, int column);
        void slotWantsToSelect(QListViewItem *item, bool add);
        void slotWantsToClear(QListViewItem *item);
        void slotRMB(QListViewItem *item, const QPoint &p, int column);

    private:

        /**
         * Performs an initial setup based on the Tracks in the Song.
         */
        void updateTracks();

        TSE3::Song                *song;
        TSE3::App::TrackSelection *trackSelection;
        FilteredKListView         *listView;
        int                        lastClickCol;
        QPopupMenu                *rmbMenu;
};

#endif
