/*
 * @(#)ClockWidget.h 1.00 1 June 2000
 *
 * Copyright (c) Pete Goodliffe 2000 (pete@cthree.org)
 *
 * This file is part of anthem - the TSE3 sequencer.
 *
 * This program is modifiable/redistributable under the terms of the GNU
 * General Public License.
 *
 * You should have recieved a copy of the GNU General Public License along
 * with this program; see the file COPYING. If not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 0211-1307, USA.
 */

#ifndef TSE3_KDEUI_CLOCKWIDGET_H
#define TSE3_KDEUI_CLOCKWIDGET_H

#include <qwidget.h>
#include <qstring.h>

#include "tse3/Midi.h"

class ClockSpinBox;

/**
 * This widget provides a standard time selector - it produces a value
 * in pulses (a @ref TSE3 @ref Clock) but provides the user
 * with the abilty to see beats/pulses which is more intuitive.
 *
 * You can allow or disallow negative values, and can also set a
 * special value text (in a similar manner to a QSpinBox) which will
 * display some text at value zero.
 *
 * @short   Clock value selector
 * @author  Pete Goodliffe
 * @version 1.0
 */
class ClockWidget : public QWidget
{
        Q_OBJECT

    public:

        /**
         * Creates a ClockWidget.
         *
         * You can specify a time to display, or leave it unspecified where
         * it will default to zero.
         *
         * @param parent   Parent widget
         * @param negative Whether to allow negative values
         * @param mc       Initial @ref Clock value
         * @param name     Name of this widget
         */
        ClockWidget(QWidget *parent = 0,
                    bool negative = false, TSE3::Clock mc = 0,
                    const char *name = 0);

        virtual ~ClockWidget();

        /**
         * Returns the @ref Clock value of the widget
         *
         * @see setValue
         */
        TSE3::Clock value();

        /**
         * Sets the value of this widget.
         *
         * @see value
         */
        void setValue(TSE3::Clock mc);

        /**
         * Works similarly to the QSpinBox method of the same name - but
         * this text only appears in the 'beats' spin box.
         *
         * The special value is not the mimimum possible value of the
         * ClockWidget, but always zero.
         *
         * @see specialValueText
         */
        void setSpecialValueText(QString s);

        /**
         * Sets the widget's special value text.
         *
         * @see setSpecialValueText
         */
        QString specialValueText();

        /**
         * Converts a clock value to a text value.
         */
        static QString toText(TSE3::Clock c);

    signals:

        /**
         * This signal is emitted when the value of the widget is changed.
         *
         * It sends the @ref Clock value, although it uses an integer
         * value to make the signal/slot API cleaner.
         */
        virtual void valueChanged(int);

    private slots:

        /**
         * Called when the beats spinbox changes.
         */
        void beatsChanged(int b);

        /**
         * Called when the pulses spinbox changes.
         */
        void pulsesChanged(int p);

    private:

        bool          negative;
        QString       _specialValueText;
        ClockSpinBox *beats;
        ClockSpinBox *pulses;
};

#endif
