/*
 * @(#)NewPhrase.h 1.00 19 June 2000
 *
 * Copyright (c) Pete Goodliffe 2000 (pete@cthree.org)
 *
 * This file is part of anthem - the TSE3 sequencer.
 *
 * This program is modifiable/redistributable under the terms of the GNU
 * General Public License.
 *
 * You should have recieved a copy of the GNU General Public License along
 * with this program; see the file COPYING. If not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 0211-1307, USA.
 */

#ifndef ANTHEM_NEWPHRASE_H
#define ANTHEM_NEWPHRASE_H

#include <kdialogbase.h>

class QCheckBox;
class PhraseSelectWidget;

namespace TSE3
{
    class PhraseList;
}

/**
 * The NewPhraseWindow is opened by the @ref SongWindow when the user has
 * finished recording a @ref Phrase. The recording is carried out with a
 * @ref TSE3_Application::Record, but this class never needs to see it.
 *
 * It just allows you to choose a new name for the @ref Phrase to be
 * inserted, plus specify whether a @ref Part should be automatically
 * inserted, and whether any @ref Phrase with the same name as that specified
 * should be replaced.
 *
 * The user can alternatively choose to cancel the operation, at which point
 * the @ref SongWindow will delete the recorded @ref PhraseEdit.
 *
 * When the dialogue is done with (i.e. cancelled or OKed) it will raise
 * the newPhraseDone signal.
 *
 * You can be guaranteed that the signal will only be raised if the new
 * @ref Phrase title doesn't already exist in the @ref Song (and the replace
 * checkbox isn't checked).
 *
 * @short   New Phrase name selection window
 * @author  Pete Goodliffe
 * @version 1.0
 */
class NewPhraseWindow : public KDialogBase
{
        Q_OBJECT

    public:

        /**
         * Creates a new phrase window. You may suggest a title to be
         * put in the line editor. You must supply the @ref Song's
         * @ref PhraseList so that the dialogue knows what @ref Phrase
         * titles already exist.
         *
         * Creating the window will not show it, to give you a chance to
         * connect to it's signal.
         *
         * @param phraseList    @ref TSE3::Phrase to use
         * @param title         Initial suggested title
         * @param replacePhrase Whether to show a "replace phrase" check box
         * @param insertPart    Whether to show a "insert part" check box
         * @param keepOpen      Whether to show a "keep open" check box
         */
        NewPhraseWindow(QWidget *parent,
                        TSE3::PhraseList *phraseList,
                        QString title       = "",
                        bool replacePhrase  = true,
                        bool insertPart     = true,
                        bool keepOpen       = false);

    protected slots:

        void accept();
        void done(int reason);

    signals:

        /**
         * This signal is emitted when the dialogue is done with. The
         * information supplied is:
         *
         * @li @p success       True if the NewPhraseWindow was OKed,
         *                      false if cancelled.
         * @li @p newName       The new name for the @ref phrase
         * @li @p replacePhrase If there was a "replace Phrase" checkbox
         *                      - wheter it was selected
         * @li @p insertPart    If there was an "insert Part" checkbox - whether
         *                      it was selected
         * @li @p keepOpen      If there was a "keep open" checkbox - whether
         *                      it was checked
         */
        void newPhraseDone(bool success, QString title,
                           bool replacePhrase, bool insertPart,
                           bool keepOpen);

    private:

        TSE3::PhraseList   *phraseList;
        PhraseSelectWidget *titleEdit;
        QCheckBox          *replace;
        QCheckBox          *insert;
        QCheckBox          *keepOpen;

};

#endif
