/*
 * @(#)SongInfo.cpp 1.00 22 May 2000
 *
 * Copyright (c) Pete Goodliffe 2000 (pete@cthree.org)
 *
 * This file is part of anthem - the TSE3 sequencer.
 *
 * This program is modifiable/redistributable under the terms of the GNU
 * General Public License.
 *
 * You should have recieved a copy of the GNU General Public License along
 * with this program; see the file COPYING. If not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 0211-1307, USA.
 */

#include "dialogues/SongInfo.h"

#include "Application.h"

#include "tse3/Song.h"
#include "tse3/cmd/Song.h"
#include "tse3/cmd/CommandHistory.h"

#include <qlabel.h>
#include <qlayout.h>
#include <qlineedit.h>

namespace
{
    const int LINEEDIT_WIDTH = 300;
}

SongInfoWindow::SongInfoWindow(QWidget *parent, TSE3::Song *song)
: KDialogBase(Plain, "Song Information", Cancel|Ok, Ok, parent,
              "SongInfoWindow", false, true),
  song(song)
{
    QGridLayout *gl = new QGridLayout(plainPage(), 4, 2, spacingHint());

    // Title
    QLabel *label = new QLabel("Title", plainPage());
    label->setMinimumSize(label->sizeHint());
    label->setAlignment(AlignRight|AlignVCenter);
    gl->addWidget(label, 0, 0);
    titleEdit = new QLineEdit(QString(song->title().c_str()), plainPage());
    titleEdit->setMinimumWidth(LINEEDIT_WIDTH);
    gl->addWidget(titleEdit, 0, 1);
    titleEdit->setFocus();

    // Author
    label = new QLabel("Author", plainPage());
    label->setMinimumSize(label->sizeHint());
    label->setAlignment(AlignRight|AlignVCenter);
    gl->addWidget(label, 1, 0);
    authorEdit = new QLineEdit(QString(song->author().c_str()), plainPage());
    authorEdit->setMinimumWidth(LINEEDIT_WIDTH);
    gl->addWidget(authorEdit, 1, 1);

    // Copyright
    label = new QLabel("Copyright", plainPage());
    label->setMinimumSize(label->sizeHint());
    label->setAlignment(AlignRight|AlignVCenter);
    gl->addWidget(label, 2, 0);
    copyrightEdit
        = new QLineEdit(QString(song->copyright().c_str()), plainPage());
    copyrightEdit->setMinimumWidth(LINEEDIT_WIDTH);
    gl->addWidget(copyrightEdit, 2, 1);

    // Date
    label = new QLabel("Date", plainPage());
    label->setMinimumSize(label->sizeHint());
    label->setAlignment(AlignRight|AlignVCenter);
    gl->addWidget(label, 3, 0);
    dateEdit = new QLineEdit(QString(song->date().c_str()), plainPage());
    dateEdit->setMinimumWidth(LINEEDIT_WIDTH);
    gl->addWidget(dateEdit, 3, 1);

    attachTo(song);
    show();
}


void SongInfoWindow::updateWidgets()
{
    titleEdit->setText(song->title().c_str());
    authorEdit->setText(song->author().c_str());
    copyrightEdit->setText(song->copyright().c_str());
    dateEdit->setText(song->date().c_str());
}


void SongInfoWindow::accept()
{
    /*
     * Without using the TSE3::CommandHistory classes this would have been
     * simply the following:
     *
     * song->setTitle(titleEdit->text().latin1());
     * song->setAuthor(authorEdit->text().latin1());
     * song->setCopyright(copyrightEdit->text().latin1());
     * song->setDate(dateEdit->text().latin1());
     *
     * but the extra effort is worth it...
     */

    TSE3::Cmd::Song_SetInfo *s
        = new TSE3::Cmd::Song_SetInfo(song,
                                      titleEdit->text().latin1(),
                                      authorEdit->text().latin1(),
                                      copyrightEdit->text().latin1(),
                                      dateEdit->text().latin1());
    s->execute();
    Application::application()->history(song)->add(s);
    done(Accepted);
}


void SongInfoWindow::done(int reason)
{
    closed();
}


void SongInfoWindow::Song_InfoAltered(TSE3::Song *src)
{
    updateWidgets();
}

