/*
 * @(#)DPColourWidget.cpp 1.00 16 November 2000
 *
 * Copyright (c) Pete Goodliffe 2000 (pete@cthree.org)
 *
 * This file is part of anthem - the TSE3 sequencer.
 *
 * This program is modifiable/redistributable under the terms of the GNU
 * General Public License.
 *
 * You should have recieved a copy of the GNU General Public License along
 * with this program; see the file COPYING. If not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 0211-1307, USA.
 */

#include "gadgets/DPColourWidget.h"

#include "Application.h"
#include "gadgets/ColourPixmap.h"

#include "tse3/DisplayParams.h"

#include <kcolordialog.h>

namespace
{
    /**
     * The positions of the menu entries.
     */
    enum QCOMBOBOXITEMINDEXES
    {
        DEFAULT,
        CUSTOM,
        NONE,
        PRESET
    };

    /**
     * Textual names of the standard menu entries.
     */
    const char *standardItems[] =
    {
        "Default",
        "Custom",
        "No colour"
    };
}


DPColourWidget::DPColourWidget(TSE3::DisplayParams *dp = 0, bool useDefault,
                               QWidget *parent = 0, const char *name = 0)
: QComboBox(parent, name), _style(0), _preset(0)
{
    if (useDefault)
        insertItem(ColourPixmap::newPixmap(), standardItems[DEFAULT]);
    else
        insertItem(ColourPixmap::newPixmap(), "None");
    insertItem(ColourPixmap::newPixmap(0, 0, 0), standardItems[CUSTOM]);
    insertItem(ColourPixmap::newPixmap(),        standardItems[NONE]);
    // I long for a separator here
    for (size_t n = 0; n < TSE3::DisplayParams::NoPresetColours; n++)
    {
        int r, g, b;
        Application::application()->presetColours()->colour(n, r, g, b);
        insertItem(ColourPixmap::newPixmap(r, g, b),
                   TSE3::DisplayParams::presetColourString(n));
    }
    connect(this, SIGNAL(activated(int)), SLOT(slotActivated(int)));
    if (dp) valuesFrom(dp);
}


DPColourWidget::~DPColourWidget()
{
}


void DPColourWidget::valuesTo(TSE3::DisplayParams *dp)
{
    dp->setStyle(_style);
    int r, g, b;
    _colour.rgb(&r, &g, &b);
    dp->setColour(r, g, b);
    dp->setPresetColour(_preset);
}


void DPColourWidget::valuesFrom(TSE3::DisplayParams *dp)
{
    int r, g, b;
    dp->colour(r, g, b);
    changeColour(r, g, b);
    changePresetColour(dp->presetColour());
    changeStyle(dp->style());
}


void DPColourWidget::changeStyle(int s)
{
    _style = s;

    switch (s)
    {
        case TSE3::DisplayParams::Default:
        {
            setCurrentItem(DEFAULT);
            break;
        }
        case TSE3::DisplayParams::Colour:
        {
            setCurrentItem(CUSTOM);
            break;
        }
        case TSE3::DisplayParams::PresetColour:
        {
            setCurrentItem(PRESET+_preset);
            break;
        }
        case TSE3::DisplayParams::None:
        {
            setCurrentItem(NONE);
            break;
        }
    }
}


void DPColourWidget::changeColour(int r, int g, int b)
{
    _colour = QColor(r, g, b);
    setCurrentItem(CUSTOM);
    changeItem(ColourPixmap::newPixmap(r, g, b), standardItems[CUSTOM], CUSTOM);
}


void DPColourWidget::changePresetColour(int pc)
{
    if (pc >= 0 && pc < TSE3::DisplayParams::NoPresetColours)
    {
        _preset = pc;
        setCurrentItem(PRESET+pc);
    }
}


void DPColourWidget::slotActivated(int index)
{
    if (index == DEFAULT)
    {
        _style = TSE3::DisplayParams::Default;
    }
    else if (index == CUSTOM)
    {
        _style = TSE3::DisplayParams::Colour;
        KColorDialog::getColor(_colour, this);
        changeItem(ColourPixmap::newPixmap(_colour),
                   standardItems[CUSTOM], CUSTOM);
    }
    else if (index == NONE)
    {
        _style = TSE3::DisplayParams::None;
    }
    else
    {
        _style  = TSE3::DisplayParams::PresetColour;
        _preset = index - PRESET;
    }
}
