/*
 * @(#)Gadgets.cpp 1.00 15 June 2000
 *
 * Copyright (c) Pete Goodliffe 2000 (pete@cthree.org)
 *
 * This file is part of anthem - the TSE3 sequencer.
 *
 * This program is modifiable/redistributable under the terms of the GNU
 * General Public License.
 *
 * You should have recieved a copy of the GNU General Public License along
 * with this program; see the file COPYING. If not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 0211-1307, USA.
 */

#define QT_ALTERNATE_QTSMANIP

#include "gadgets/Gadgets.h"

#include "Application.h"
#include "gadgets/KeyWidgets.h"

#include "misc/kde2-compat.h"
#include <qlayout.h>
#include <qtimer.h>
#include <qlabel.h>
#include <qpainter.h>
#include <qstyle.h>

#include "tse3/MidiScheduler.h"
#include "tse3/Transport.h"

namespace
{
    const int MSECS = 10;
}

/******************************************************************************
 * VuBarWidget class
 *****************************************************************************/

/**
 * This widget is a single vu bar.
 *
 * It is used by the VuWidget class soley.
 *
 * @internal
 */
class VuBarWidget : public QWidget
{
    public:

        /**
         * Creates a new VuBarWidget with height of zero.
         */
        VuBarWidget(QWidget *parent);

        /**
         * Set a new height, and cause as much redraw as necessary.
         * This value must be between 0 - 127.
         */
        void newHeight(int newHeight);

        /**
         * Decrement the vu bar by one value. Call this as often as
         * you want.
         */
        void decrement();

        /**
         * Level the vu bar.
         */
        void level();

    protected:

        virtual void paintEvent(QPaintEvent *e);

        int _height;
};


VuBarWidget::VuBarWidget(QWidget *parent)
: QWidget(parent), _height(0)
{
}


void VuBarWidget::newHeight(int newHeight)
{
    if (newHeight > _height)
    {
        _height = newHeight;
        update();
    }
}


void VuBarWidget::decrement()
{
    if (_height > 0)
    {
        _height--;
        update();
    }
}


void VuBarWidget::level()
{
    if (_height > 0)
    {
        _height = 0;
        update();
    }
}


void VuBarWidget::paintEvent(QPaintEvent *e)
{
    QPainter p(this);
    p.setPen(colorGroup().foreground());
#if KDE_VERSION < 300
    style().drawPanel(&p, 0, 0, width(), height(), colorGroup(), true);
#else
    style().drawPrimitive(QStyle::PE_Panel, &p, QRect(0,0, width(), height()),
                          colorGroup());
#endif
    p.fillRect(0, height(), width(), -height()*_height/127, Qt::blue);
}


/******************************************************************************
 * VuWidget class
 *****************************************************************************/

/**
 * This widget is a set of VU bars for an entire port (i.e. a collection
 * of 16 bars, one for each channel. It uses the VuBarWidget class.
 *
 * @internal
 */
class VuWidget : public QWidget
{
    public:

        /**
         * The @p portNo is only used to show an port no at the
         * bottom.
         */
        VuWidget(QWidget *parent, int portNo);

        /**
         * 'Tickles' a vu bar. This tells the vu bar for this channel
         * to raise to the specified value (which is between 0-127).
         */
        void tickle(int channel, int value);

        /**
         * Call this as often as you feel the urge in order to
         * decrement each vu bar down on value.
         */
        void decrement();

        /**
         * Level all the vu bars. The would be useful when the window is
         * newly opened and should have no raised vu levels.
         */
        void level();

    private:

        VuBarWidget *vubar[16];
};


VuWidget::VuWidget(QWidget *parent, int portNo)
: QWidget(parent)
{
    QGridLayout *layout = new QGridLayout(this, 3, 16);
    for (size_t n = 0; n < 16; n++)
    {
        vubar[n] = new VuBarWidget(this);
        QString s;
        s.sprintf("%d", n+1);
        QLabel *label = new QLabel(s, this);
        layout->addWidget(label, 1, n);
        s.sprintf("%d", portNo+1);
        label = new QLabel(s, this);
        layout->addWidget(label, 2, n);
        vubar[n]->setFixedWidth(label->sizeHint().width());
        vubar[n]->setFixedHeight(label->sizeHint().height()*10);
        layout->addWidget(vubar[n], 0, n);
    }
}


void VuWidget::tickle(int channel, int value)
{
    vubar[channel]->newHeight(value);
}


void VuWidget::decrement()
{
    for (size_t n = 0; n < 16; n++)
    {
        vubar[n]->decrement();
    }
}


void VuWidget::level()
{
    for (size_t n = 0; n < 16; n++)
    {
        vubar[n]->level();
    }
}


/******************************************************************************
 * VuWindow class
 *****************************************************************************/

int VuWindow::_status = VuWindow::Both;

VuWindow::VuWindow()
: KDialog(0, 0, false),  _allCommands(false)
{
    QHBoxLayout *layout = new QHBoxLayout(this);
    noPorts = Application::application()->scheduler()->numPorts();
    timer = new QTimer(this);
    connect(timer, SIGNAL(timeout()), SLOT(slotTimeout()));
    vuwidgets = new VuWidget*[noPorts];
    for (size_t n = 0; n < noPorts; n++)
    {
        vuwidgets[n] = new VuWidget(this, n);
        layout->addWidget(vuwidgets[n]);
    }
    Application::application()->transport()->attachCallback(this);
    setCaption("VU bars");
}


VuWindow::~VuWindow()
{
    Application::application()->transport()->detachCallback(this);
    for (size_t n = 0; n < noPorts; n++)
    {
        delete vuwidgets[n];
    }
    delete vuwidgets;
}


void VuWindow::setAllCommands(bool allCommands)
{
    _allCommands = allCommands;
}


void VuWindow::setStatus(int status)
{
    _status = status;
}


int VuWindow::status()
{
    return _status;
}


bool VuWindow::allCommands()
{
    return _allCommands;
}


void VuWindow::Transport_MidiIn(TSE3::MidiCommand c)
{
    if (c.status == TSE3::MidiCommand_NoteOn)
    {
        vuwidgets[c.port]->tickle(c.channel, c.data2);
    }
}


void VuWindow::Transport_MidiOut(TSE3::MidiCommand c)
{
    if (c.status == TSE3::MidiCommand_NoteOn)
    {
        vuwidgets[c.port]->tickle(c.channel, c.data2);
    }
}


void VuWindow::showEvent(QShowEvent *e)
{
    timer->start(MSECS, false);
    for (size_t n = 0; n < noPorts; n++)
    {
        vuwidgets[n]->level();
    }
    KDialog::showEvent(e);
}


void VuWindow::hideEvent(QHideEvent *e)
{
    timer->stop();
    KDialog::hideEvent(e);
}


void VuWindow::slotTimeout()
{
    for (size_t n = 0; n < noPorts; n++)
    {
        vuwidgets[n]->decrement();
    }
}


/******************************************************************************
 * KeyboardWindow class
 *****************************************************************************/

KeyboardWindow::KeyboardWindow()
: KDialog(0, 0, false)
{
    QHBoxLayout *layout = new QHBoxLayout(this);
    for (size_t octave = 0; octave < 11; octave++)
    {
        KeyOctaveWidget *kow = new KeyOctaveWidget(this, false, octave*12);
        kow->setMinimumWidth(50);
        kow->setMinimumHeight(50);
        layout->addWidget(kow);
        connect(kow, SIGNAL(clicked(int)), SLOT(slotClicked(int)));
        connect(kow, SIGNAL(released()), SLOT(slotReleased()));
    }
    setCaption("Keyboard");
}


KeyboardWindow::~KeyboardWindow()
{
}


void KeyboardWindow::slotClicked(int key)
{
    lastKeyClicked = key;
    const int channel = 0, port = 0, vel = 127;
    TSE3::MidiCommand c(TSE3::MidiCommand_NoteOn, channel, port, key, vel);
    Application::application()->transport()->inject(c);
}


void KeyboardWindow::slotReleased()
{
    const int channel = 0, port = 0, vel = 127;
    TSE3::MidiCommand c(TSE3::MidiCommand_NoteOff, channel, port,
                        lastKeyClicked, vel);
    Application::application()->transport()->inject(c);
}

