/*
 * @(#)KeyWidgets.cpp 1.00 1 June 2000
 *
 * Copyright (c) Pete Goodliffe 2000 (pete@cthree.org)
 *
 * This file is part of anthem - the TSE3 sequencer.
 *
 * This program is modifiable/redistributable under the terms of the GNU
 * General Public License.
 *
 * You should have recieved a copy of the GNU General Public License along
 * with this program; see the file COPYING. If not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 0211-1307, USA.
 */

#include "gadgets/KeyWidgets.h"

#include <qlayout.h>
#include <qspinbox.h>
#include <qpainter.h>
#include <qpushbutton.h>

#include <kdialogbase.h>

#include "tse3/util/NoteNumber.h"

/******************************************************************************
 * KeyOctaveWidgetclass
 *****************************************************************************/

namespace
{
    /**
     * For each white key in order, this table gives the relative MIDI
     * key value (where C is 0)
     */
    const int  whiteKeyLookup[] = {0, 2, 4, 5, 7, 9, 11};

    /**
     * This table indicates whether a given white key has a black key to
     * it's right.
     */
    const bool blackKeyLookup[] = {1, 1, 0, 1, 1, 1, 0 };
}

KeyOctaveWidget::KeyOctaveWidget(QWidget *parent, bool lastLine, int base)
: QFrame(parent, 0), lastLine(lastLine), base(base), highlight(-1)
{
    setBackgroundColor(Qt::white);
}


KeyOctaveWidget::~KeyOctaveWidget()
{
}


void KeyOctaveWidget::setHighlight(int h)
{
    highlight = h;
}


void KeyOctaveWidget::mousePressEvent(QMouseEvent *e)
{
    const int keyWidth = width()/7;
    int key            = whiteKeyLookup[e->x() / keyWidth];
    int x              = e->x()-1;
    int y              = e->y();
    if (y < height()/2)
    {
        if (key > 0 && (x%keyWidth) < keyWidth/3 && blackKeyLookup[key-1])
        {
            key--;
        }
        else if ((x%keyWidth) > keyWidth*2/3 && blackKeyLookup[key])
        {
            key++;
        }
    }
    emit clicked(key+base);
}


void KeyOctaveWidget::mouseReleaseEvent(QMouseEvent *e)
{
    emit released();
}


void KeyOctaveWidget::drawContents(QPainter *p)
{
    const float keyWidth = (width()+(lastLine?0:1))/7;
    static const bool blackKeyLookup[] = {1, 1, 0, 1, 1, 1, 0 };

    p->setPen(Qt::black);
    p->drawRect(0, 0, width() + (lastLine?0:1), height());

    for (size_t n = 0; n < 7; n++)
    {
        p->setPen(QColor(0xdd,0xdd,0xdd));
        p->drawLine(keyWidth*n+1, 1, keyWidth*n+1, height()-2);

        p->setPen(QColor(0x66,0x66,0x66));
        p->drawLine(keyWidth*n+1,     height()-2, keyWidth*(n+1)-1, height()-2);
        p->drawLine(keyWidth*(n+1)-1, 1,          keyWidth*(n+1)-1, height()-2);

        p->setPen(Qt::black);
        p->drawLine(keyWidth*n, 0, keyWidth*n, height());

        if (blackKeyLookup[n])
        {
            p->fillRect(1 + keyWidth*n + (keyWidth*2/3), 0,
                        keyWidth*2/3, height()/2,
                        QColor(0,0,0));
        }
    }
}


/******************************************************************************
 * KeySpinBox widget class
 *****************************************************************************/

/**
 * This is a special kind of spin box that handles values in the
 * KeyWidget class.
 *
 * @short   Key value spin box
 * @author  Pete Goodliffe
 * @version 1.0
 */
class KeySpinBox : public QSpinBox
{
    public:
        KeySpinBox(QWidget *parent = 0, const char *name = 0);
    protected:
        virtual QString mapValueToText(int v);
        virtual int mapTextToValue(bool *ok);
};


KeySpinBox::KeySpinBox(QWidget *parent = 0, const char *name = 0)
: QSpinBox(0, 127, 1, parent, name)
{
}


QString KeySpinBox::mapValueToText(int k)
{
    return TSE3::Util::numberToNote(k).c_str();
}


int KeySpinBox::mapTextToValue(bool *ok)
{
    std::string str(text());
    return TSE3::Util::noteToNumber(str);
}


/******************************************************************************
 * KeyWidgetPopup class
 *****************************************************************************/

KeyWidgetPopup::KeyWidgetPopup(int key, QWidget *parent)
: QPopupMenu(parent, 0), key(key)
{
    QWidget *item = new QWidget(this);

    QVBoxLayout *layout = new QVBoxLayout(item, KDialogBase::spacingHint());

    whole = new KeySpinBox(item);
    whole->setValue(key);
    layout->addWidget(whole);
    connect(whole, SIGNAL(valueChanged(int)), SLOT(slotWholeValueChanged(int)));

    keyboard = new KeyOctaveWidget(item);
    keyboard->setMinimumWidth(7*10);
    keyboard->setMinimumHeight(40);
    layout->addWidget(keyboard);
    connect(keyboard, SIGNAL(clicked(int)),
            SLOT(slotKeyValueChanged(int)));

    octave = new QSpinBox(0, 10, 1, item);
    octave->setValue(key/12);
    layout->addWidget(octave);
    connect(octave, SIGNAL(valueChanged(int)),
            SLOT(slotOctaveValueChanged(int)));

    updateWidgets();

    insertItem(item);
}


KeyWidgetPopup::~KeyWidgetPopup()
{
}


void KeyWidgetPopup::setValue(int value)
{
    if (value >= 0 && value <= 127)
    {
        key = value;
        updateWidgets();
    }
}


void KeyWidgetPopup::hideEvent(QHideEvent *e)
{
    QWidget::hide();
    emit hiden();
}


void KeyWidgetPopup::updateWidgets()
{
    // We block signals to prevent signal cascading
    {
        // Update the whole value spin box
        bool b = whole->signalsBlocked();
        whole->blockSignals(true);
        whole->setValue(key);
        whole->blockSignals(b);
    }
    {
        // Update the octave spin box
        bool b = octave->signalsBlocked();
        octave->blockSignals(true);
        octave->setValue(key/12);
        octave->blockSignals(b);
    }
}


void KeyWidgetPopup::slotKeyValueChanged(int value)
{
    key -= key%12;
    key += value;
    updateWidgets();
    emit valueChanged(key);
}


void KeyWidgetPopup::slotOctaveValueChanged(int value)
{
    key = (key%12) + (value*12);
    updateWidgets();
    emit valueChanged(key);
}


void KeyWidgetPopup::slotWholeValueChanged(int value)
{
    key = value;
    updateWidgets();
    emit valueChanged(key);
}


/******************************************************************************
 * KeyWidget class
 *****************************************************************************/

KeyWidget::KeyWidget(int key, QWidget *parent, const char *name)
: QWidget(parent, name)
{
    // This widget
    QHBoxLayout *layout = new QHBoxLayout(this, KDialogBase::spacingHint());
    sb = new KeySpinBox(this);
    sb->setValue(key);
    layout->addWidget(sb);
    connect(sb, SIGNAL(valueChanged(int)), SIGNAL(valueChanged(int)));

    QPushButton *pb = new QPushButton(">", this);
    pb->setFixedSize(sb->sizeHint());
    layout->addWidget(pb);

    // The popup
    p = new KeyWidgetPopup(sb->value(), this);
    pb->setPopup(p);
    connect(p, SIGNAL(valueChanged(int)), SIGNAL(valueChanged(int)));
    connect(p, SIGNAL(valueChanged(int)), SLOT(slotValueChanged(int)));
    connect(p, SIGNAL(hiden()),           SLOT(slotPopupClosed()));
}


KeyWidget::~KeyWidget()
{
}


void KeyWidget::slotValueChanged(int value)
{
    sb->setValue(value);
}


void KeyWidget::slotShowPopup()
{
    p->show();
}


void KeyWidget::slotPopupClosed()
{
}


int KeyWidget::value()
{
    return sb->value();
}


void KeyWidget::setValue(int k)
{
    sb->setValue(k);
    emit valueChanged(sb->value());
}

