/*
 * @(#)KeyWidgets.h 1.00 1 June 2000
 *
 * Copyright (c) Pete Goodliffe 2000 (pete@cthree.org)
 *
 * This file is part of anthem - the TSE3 sequencer.
 *
 * This program is modifiable/redistributable under the terms of the GNU
 * General Public License.
 *
 * You should have recieved a copy of the GNU General Public License along
 * with this program; see the file COPYING. If not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 0211-1307, USA.
 */

#ifndef ANTHEM_GADGETS_KEYWIDGETS_H
#define ANTHEM_GADGETS_KEYWIDGETS_H

#include <qframe.h>
#include <qpopupmenu.h>

class QSpinBox;

/**
 * This is a class that draws a kayboard octave on the screen and can
 * manage clicks on it.
 *
 * It is used by the @ref KeyPopupWidget.
 *
 * @internal
 * @short   On screen keyboard cotave widget.
 * @author  Pete Goodliffe
 * @version 1.0
 */
class KeyOctaveWidget : public QFrame
{
        Q_OBJECT;

    public:

        /**
         * Specify a parent widget, and an optional base value for the
         * widgets return values (see @ref clicked()).
         *
         * @param parent   Parent widget
         * @param lastLine Whether or not to draw the last line of the
         *                 widget - you don't want to draw this if you are
         *                 stringing another KeyOctaveWidget directly beside
         *                 this one
         * @param base     MIDI key number of base C key
         */
        KeyOctaveWidget(QWidget *parent, bool lastLine = true, int base = 0);

        virtual ~KeyOctaveWidget();

        /**
         * Set which key is drawn with a highlight. Forces a redraw to occur
         * if the highlight changes.
         *
         * If you specify the value -1, then there will be no highlight.
         */
        void setHighlight(int highlight = -1);

    signals:

        /**
         * The is emited when the user clicks on the widget. It returns
         * the key value that was clicked, one of:
         *
         * @li @p 0  - C
         * @li @p 1  - C#
         * @li @p 2  - D
         * @li @p 3  - D#
         * @li @p 4  - E
         * @li @p 5  - F
         * @li @p 6  - F#
         * @li @p 7  - G
         * @li @p 8  - G#
         * @li @p 9  - A
         * @li @p 10 - A#
         * @li @p 11 - B
         *
         * This value is added to the base value specified in the ctor.
         */
        void clicked(int key);

        /**
         * Emitted whenever the clicked key is released.
         */
        void released();

    protected:

        virtual void mousePressEvent(QMouseEvent *p);
        virtual void mouseReleaseEvent(QMouseEvent *p);
        virtual void drawContents(QPainter *p);

    private:

        bool lastLine;
        int  base;
        int  highlight;
};

/**
 * This is the class that provides a popup key choice widget.
 *
 * It is used by the @ref KeyWidget.
 *
 * @internal
 * @short   Key choice popup widget
 * @author  Pete Goodliffe
 * @version 1.0
 */
class KeyWidgetPopup : public QPopupMenu
{
        Q_OBJECT;

    public:

        /**
         * Specify where on the screen to 'popup'
         */
        KeyWidgetPopup(int key, QWidget *parent);

        /**
         * Dtor - the @Ref Track is not deleted.
         */
        virtual ~KeyWidgetPopup();

        /**
         * Sets a new key value
         */
        void setValue(int value);

        /**
         * Returns the current value
         */
        int value();

    private slots:

        void slotKeyValueChanged(int value);
        void slotOctaveValueChanged(int value);
        void slotWholeValueChanged(int value);

    signals:

        void valueChanged(int value);
        void hiden();

    protected:

        virtual void hideEvent(QHideEvent *e);

    private:

        void updateWidgets();

        int key;

        QSpinBox        *whole;
        KeyOctaveWidget *keyboard;
        QSpinBox        *octave;
};

/**
 * A widget that allows the user to select a MIDI key.
 *
 * @short   Key choice widget
 * @author  Pete Goodliffe
 * @version 1.0
 */
class KeyWidget : public QWidget
{
        Q_OBJECT

    public:

        /**
         * Creates a KeyWidget.
         */
        KeyWidget(int key = 0x40, QWidget *parent = 0, const char *name = 0);
        ~KeyWidget();

        /**
         * Returns the currently selected key.
         */
        int value();

    public slots:

        /**
         * Sets the current key value.
         */
        void setValue(int key);

    signals:

        void valueChanged(int value);

    private slots:

        void slotValueChanged(int value);
        void slotShowPopup();
        void slotPopupClosed();

    private:

        QSpinBox       *sb;
        KeyWidgetPopup *p;
};

#endif
