/*
 * @(#)MidiSchedulerWidgets.cpp 1.00 1 June 2000
 *
 * Copyright (c) Pete Goodliffe 2000 (pete@cthree.org)
 *
 * This file is part of anthem - the TSE3 sequencer.
 *
 * This program is modifiable/redistributable under the terms of the GNU
 * General Public License.
 *
 * You should have recieved a copy of the GNU General Public License along
 * with this program; see the file COPYING. If not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 0211-1307, USA.
 */

#include "gadgets/MidiSchedulerWidgets.h"

#include <qlabel.h>
#include <qlayout.h>
#include <qtimer.h>

#include <kdialogbase.h>

#include "tse3/MidiScheduler.h"
#include "tse3/Song.h"
#include "tse3/util/Snap.h"

#include "tse3/kdeui/ClockDisplayWidget.h"

/******************************************************************************
 * MidiSchedulerTimeWidget widget class
 *****************************************************************************/

MidiSchedulerTimeWidget::MidiSchedulerTimeWidget(QWidget *parent,
                                                 TSE3::MidiScheduler *scheduler)
: QFrame(parent, "MidiSchedulerTimeWidget"), scheduler(scheduler)
{
    QHBoxLayout *layout = new QHBoxLayout(this);
    display = new ClockDisplayWidget(scheduler->clock(), this);
    layout->add(display);

    timer = new QTimer(this);
    timer->start(50, false);
    connect(timer, SIGNAL(timeout()), SLOT(update()));
}


void MidiSchedulerTimeWidget::update()
{
    display->setValue(scheduler->clock());
}


/*
void MidiSchedulerTimeWidget::setBackgroundColor(const QColor &background)
{
    QFrame::setBackgroundColor(background);
    bar->setBackgroundColor(background);
    colon1->setBackgroundColor(background);
    beat->setBackgroundColor(background);
    colon2->setBackgroundColor(background);
    pulse->setBackgroundColor(background);
}
*/


/******************************************************************************
 * MidiSchedulerTimeSlider widget class
 *****************************************************************************/

namespace
{
    const int         SLIDER_MAX_VALUE = 10000;
    const int         SLIDER_STEP      = 500;
    const TSE3::Clock SLIDER_MAX_CLOCK = TSE3::Clock::PPQN * 100 * 4;
    const int         SLIDER_DELAY_MS  = 800;
}

MidiSchedulerTimeSlider::MidiSchedulerTimeSlider(QWidget *parent,
                                                 TSE3::MidiScheduler *scheduler,
                                                 TSE3::Song          *song,
                                                 TSE3::Util::Snap    *snap)
: QSlider(0, SLIDER_MAX_VALUE, SLIDER_STEP, 0, Horizontal,
          parent, "MidiSchedulerSlider"),
  scheduler(scheduler), song(song), snap(snap), sliderMoved(false)
{
    if (song)
    {
        attachTo(song);
    }

    connect(this, SIGNAL(valueChanged(int)), SLOT(slotSliderMoved(int)));

    timer = new QTimer(this);
    timer->start(SLIDER_DELAY_MS, false);
    connect(timer, SIGNAL(timeout()), SLOT(update()));
}


void MidiSchedulerTimeSlider::setSong(TSE3::Song *newSong)
{
    if (song)
    {
        detachFrom(song);
    }
    song = newSong;
    if (song)
    {
        attachTo(song);
    }
}


void MidiSchedulerTimeSlider::update()
{
    TSE3::Clock lastClock = song ? song->lastClock() : SLIDER_MAX_CLOCK;
    if (lastClock == 0) lastClock = 1;

    if (sliderMoved)
    {
        TSE3::Clock newClock = value() * lastClock / SLIDER_MAX_VALUE;
        if (snap)   newClock = (*snap)(newClock);
        scheduler->moveTo(newClock);
        sliderMoved = false;
        return;
    }

    int newValue = SLIDER_MAX_VALUE * scheduler->clock() / lastClock;

    blockSignals(true);
    setValue(newValue);
    blockSignals(false);
}


void MidiSchedulerTimeSlider::slotSliderMoved(int value)
{
    sliderMoved = true;
}


void MidiSchedulerTimeSlider::Notifier_Deleted(TSE3::Song *s)
{
    if (song == s)
    {
        song = 0;
    }
}


/******************************************************************************
 * MidiSchedulerInfoWidget widget class
 *****************************************************************************/

MidiSchedulerInfoWidget::MidiSchedulerInfoWidget(QWidget *parent,
                                                 TSE3::MidiScheduler *scheduler)
: QFrame(parent, "MidiSchedulerInfoWidget")
{
    QVBoxLayout *layout  = new QVBoxLayout(this);
    QGridLayout *layout1 =
        new QGridLayout(layout, 2, 2, KDialogBase::spacingHint());

    QLabel *label = new QLabel("Implementation name:", this);
    label->setAlignment(AlignRight|AlignVCenter);
    layout1->addWidget(label, 0, 0);
    label = new QLabel(scheduler->implementationName(), this);
    layout1->addWidget(label, 0, 1);

    label = new QLabel("Number of ports:", this);
    label->setAlignment(AlignRight|AlignVCenter);
    layout1->addWidget(label, 1, 0);
    QString s;
    s.sprintf("%d", scheduler->numPorts());
    label = new QLabel(s, this);
    layout1->addWidget(label, 1, 1);

    for (size_t port = 0; port < scheduler->numPorts(); port++)
    {

        QFrame *separator = new QFrame(this);
        separator->setFixedHeight(label->sizeHint().height());
        separator->setFrameStyle(QFrame::HLine | QFrame::Sunken);
        layout->addWidget(separator);

        QGridLayout *layout2 =
            new QGridLayout(layout, 5, 2, KDialogBase::spacingHint());

        label = new QLabel("Port:", this);
        label->setAlignment(AlignRight|AlignVCenter);
        layout2->addWidget(label, 0, 0);
        QString s;
        s.sprintf("%d", scheduler->portNumber(port));
        label = new QLabel(s, this);
        layout2->addWidget(label, 0, 1);

        label = new QLabel("Name:", this);
        label->setAlignment(AlignRight|AlignVCenter);
        layout2->addWidget(label, 1, 0);
        label = new QLabel(scheduler->portName(scheduler->portNumber(port)),
                           this);
        layout2->addWidget(label, 1, 1);

        label = new QLabel("Type:", this);
        label->setAlignment(AlignRight|AlignVCenter);
        layout2->addWidget(label, 2, 0);
        label = new QLabel(scheduler->portType(scheduler->portNumber(port)),
                           this);
        layout2->addWidget(label, 2, 1);

        label = new QLabel("Readable:", this);
        label->setAlignment(AlignRight|AlignVCenter);
        layout2->addWidget(label, 3, 0);
        label = new QLabel(scheduler->portReadable(scheduler->portNumber(port))
                           ? "Yes" : "No",
                           this);
        layout2->addWidget(label, 3, 1);

        label = new QLabel("Readable:", this);
        label->setAlignment(AlignRight|AlignVCenter);
        layout2->addWidget(label, 4, 0);
        label = new QLabel(scheduler->portWriteable(scheduler->portNumber(port))
                           ? "Yes" : "No",
                           this);
        layout2->addWidget(label, 4, 1);
    }
}

