/*
 * @(#)PhraseSelectWidget.cpp 1.00 1 June 2000
 *
 * Copyright (c) Pete Goodliffe 2000 (pete@cthree.org)
 *
 * This file is part of anthem - the TSE3 sequencer.
 *
 * This program is modifiable/redistributable under the terms of the GNU
 * General Public License.
 *
 * You should have recieved a copy of the GNU General Public License along
 * with this program; see the file COPYING. If not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 0211-1307, USA.
 */

#include "gadgets/PhraseSelectWidget.h"

#include "phraseeditors/PhraseEditor.h"
#include "songwindow/SongWindow.h"
#include "Application.h"

#include "tse3/Phrase.h"
#include "tse3/PhraseList.h"

#include <qlayout.h>
#include <qpushbutton.h>
#include <kdialogbase.h>
#include <kiconloader.h>

/******************************************************************************
 * PhraseSelectWidget class
 *****************************************************************************/

PhraseSelectWidget::PhraseSelectWidget(TSE3::PhraseList *phraseList,
                                       bool allowNone,
                                       TSE3::Phrase *selected,
                                       QWidget *parent, const char *name = 0)
: QComboBox(false, parent, name), phraseList(phraseList), allowNone(allowNone)
{
    setInsertionPolicy(NoInsertion);
    if (allowNone)
    {
        insertItem("<None>");
    }
    for (size_t n = 0; n < phraseList->size(); n++)
    {
        insertItem(QString((*phraseList)[n]->title().c_str()));
    }

    connect(this, SIGNAL(activated(int)), SLOT(slotActivated(int)));
}


TSE3::Phrase *PhraseSelectWidget::phrase()
{
    if (allowNone && currentItem() == 0)
    {
        return 0;
    }
    else
    {
        int delta = allowNone ? 1 : 0;
        return (*phraseList)[currentItem() - delta];
    }
}


void PhraseSelectWidget::setPhrase(TSE3::Phrase *phrase)
{
    if (phrase)
    {
        setCurrentItem(phraseList->index(phrase) + (allowNone ? 1 : 0));
    }
    else
    {
        if (allowNone)
        {
            setCurrentItem(0);
        }
    }
}


void PhraseSelectWidget::slotActivated(int index)
{
    if (allowNone && index == 0)
    {
        emit phraseSelected(0);
    }
    else
    {
        if (allowNone && index != 0)
        {
            index--;
        }
        emit phraseSelected((*phraseList)[index]);
    }
}


/******************************************************************************
 * PhraseSelectEditWidget class
 *****************************************************************************/

PhraseSelectEditWidget::PhraseSelectEditWidget(TSE3::PhraseList *phraseList,
                                               bool allowNone,
                                               TSE3::Phrase *selected,
                                               TSE3::Song *song,
                                               QWidget *parent,
                                               const char *name = 0)
: QWidget(parent, name), song(song)
{
    QHBoxLayout *layout = new QHBoxLayout(this, 0, KDialogBase::spacingHint());
    select = new PhraseSelectWidget(phraseList, allowNone, selected, this);
    layout->add(select);
    button = new QPushButton("", this);
    button->setPixmap(UserIcon("edit"));
    button->setMaximumSize(button->sizeHint());
    layout->add(button);

    updateWidgets();

    connect(select, SIGNAL(phraseSelected(TSE3::Phrase*)),
            SIGNAL(phraseSelected(TSE3::Phrase*)));
    connect(select, SIGNAL(phraseSelected(TSE3::Phrase*)),
            SLOT(slotPhrase(TSE3::Phrase*)));
    connect(button, SIGNAL(clicked()), SLOT(slotButton()));
}


void PhraseSelectEditWidget::slotPhrase(TSE3::Phrase *phrase)
{
    updateWidgets();
}


void PhraseSelectEditWidget::slotButton()
{
    if (select->phrase())
    {
        PhraseEditor *pe
            = new PhraseEditor(Application::application()->songWindow(song),
                               song,
                               Application::application()->history(song),
                               select->phrase());
        pe->show();
    }
}


TSE3::Phrase *PhraseSelectEditWidget::phrase()
{
    return select->phrase();
}


void PhraseSelectEditWidget::setPhrase(TSE3::Phrase *phrase)
{
    select->setPhrase(phrase);
    updateWidgets();
}


void PhraseSelectEditWidget::updateWidgets()
{
    button->setEnabled(select->phrase() != 0);
}

