/*
 * @(#)VoiceWidget.cpp 1.00 3 April 2001
 *
 * Copyright (c) Pete Goodliffe 2000 (pete@cthree.org)
 *
 * This file is part of anthem - the TSE3 sequencer.
 *
 * This program is modifiable/redistributable under the terms of the GNU
 * General Public License.
 *
 * You should have recieved a copy of the GNU General Public License along
 * with this program; see the file COPYING. If not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 0211-1307, USA.
 */


#include "gadgets/VoiceWidget.h"

#include "Application.h"
#include "gadgets/MidiParamsSpinBox.h"
#include "tse3/kdeui/PortWidget.h"
#include "tse3/kdeui/ChannelWidget.h"

#include "tse3/ins/Destination.h"
#include "tse3/ins/Instrument.h"

#include <qlayout.h>
#include <qhbox.h>
#include <qlabel.h>
#include <qpushbutton.h>

#include <kdialogbase.h>
#include <klistview.h>

#include "misc/debug.h"

namespace
{
    const QString NONE_STR = "<None>";
}

/******************************************************************************
 * VoiceWidget class
 *****************************************************************************/

VoiceWidget::VoiceWidget(bool any, int p, int bLSB, int bMSB,
                         int ch, int prt,
                         QWidget *parent, const char *n)
: QWidget(parent, n), any(any),
  _channel(ch), _port(prt), _program(p), _bankLSB(bLSB), _bankMSB(bMSB)
{
    QVBoxLayout *layoutV = new QVBoxLayout(this, 0, KDialogBase::spacingHint());

    QHBox *frame = new QHBox(this);
    frame->setFrameStyle(QFrame::Panel | QFrame::Sunken);
    frame->setLineWidth(2);
    frame->setMargin(KDialogBase::spacingHint());
    layoutV->addWidget(frame);

    name = new QLabel("Name", frame, "Voice Name");
    number = new QLabel("Number", frame, "Voice Number");
    number->setAlignment(QLabel::AlignRight);

    QHBoxLayout *layoutH = new QHBoxLayout(layoutV);

    byName = new QPushButton("By name...", this);
    layoutH->addWidget(byName);
    byNumber = new QPushButton("By number...", this);
    layoutH->addWidget(byNumber);

    namePopup = new VoiceByNamePopup(this);
    byName->setPopup(namePopup);
    namePopup->slotSelect(_program ,_bankLSB, _bankMSB);

    numberPopup = new VoiceByNumberPopup(this);
    byNumber->setPopup(numberPopup);
    numberPopup->slotSelect(_program ,_bankLSB, _bankMSB);

    connect(numberPopup, SIGNAL(selected(int,int,int)),
            SLOT(slotVoiceSelected(int,int,int)));
    connect(numberPopup, SIGNAL(selected(int,int,int)),
            namePopup, SLOT(slotSelect(int,int,int)));
    connect(namePopup, SIGNAL(selected(int,int,int)),
            SLOT(slotVoiceSelected(int,int,int)));
    connect(namePopup, SIGNAL(selected(int,int,int)),
            numberPopup, SLOT(slotSelect(int,int,int)));

    updateWidgets();
}


int VoiceWidget::program()
{
    return _program;
}


void VoiceWidget::setProgram(int p)
{
    _program = p;
    numberPopup->slotSelect(_program, _bankLSB, _bankMSB);
    updateWidgets();
}


int VoiceWidget::bankLSB()
{
    return _bankLSB;
}


void VoiceWidget::setBankLSB(int b)
{
    _bankLSB = b;
    numberPopup->slotSelect(_program, _bankLSB, _bankMSB);
    updateWidgets();
}


int VoiceWidget::bankMSB()
{
    return _bankMSB;
}


void VoiceWidget::setBankMSB(int b)
{
    _bankMSB = b;
    numberPopup->slotSelect(_program, _bankLSB, _bankMSB);
    updateWidgets();
}


void VoiceWidget::setValues(int program, int bankLSB, int bankMSB)
{
    _program = program;
    _bankLSB = bankLSB;
    _bankMSB = bankMSB;
    numberPopup->slotSelect(_program, _bankLSB, _bankMSB);
    updateWidgets();
}


void VoiceWidget::slotVoiceSelected(int program, int bankLSB, int bankMSB)
{
    _program = program;
    _bankLSB = bankLSB;
    _bankMSB = bankMSB;
    updateWidgets();
}


void VoiceWidget::updateWidgets()
{
    name->setText
        (toTextName(_program, _bankLSB, _bankMSB, _channel, _port));
    name->setEnabled(name->text() != NONE_STR);
    number->setText
        (toTextNumber(_program, _bankLSB, _bankMSB, _channel, _port));
}


QString VoiceWidget::toTextName(int program, int bankLSB, int bankMSB,
                                int channel, int port)
{
    QString ret = NONE_STR;
    if (program != -1)
    {
        TSE3::Ins::Instrument *ins
            = Application::application()->destination()->channel(channel, port);

        if (ins)
        {
            TSE3::Ins::PatchData *patch = ins->patchForBank(bankLSB, bankMSB);
            if (patch)
            {
                ret = patch->name(program).c_str();
            }
        }
    }
    return ret;
}


QString VoiceWidget::toTextNumber(int program, int bankLSB, int bankMSB,
                                  int channel, int port)
{
    QString ret(MidiParamsSpinBox::valueToText(program));
    ret.append(":");
    ret.append(MidiParamsSpinBox::valueToText(bankLSB));
    ret.append(":");
    ret.append(MidiParamsSpinBox::valueToText(bankMSB));
    return ret;
}


/******************************************************************************
 * VoiceByNumberPopup class
 *****************************************************************************/

VoiceByNumberPopup::VoiceByNumberPopup(QWidget *parent, const char *name = 0)
: QPopupMenu(parent, name)
{
    QWidget *item = new QWidget(this);

    QGridLayout *layout
        = new QGridLayout(item, 2, 4, KDialogBase::spacingHint());

    // Program
    QLabel *label = new QLabel("Program", item);
    label->setAlignment(AlignRight|AlignVCenter);
    layout->addWidget(label, 0, 0);
    program = new MidiParamsSpinBox(item);
    layout->addWidget(program, 0, 1);

    // Bank MSB and LSB
    label = new QLabel("Bank LSB", item);
    label->setAlignment(AlignRight|AlignVCenter);
    layout->addWidget(label, 1, 0);
    bankLSB = new MidiParamsSpinBox(item);
    layout->addWidget(bankLSB, 1, 1);

    label = new QLabel("Bank MSB", item);
    label->setAlignment(AlignRight|AlignVCenter);
    layout->addWidget(label, 2, 0);
    bankMSB = new MidiParamsSpinBox(item);
    layout->addWidget(bankMSB, 2, 1);

    // Bank
    bankLabel = new QLabel("Bank", item);
    bankLabel->setAlignment(AlignRight|AlignVCenter);
    layout->addWidget(bankLabel, 3, 0);
    bank = new MidiParamsSpinBox(item, "", (1<<14)-1);
    layout->addWidget(bank, 3, 1);

    insertItem(item);

    connect(program, SIGNAL(valueChanged(int)), SLOT(slotChanged()));
    connect(bankLSB, SIGNAL(valueChanged(int)), SLOT(slotChanged()));
    connect(bankMSB, SIGNAL(valueChanged(int)), SLOT(slotChanged()));
    connect(bank,    SIGNAL(valueChanged(int)), SLOT(slotChangedBank()));
}


void VoiceByNumberPopup::slotSelect(int p, int bLSB, int bMSB)
{
    program->blockSignals(true);
    bankLSB->blockSignals(true);
    bankMSB->blockSignals(true);
    bank->blockSignals(true);

    program->setValue(p);
    bankLSB->setValue(bLSB);
    bankMSB->setValue(bMSB);
    bank->setValue(TSE3::Ins::bankFromBytes(bLSB, bMSB));

    program->blockSignals(false);
    bankLSB->blockSignals(false);
    bankMSB->blockSignals(false);
    bank->blockSignals(false);
}


void VoiceByNumberPopup::slotChanged()
{
    bank->blockSignals(true);
    bank->setValue
        (TSE3::Ins::bankFromBytes(bankLSB->value(), bankMSB->value()));
    bank->blockSignals(false);

    bankLabel->setEnabled
        (TSE3::Ins::bankToLSB(bank->value()) == bankLSB->value()
         && TSE3::Ins::bankToMSB(bank->value()) == bankMSB->value());

    emit selected(program->value(), bankLSB->value(), bankMSB->value());
}


void VoiceByNumberPopup::slotChangedBank()
{
    bankLSB->blockSignals(true);
    bankMSB->blockSignals(true);
    bankLSB->setValue(TSE3::Ins::bankToLSB(bank->value()));
    bankMSB->setValue(TSE3::Ins::bankToMSB(bank->value()));
    bankLSB->blockSignals(false);
    bankMSB->blockSignals(false);

    emit selected(program->value(), bankLSB->value(), bankMSB->value());
}


/******************************************************************************
 * VoiceByNamePopup class
 *****************************************************************************/

namespace
{
    class VoiceListViewItem : public QListViewItem
    {
        public:
            VoiceListViewItem(QListViewItem *parent,
                              int prog, int bLSB, int bMSB,
                              const QString &s1,
                              const QString &s2,
                              const QString &s3)
             : QListViewItem(parent, s1, s2, s3),
               program(prog), bankLSB(bLSB), bankMSB(bMSB)
             {
             }
             int program;
             int bankLSB;
             int bankMSB;
    };
}

VoiceByNamePopup::VoiceByNamePopup(QWidget *parent, const char *name = 0)
: QPopupMenu(parent, name), channel(0), port(0)
{
    QWidget *item = new QWidget(this);

    QGridLayout *layout
        = new QGridLayout(item, 2, 4, KDialogBase::spacingHint());

    // Destination
    QLabel *label = new QLabel("Show voices for channel/port", item);
    label->setAlignment(AlignRight|AlignVCenter);
    layout->addWidget(label, 0, 0);
    channelWidget = new ChannelWidget(false, false, false, 0, item);
    layout->addWidget(channelWidget, 0, 1);
    label = new QLabel(":", item);
    layout->addWidget(label, 0, 2);
    portWidget = new PortWidget(true, false, false, false, 0, item);
    layout->addWidget(portWidget, 0, 3);

    // List
    listView = new KListView(item);
    layout->addMultiCellWidget(listView, 1, 1, 0, 3);
    listView->addColumn("Name");
    listView->addColumn("Bank LSB : Bank MSB");
    listView->addColumn("Program");
    listView->setRootIsDecorated(true);
    listView->setAllColumnsShowFocus(true);

    connect(listView, SIGNAL(selectionChanged()), SLOT(slotChanged()));

    updateList();

    insertItem(item);
}


void VoiceByNamePopup::updateList()
{
    listView->clear();

    int c = channel < 0 ? 0 : channel;
    int p = port    < 0 ? 0 : port;

    TSE3::Ins::Instrument *ins
        = Application::application()->destination()->channel(c, p);

    if (!ins) return;

    for (size_t bank = 0; bank < ins->numBanks(); bank++)
    {
        TSE3::Ins::PatchData *patch = ins->patch(bank);
        QString s = "None";
        if (ins->bank(bank) != -1)
        {
            s.sprintf("%d", ins->bank(bank));
        }
        QListViewItem *bankItem
            = new QListViewItem(listView,
                                 patch->title().c_str(),
                                 s,
                                 "");
        bankItem->setSelectable(false);
        int lsb = ins->bankLSB(bank), msb = ins->bankMSB(bank);
        for (size_t p = 128; p > 0; --p)
        {
            s.sprintf("%d: %s", p-1, patch->name(p-1).c_str());
            (void) new VoiceListViewItem(bankItem, p-1, lsb, msb, "", "", s);
        }
    }
}


void VoiceByNamePopup::slotDestination(int c, int p)
{
    channel = c;
    port    = p;
}


void VoiceByNamePopup::slotSelect(int p, int bLSB, int bMSB)
{
    program = p;
    bankLSB = bLSB;
    bankMSB = bMSB;

    bool b = listView->signalsBlocked();
    listView->blockSignals(true);
    listView->clearSelection();
    listView->blockSignals(b);
}


void VoiceByNamePopup::slotChanged()
{
    VoiceListViewItem *vlvi
        = static_cast<VoiceListViewItem*>(listView->selectedItem());
    program = vlvi->program;
    bankLSB = vlvi->bankLSB;
    bankMSB = vlvi->bankMSB;
    emit selected(program, bankLSB, bankMSB);
    hide();
}

