/*
 * @(#)misc/Streambuf.h 1.00 13 November 2000
 *
 * Copyright (c) Pete Goodliffe 2000 (pete@cthree.org)
 *
 * This file is part of anthem - the TSE3 sequencer.
 *
 * This program is modifiable/redistributable under the terms of the GNU
 * General Public License.
 *
 * You should have recieved a copy of the GNU General Public License along
 * with this program; see the file COPYING. If not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 0211-1307, USA.
 */

#ifndef ANTHEM_STREAMBUF_H
#define ANTHEM_STREAMBUF_H

#include <iostream>

#include <qobject.h>

class QString;
class QTextView;
class QTimer;

/******************************************************************************
 * QString streambuf
 *****************************************************************************/

/**
 * A streambuf that takes the input from a C++ ostream, and places it into
 * a Qt @ref QString object.
 *
 * @short   C++ streambuf that directs output to a QString
 * @author  Pete Goodliffe
 * @version 1.0
 */
class QStringStreambuf : public std::streambuf
{
    public:

        /**
         * Creates a QStringStreambuf that apends output to the
         * specified @ref QString.
         *
         * @param str @ref QString to append data to
         */
        QStringStreambuf(QString &str, int bsize = 0);
        ~QStringStreambuf();

    protected:

        /**
         * Called if the internal buffer of the QStringStreambuf is full
         * and some character should be placed in there.
         */
        int overflow(int);

        /**
         * Called when the external representation is to be brought into
         * synchronization with the buffer's representation.
         */
        int sync();

    private:

        void put_buffer(void);
        void put_char(int);
        QString &str;
};

/**
 * A class derived from C++ ostream that conveniently creates an ostream
 * using a @ref QStringStreambuf as its streambuf.
 *
 * @short   C++ ostream using a @ref QStringStreambuf
 * @author  Pete Goodliffe
 * @version 1.0
 */
class QStringOStream : public std::ostream
{
    public:
        QStringOStream(QString &str, int bsize = 0);
};

/******************************************************************************
 * QTextView streambuf
 *****************************************************************************/

/**
 * A streambuf that takes the input from a C++ ostream, and places it into
 * a Qt @ref QTextView object.
 *
 * @short   C++ streambuf that directs output to a QTextView
 * @author  Pete Goodliffe
 * @version 1.0
 */
class QTextViewStreambuf : public QObject,
                           public std::streambuf
{
        Q_OBJECT

    public:

        /**
         * Creates a QTextViewStreambuf that apends output to the
         * specified @ref QTextView.
         *
         * @param tv @ref QTextView to append data to
         */
        QTextViewStreambuf(QTextView &tv, int bsize = 0);
        virtual ~QTextViewStreambuf();

    protected:

        /**
         * Called if the internal buffer of the QTextViewStreambuf is full
         * and some character should be placed in there.
         */
        int overflow(int);

        /**
         * Called when the external representation is to be brought into
         * synchronization with the buffer's representation.
         */
        int sync();

    protected slots:

        void slotScrollDown();

    private:

        void put_buffer(void);
        void put_char(int);

        QTextView &tv;
        QTimer    *timer;
};

/**
 * A class derived from C++ ostream that conveniently creates an ostream
 * using a @ref QTextViewStreambuf as its streambuf.
 *
 * @short   C++ ostream using a @ref QTextViewStreambuf
 * @author  Pete Goodliffe
 * @version 1.0
 */
class QTextViewOStream : public std::ostream
{
    public:
        QTextViewOStream(QTextView &tv, int bsize = 0);
};

#endif
