/*
 * @(#)PhraseEditor.h 1.00 8 September 2000
 *
 * Copyright (c) Pete Goodliffe 2000 (pete@cthree.org)
 *
 * This file is part of anthem - the TSE3 sequencer.
 *
 * This program is modifiable/redistributable under the terms of the GNU
 * General Public License.
 *
 * You should have recieved a copy of the GNU General Public License along
 * with this program; see the file COPYING. If not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 0211-1307, USA.
 */

#ifndef ANTHEM_PHRASEEDITOR_H
#define ANTHEM_PHRASEEDITOR_H

#include <kdialog.h>

#include "tse3/Notifier.h"
#include "tse3/listen/PhraseEdit.h"
#include "tse3/listen/Phrase.h"
#include "tse3/listen/Transport.h"

/******************************************************************************
 * TSE3 forward declarations
 *****************************************************************************/
namespace TSE3
{
    class Song;
    class PhraseEdit;

    namespace Cmd
    {
        class CommandHistory;
    }
}

/******************************************************************************
 * KDE/Qt forward declarations
 *****************************************************************************/
class KToolBar;
class KComboBox;

/******************************************************************************
 * Anthem forward declataions
 *****************************************************************************/

class PhraseEditorBase;

/**
 * Plugins that create @ref PhraseEditor implementations implement this
 * abstract interface.
 *
 * @short   @ref PhraseEditorBase factory
 * @author  Pete Goodliffe
 * @version 1.0
 */
class PhraseEditorFactory
{
    public:

        virtual ~PhraseEditorFactory() {}

        /**
         * Returns a @ref QString describing this editor type.
         */
        virtual QString editorType() const = 0;

        /**
         * Creates a PhraseEditor of the relevant type.
         *
         * The PhraseEditor is created with new, it is the callers
         * responsibility to delete it (it should get deleted with the
         * parent automatically).
         *
         * The parameters are those passed to the PhraseEditorBase ctor.
         */
        virtual PhraseEditorBase *createEditor(TSE3::PhraseEdit *phraseEdit,
                                               KToolBar         *toolbar,
                                               QWidget          *parent) = 0;
};

/**
 * The abstract interface class that the various @ref PhraseEditor
 * implementation windows must implement.
 *
 * @short   Base @ref Phrase editor window
 * @author  Pete Goodliffe
 * @version 1.0
 */
class PhraseEditorBase : public QWidget
{
        Q_OBJECT

    public:

        /**
         * The @ref TSE3::PhraseEdit will not be deleted whilst this
         * object is in existance.
         *
         * @param phraseEdit @ref TSE3::PhraseEdit to work on
         * @param toolbar    Where to place/remove buttons to/from
         * @param parent     Frame to put editor in
         */
        PhraseEditorBase(TSE3::PhraseEdit *phraseEdit, KToolBar *toolbar,
                         QWidget *parent)
            : QWidget(parent), phraseEdit(phraseEdit), toolbar(toolbar) {}
        virtual ~PhraseEditorBase() = 0;

    protected:

        TSE3::PhraseEdit *phraseEdit;
        KToolBar         *toolbar;
};

/**
 * The PhraseEditow window is a main container window for the editing of
 * musical data in a @ref TSE3::PhraseEdit object. It manages the @ref
 * PhraseEdit object and allows you to switch the type of editor you are
 * using.
 *
 * To implement the different types of editor, there is a plugin widget
 * interface to implement. Plugin sections of code can register a new
 * PhraseEditor implementation with the static @ref addEditor() method.
 *
 * @short   Main @ref Phrase editor window
 * @author  Pete Goodliffe
 * @version 1.0
 */
class PhraseEditor : public KDialog,
                     public TSE3::Listener<TSE3::TransportListener>,
                     public TSE3::Listener<TSE3::PhraseListener>,
                     public TSE3::Listener<TSE3::PhraseEditListener>
{
        Q_OBJECT

    public:

        /**
         * enum type describing the sorts of PhraseEditor implementations
         * that exist.
         *
         * You can plug in more types, see @ref addEditor().
         */
        enum EditorType
        {
            ListEditor,
            PianoRollEditor,
            DrumEditor,
            StaveRollEditor
        };

        /**
         * Creates a PhraseEditor window so that the data in a particular
         * @ref TSE3::Phrase can be edited.
         *
         * This is a fairly sophisticated window that allows you to "plug in"
         * various different implementations, e.g. List Editor, Piano Roll
         * Editor, etc.
         *
         * @param song       The @ref TSE3::Song to insert the final created
         *                   @ref TSE3::Phrase (if any) into.
         *                   Don't delete this Song whilst the window is open,
         *                   or else.
         * @param history    Specify this if you want commands to be
         *                   placed on a history list.
         * @param phrase     The @ref TSE3::Phrase to base this on, or zero for
         *                   no @ref TSE3::Phrase.
         * @param editorType The type of editor to open with, or leave as
         *                   default for last selected type.
         */
        PhraseEditor(QWidget                   *parent,
                     TSE3::Song                *song,
                     TSE3::Cmd::CommandHistory *history    = 0,
                     TSE3::Phrase              *phrase     = 0,
                     int                        editorType = -1);

        /**
         * Dtor - the @ref PhraseEdit is deleted with the window.
         */
        ~PhraseEditor();

        /**
         * @reimplemented
         */
        virtual void Transport_Status(TSE3::Transport *src, int status);

        /**
         * @reimplemented
         */
        virtual void Phrase_Reparented(TSE3::Phrase *p);

        /**
         * @reimplemented
         */
        virtual void Phrase_TitleAltered(TSE3::Phrase *p);

        /**
         * @reimplemented
         */
        virtual void Notifier_Deleted(TSE3::Phrase *p);

        /**
         * @reimplemented
         */
        virtual void PhraseEdit_Modified(TSE3::PhraseEdit *, bool mod);

        /**********************************************************************
         * Accessing editor types added at run time
         *********************************************************************/

        /**
         * Returns the number of editors registered with Anthem.
         *
         * You are guaranteed at least the number of editors described in
         * the @ref EditorType enum.
         *
         * @return The number of editors in the system.
         */
        static int numEditors();

        /**
         * Returns a @ref QString with the name of the editor type with
         * this index.
         */
        static QString editorType(int index);

        /**
         * Adds a new editor type to the application.
         *
         * This editor type cannot be removed.
         *
         * The factory object passed should have been created with new, and
         * after calling the method is considered to be 'owned' by the
         * PhraseEditor class.
         *
         * @param  factory A factory that creates this type of editor.
         * @return The editor type index.
         */
        static int addEditor(PhraseEditorFactory *factory);

    protected slots:

        /**
         * A new editor type has been requested.
         */
        void slotEditorType(int editorType);

        /**
         * A toolbar button has been pressed.
         */
        void slotToolbar(int id);

        /**
         * Used when the edits are being accepted.
         */
        void slotNewPhraseDone(bool success, QString title,
                               bool replacePhrase, bool insertPart,
                               bool keepOpen);

    protected:

        /**
         * @reimplemented
         */
        virtual void closeEvent(QCloseEvent *e);

    private:

        /**
         * Changes the editor implementation.
         */
        void changeEditor(PhraseEditorBase *editor);

        /**
         * Sets the window caption.
         */
        void updateCaption();

        /**
         * Asks whether the user want to reject the changes, and then
         * reports this back in a boolean.
         */
        bool messageBoxReject();

        // IDs for toolbar icons
        enum Toolbar_Std       { Toolbar_Type, Toolbar_Play, Toolbar_Stop,
                                 Toolbar_Accept, Toolbar_Reject,
                                 Toolbar_Revert,
                                 Toolbar_NB};

        class PhraseEditorImpl *pimpl;

        // UI components
        KToolBar               *toolbar;

    public:

        static const int Toolbar_NoButtons = Toolbar_NB;
};

#endif
