/*
 * @(#)Panic.cpp 1.00 1 June 2000
 *
 * Copyright (c) Pete Goodliffe 2000 (pete@cthree.org)
 *
 * This file is part of anthem - the TSE3 sequencer.
 *
 * This program is modifiable/redistributable under the terms of the GNU
 * General Public License.
 *
 * You should have recieved a copy of the GNU General Public License along
 * with this program; see the file COPYING. If not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 0211-1307, USA.
 */

#include "settings/Panic.h"

#include "Application.h"
#include "tse3/kdeui/PortWidget.h"

#include <qlayout.h>
#include <qcheckbox.h>
#include <qspinbox.h>
#include <qlabel.h>

#include <kdialogbase.h>

#include "tse3/Panic.h"

/******************************************************************************
 * IdMaskWidget class
 *****************************************************************************/
IdMaskWidget::IdMaskWidget(int nobits, QWidget *parent, const char *name)
: QWidget(parent, name), _mask(~0)
{
    QHBoxLayout *layout = new QHBoxLayout(this);
    spin = new QSpinBox(0, nobits-1, 1, this);
    check = new QCheckBox(this);
    layout->add(spin);
    layout->add(check);

    connect(spin,  SIGNAL(valueChanged(int)), SLOT(slotSpin(int)));
    connect(check, SIGNAL(toggled(bool)),     SLOT(slotCheckBox()));

    check->setChecked(_mask & (1<<spin->value()));
}


void IdMaskWidget::setMask(unsigned int newMask)
{
    _mask = newMask;
    check->setChecked(_mask & (1<<spin->value()));
}


void IdMaskWidget::slotSpin(int)
{
    check->setChecked(_mask & (1<<spin->value()));
}


void IdMaskWidget::slotCheckBox()
{
    _mask &= ~(1<<spin->value());
    if (check->isChecked())
    {
        _mask |= 1<<spin->value();
    }
}


/******************************************************************************
 * PanicWidget class
 *****************************************************************************/

PanicWidget::PanicWidget(TSE3::Panic *p, QWidget *parent, const char *name)
: SettingsWidget(new SettingsImpl<TSE3::PanicListener>(p), parent, name), p(p)
{
    QVBoxLayout *layout = new QVBoxLayout(this, KDialogBase::spacingHint(), -1);

    QGridLayout *gl = new QGridLayout(layout, 8, 2);

    QLabel *label = new QLabel("Port", this);
    label->setMinimumSize(label->sizeHint());
    gl->addWidget(label, 0, 0);
    port = new PortWidget(false, false, true, false, 0, this);
    gl->addWidget(port, 0, 1);

    status = new QCheckBox("Status", this);
    gl->addWidget(status, 1, 0);

    connect(status, SIGNAL(toggled(bool)), SLOT(updateDisable()));

    midiReset = new QCheckBox("Midi reset", this);
    gl->addWidget(midiReset, 2, 0);
    gmReset = new QCheckBox("GM reset", this);
    gl->addWidget(gmReset, 2, 1);

    gsReset = new QCheckBox("GS reset", this);
    gl->addWidget(gsReset, 3, 0);
    xgReset = new QCheckBox("XG reset", this);
    gl->addWidget(xgReset, 4, 0);

    gsMask = new IdMaskWidget(32, this);
    xgMask = new IdMaskWidget(16, this);
    gl->addWidget(gsMask, 3, 1);
    gl->addWidget(xgMask, 4, 1);

    allNotesOff = new QCheckBox("All notes off", this);
    gl->addWidget(allNotesOff, 5, 0);
    allNotesOffManually = new QCheckBox("All notes off manually", this);
    gl->addWidget(allNotesOffManually, 5, 1);

    allModsOff = new QCheckBox("All mods off", this);
    gl->addWidget(allModsOff, 6, 0);
    allPitchOff = new QCheckBox("All pitch bend off", this);
    gl->addWidget(allPitchOff, 6, 1);

    allCtrlOff = new QCheckBox("All controllers off", this);
    gl->addWidget(allCtrlOff, 7, 0);
    liftSustain = new QCheckBox("List sustain", this);
    gl->addWidget(liftSustain, 7, 1);

    layout->addStretch(1);

    updateWidgets();
}


PanicWidget::~PanicWidget()
{
}


void PanicWidget::setValues()
{
    setValues(*p);
}


void PanicWidget::setValues(TSE3::Panic &p)
{
    // XXX we get called back for the alteration which muffs things up
    p.setPort(port->value());
    p.setStatus(status->isChecked());
    p.setMidiReset(midiReset->isChecked());
    p.setGmReset(gmReset->isChecked());
    p.setGsReset(gsReset->isChecked());
    p.setXgReset(xgReset->isChecked());
    p.setAllNotesOff(allNotesOff->isChecked());
    p.setAllNotesOffManually(allNotesOffManually->isChecked());
    p.setAllModsOff(allModsOff->isChecked());
    p.setAllPitchOff(allPitchOff->isChecked());
    p.setAllCtrlOff(allCtrlOff->isChecked());
    p.setLiftSustain(liftSustain->isChecked());

    unsigned int mask = gsMask->mask();
    for (size_t n = 0; n < 32; n++)
    {
        p.setGsIDMask(n, mask & (1<<n));
    }
    mask = xgMask->mask();
    for (size_t n = 0; n < 16; n++)
    {
        p.setXgIDMask(n, mask & (1<<n));
    }
}


void PanicWidget::updateWidgets()
{
    port->setValue(p->port());
    status->setChecked(p->status());
    midiReset->setChecked(p->midiReset());
    gmReset->setChecked(p->gmReset());
    gsReset->setChecked(p->gsReset());
    xgReset->setChecked(p->xgReset());
    allNotesOff->setChecked(p->allNotesOff());
    allNotesOffManually->setChecked(p->allNotesOffManually());
    allModsOff->setChecked(p->allModsOff());
    allPitchOff->setChecked(p->allPitchOff());
    allCtrlOff->setChecked(p->allCtrlOff());
    liftSustain->setChecked(p->liftSustain());
    updateDisable();

    unsigned int mask = 0;
    for (size_t n = 0; n < 32; n++)
    {
        if (p->gsIDMask(n)) mask += 1<<n;
    }
    gsMask->setMask(mask);
    mask = 0;
    for (size_t n = 0; n < 16; n++)
    {
        if (p->xgIDMask(n)) mask += 1<<n;
    }
    xgMask->setMask(mask);
}


void PanicWidget::updateDisable()
{
    bool enabled = status->isChecked();
    midiReset->setEnabled(enabled);
    gmReset->setEnabled(enabled);
    gsReset->setEnabled(enabled);
    xgReset->setEnabled(enabled);
    gsMask->setEnabled(enabled);
    xgMask->setEnabled(enabled);
    allNotesOff->setEnabled(enabled);
    allNotesOffManually->setEnabled(enabled);
    allModsOff->setEnabled(enabled);
    allPitchOff->setEnabled(enabled);
    allCtrlOff->setEnabled(enabled);
    liftSustain->setEnabled(enabled);
}

