/*
 * @(#)PartView.cpp 1.00 15 November 2000
 *
 * Copyright (c) Pete Goodliffe 2000 (pete@cthree.org)
 *
 * This file is part of anthem - the TSE3 sequencer.
 *
 * This program is modifiable/redistributable under the terms of the GNU
 * General Public License.
 *
 * You should have recieved a copy of the GNU General Public License along
 * with this program; see the file COPYING. If not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 0211-1307, USA.
 */

#include "settings/PartView.h"

#include "songwindow/SongWindow.h"
#include "songwindow/TimeLine.h"

#include <list>

#include <qlayout.h>
#include <qcheckbox.h>

#include <kdialogbase.h>
#include <kcombobox.h>

/******************************************************************************
 * SongWindowList class
 *****************************************************************************/

namespace
{
    std::list<SongWindow*> sw_list;
}


SongWindowList::SongWindowList()
: _followCursor(true)
{
}


SongWindowList *SongWindowList::songWindowList()
{
    static SongWindowList swl;
    return &swl;
}


void SongWindowList::add(SongWindow *sw)
{
    sw_list.push_back(sw);
}


void SongWindowList::remove(SongWindow *sw)
{
   sw_list.remove(sw);
}


void SongWindowList::setPVViewOptions(const PartView::ViewOptions &o)
{
    pvoptions = o;
    std::list<SongWindow*>::iterator i = sw_list.begin();
    while (i != sw_list.end())
    {
        (*i)->childPartView()->setViewOptions(pvoptions);
        i++;
    }
}


void SongWindowList::setSWViewOptions(const SongWindow::ViewOptions &o)
{
    swoptions = o;
    std::list<SongWindow*>::iterator i = sw_list.begin();
    while (i != sw_list.end())
    {
        (*i)->setViewOptions(swoptions);
        i++;
    }
}


void SongWindowList::setFollowCursor(bool fc)
{
    _followCursor = fc;
    std::list<SongWindow*>::iterator i = sw_list.begin();
    while (i != sw_list.end())
    {
        (*i)->childTimeLine()->setFollowCursor(fc);
        i++;
    }
}


/******************************************************************************
 * PartWidgetView class
 *****************************************************************************/

PartViewWidget::PartViewWidget(QWidget *parent, const char *name)
: SettingsWidget(0, parent, name)
{
    QVBoxLayout *layout = new QVBoxLayout(this, KDialogBase::spacingHint(), -1);
    QGridLayout *gl = new QGridLayout(layout, 6, 2);

    verticalRules = new QCheckBox("Show vertical rules", this);
    gl->addWidget(verticalRules, 0, 0);

    phraseName = new QCheckBox("Show phrase names", this);
    gl->addWidget(phraseName, 0, 1);

    phraseEvents = new QCheckBox("Show phrase events", this);
    gl->addWidget(phraseEvents, 1, 0);

    plainBoxes = new QCheckBox("Draw plain boxes", this);
    gl->addWidget(plainBoxes, 1, 1);

    colours = new KComboBox(this);
    colours->setInsertionPolicy(QComboBox::NoInsertion);
    colours->insertItem("Don't display colours");
    colours->insertItem("Show colour in text foreground");
    colours->insertItem("Show colour in text background");
    colours->insertItem("Show colour in part box");
    colours->insertItem("Show colour as a dot");
    gl->addWidget(colours, 2, 0);

    fillBoxes = new QCheckBox("Draw filled boxes", this);
    gl->addWidget(fillBoxes, 2, 1);

    followCursor = new QCheckBox("Window follows cursor", this);
    gl->addWidget(followCursor, 3, 0);

    lmbSetsClock = new QCheckBox("Left mouse button sets clock", this);
    gl->addWidget(lmbSetsClock, 4, 0);

    rmbSetsClock = new QCheckBox("Right mouse button sets clock", this);
    gl->addWidget(rmbSetsClock, 4, 1);

    dragSetsMarkers = new QCheckBox("Drag sets markers", this);
    gl->addWidget(dragSetsMarkers, 5, 0);

    gl->setRowStretch(6, 1);

    updateWidgets();
}


PartViewWidget::~PartViewWidget()
{
}


void PartViewWidget::setValues()
{
    PartView::ViewOptions options
        = SongWindowList::songWindowList()->pvViewOptions();
    options.verticalRules   = verticalRules->isChecked();
    options.phraseName      = phraseName->isChecked();
    options.phraseEvents    = phraseEvents->isChecked();
    options.plainBoxes      = plainBoxes->isChecked();
    options.fillBoxes       = fillBoxes->isChecked();
    options.colours         = (PartView::ViewOptions::ColourOptions)
                              colours->currentItem();
    options.dragSetsMarkers = dragSetsMarkers->isChecked();
    SongWindowList::songWindowList()->setPVViewOptions(options);

    SongWindowList::songWindowList()->setFollowCursor
        (followCursor->isChecked());

    SongWindow::ViewOptions swvo
        = SongWindowList::songWindowList()->swViewOptions();
    swvo.lmbSetsClock = lmbSetsClock->isChecked();
    swvo.rmbSetsClock = rmbSetsClock->isChecked();
    SongWindowList::songWindowList()->setSWViewOptions(swvo);
}


void PartViewWidget::updateWidgets()
{
    const PartView::ViewOptions &options
        = SongWindowList::songWindowList()->pvViewOptions();
    verticalRules->setChecked(options.verticalRules);
    phraseName->setChecked(options.phraseName);
    phraseEvents->setChecked(options.phraseEvents);
    plainBoxes->setChecked(options.plainBoxes);
    fillBoxes->setChecked(options.fillBoxes);
    colours->setCurrentItem(options.colours);
    dragSetsMarkers->setChecked(options.dragSetsMarkers);

    followCursor->setChecked(SongWindowList::songWindowList()->followCursor());

    const SongWindow::ViewOptions &swvo
        = SongWindowList::songWindowList()->swViewOptions();
    lmbSetsClock->setChecked(swvo.lmbSetsClock);
    rmbSetsClock->setChecked(swvo.rmbSetsClock);
}

