/*
 * @(#)Settings.cpp 1.00 14 June 2000
 *
 * Copyright (c) Pete Goodliffe 2000 (pete@cthree.org)
 *
 * This file is part of anthem - the TSE3 sequencer.
 *
 * This program is modifiable/redistributable under the terms of the GNU
 * General Public License.
 *
 * You should have recieved a copy of the GNU General Public License along
 * with this program; see the file COPYING. If not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 0211-1307, USA.
 */

#include "settings/Settings.h"

#include "settings/DisplayParams.h"
#include "settings/Instrument.h"
#include "settings/Metronome.h"
#include "settings/MidiEcho.h"
#include "settings/MidiFilter.h"
#include "settings/MidiParams.h"
#include "settings/OSSMidiScheduler.h"
#include "settings/Panic.h"
#include "settings/Part.h"
#include "settings/Track.h"
#include "settings/Transport.h"
#include "settings/PartView.h"
#include "settings/PresetColours.h"
#include "settings/VuWindow.h"
#include "settings/App.h"
#include "settings/Plugins.h"

#include "Application.h"

#include "misc/kde2-compat.h"
#include <qlayout.h>
#include <qtabwidget.h>
#include <qvbox.h>
#include <kiconloader.h>

#include "tse3/Transport.h"

/******************************************************************************
 * SettingsWindow class
 *****************************************************************************/

SettingsWindow::SettingsWindow(QWidget *parent)
: KDialogBase(IconList, "Settings", Cancel|Ok|Apply|User1|User2, Ok, parent,
              "SettingsWindow", false, true,
              KGuiItem("Reset"), KGuiItem("Save"))
{
    /**************************************************************************
     * Set up tabs
     *************************************************************************/

    QVBox *sequencerPage
        = addVBoxPage(QString("Sequencer"),
                      "Configure TSE3 sequencer settings",
                      BarIcon("multimedia"));
    QVBox *windowsPage
        = addVBoxPage(QString("Windows"),
                      "Configure Anthem windows", BarIcon("winprops"));
    QVBox *applicationPage
        = addVBoxPage(QString("Application"),
                      "Configure general application settings",
                      BarIcon("misc"));
    QVBox *instrumentsPage
        = addVBoxPage(QString("Instruments"),
                      "Configure MIDI instrument definitions",
                      BarIcon("kcmmidi"));

    QTabWidget *sequencerTab   = new QTabWidget(sequencerPage);
    QTabWidget *windowsTab     = new QTabWidget(windowsPage);
    QTabWidget *applicationTab = new QTabWidget(applicationPage);
    QTabWidget *instrumentsTab = new QTabWidget(instrumentsPage);

    mwidget = new MetronomeWidget(Application::application()->metronome(),
                                  sequencerTab);
    sequencerTab->addTab(mwidget, "Metronome");

    ewidget = new MidiEchoWidget(Application::application()->transport()
                                     ->midiEcho(),
                                 sequencerTab);
    sequencerTab->addTab(ewidget, "MIDI echo");

    spwidget = new PanicWidget(Application::application()->transport()
                                   ->startPanic(),
                               sequencerTab);
    sequencerTab->addTab(spwidget, "Start panic");

    epwidget = new PanicWidget(Application::application()->transport()
                                   ->endPanic(),
                               sequencerTab);
    sequencerTab->addTab(epwidget, "End panic");

    fwidget
        = new MidiFilterWidget(
            Application::application()->transport()->filter(),
            sequencerTab);
    sequencerTab->addTab(fwidget, "Global MIDI Filter");

    twidget = new TransportWidget(sequencerTab);
    sequencerTab->addTab(twidget, "Transport");

    mswidget = new OSSMidiSchedulerWidget(sequencerTab);
    sequencerTab->addTab(mswidget, "MIDI Interfaces");

    pvwidget = new PartViewWidget(windowsTab);
    windowsTab->addTab(pvwidget, "Part view");

    pcwidget = new PresetColoursWidget(windowsTab);
    windowsTab->addTab(pcwidget, "Preset colours");

    vuwidget = new VuWindowWidget(windowsTab);
    windowsTab->addTab(vuwidget, "VU bars");

    awidget = new ApplicationWidget(applicationTab);
    applicationTab->addTab(awidget, "Application");

    plugwidget = new PluginsWidget(applicationTab);
    applicationTab->addTab(plugwidget, "Plugins");

    inswidget = new InstrumentWidget(instrumentsTab);
    instrumentsTab->addTab(inswidget, "Instruments");
}


SettingsWindow::~SettingsWindow()
{
}


void SettingsWindow::show()
{
    updateWidgets();
    KDialogBase::show();
}


void SettingsWindow::accept()
{
    setValues();
    done(Accepted);
}


void SettingsWindow::setValues()
{
    mwidget->accept();
    ewidget->accept();
    spwidget->accept();
    epwidget->accept();
    fwidget->accept();
    twidget->accept();
    mswidget->accept();
    pvwidget->accept();
    pcwidget->accept();
    vuwidget->accept();
    awidget->accept();
    plugwidget->accept();
    inswidget->accept();
}


void SettingsWindow::done(int reason)
{
    hide();
}


void SettingsWindow::slotApply()
{
    setValues();
    updateWidgets();
}


void SettingsWindow::slotUser1()
{
    updateWidgets();
}


void SettingsWindow::slotUser2()
{
    setValues();
    updateWidgets();
    Application::application()->saveChoices();
}


void SettingsWindow::updateWidgets()
{
    mwidget->updateWidgets();
    ewidget->updateWidgets();
    spwidget->updateWidgets();
    epwidget->updateWidgets();
    fwidget->updateWidgets();
    twidget->updateWidgets();
    mswidget->updateWidgets();
    pvwidget->updateWidgets();
    pcwidget->updateWidgets();
    vuwidget->updateWidgets();
    awidget->updateWidgets();
    plugwidget->updateWidgets();
    inswidget->updateWidgets();
}
