/*
 * @(#)SongWindow_PartView.cpp 1.00 22 May 2000
 *
 * Copyright (c) Pete Goodliffe 2000 (pete@cthree.org)
 *
 * This file is part of anthem - the TSE3 sequencer.
 *
 * This program is modifiable/redistributable under the terms of the GNU
 * General Public License.
 *
 * You should have recieved a copy of the GNU General Public License along
 * with this program; see the file COPYING. If not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 0211-1307, USA.
 */

#include "songwindow/PartView.h"

#include "songwindow/TimeLine.h"
#include "songwindow/PhraseList.h"
#include "dialogues/Part.h"
#include "Application.h"
#include "phraseeditors/PhraseEditor.h"
#include "misc/Clipboard.h"
#include "gadgets/ZoomWidget.h"
#include "settings/PartView.h"

#include <kiconloader.h>
#include <kglobalsettings.h>
#include <kmessagebox.h>

#include "misc/kde2-compat.h"
#include <qtoolbutton.h>
#include <qlayout.h>
#include <qpopupmenu.h>
#include <qpainter.h>
#include <qpen.h>
#include <qstyle.h>

#include "tse3/Song.h"
#include "tse3/Track.h"
#include "tse3/Part.h"
#include "tse3/Error.h"
#include "tse3/app/PartSelection.h"
#include "tse3/app/TrackSelection.h"
#include "tse3/DisplayParams.h"
#include "tse3/app/PartDisplay.h"
#include "tse3/cmd/CommandHistory.h"
#include "tse3/cmd/Track.h"
#include "tse3/cmd/Part.h"
#include "tse3/cmd/CommandGroup.h"
#include "tse3/Phrase.h"
#include "tse3/util/Track.h"
#include "tse3/Transport.h"

namespace
{
    /**
     * The number of pixels at the start/end of a Part that allows you to
     * drag the start/end time of that Part.
     */
    const int PART_STARTEND_DRAG_AREA = 10;

    /**
     * Lighten @p colour by a factor of @p factor / 5.
     * Carefully ensure the 0..255 value stays in range.
     */
    inline int lighten(int colour, int factor)
    {
        return colour + (255-colour)*factor/5;
    }

    /**
     * Darken @p colour by a factor of @p factor / 5
     * Carefully ensure the 0..255 value stays in range.
     */
    inline int darken(int colour, int factor)
    {
        return colour - (colour)*factor/5;
    }
}

/******************************************************************************
 * PartView class
 *****************************************************************************/

PartView::PartView(TSE3::Song *song, QWidget *parent, TimeLine *timeLine,
                   TSE3::Util::Snap &snap,
                   TSE3::App::PartSelection *ps,
                   TSE3::App::TrackSelection *ts)
: QScrollView(parent, "PartView"), rubber(new RubberSelectionBox(this)),
  song(song), partSelection(ps), trackSelection(ts),
  timeLine(timeLine),
  _pointerMode(Select), _partInsertAction(TSE3::Cmd::Part_Move::NoOverlap),
  hZoomIn(0), hZoomOut(0), vZoomIn(0), vZoomOut(0),
  dragType(DragNone), phraseListWindow(0),
  _viewOptions(SongWindowList::songWindowList()->pvViewOptions()),
  snap(snap), rmbMenu(0)
{
    TSE3::Listener<TSE3::App::PartSelectionListener>::attachTo(partSelection);
    TSE3::Listener<TSE3::SongListener>::attachTo(song);
    TSE3::Listener<TSE3::PresetColoursListener>::attachTo
        (Application::application()->presetColours());

    // UI construction

    enableClipper(true);
    QPixmap back = UserIcon("tile");
    viewport()->setBackgroundPixmap(back);
    setHScrollBarMode(AlwaysOn);
    setVScrollBarMode(AlwaysOn);
    resizeContents(800, 800); // XXX

    // Create zoom controls

    QToolButton *zoom = new QToolButton(this);
    zoom->setPixmap(UserIcon("scroll_zoominout"));
    setCornerWidget(zoom);
    connect(zoom, SIGNAL(clicked()), SIGNAL(zoomWidget()));

    hZoomIn  = new QToolButton(this);
    hZoomOut = new QToolButton(this);
    vZoomIn  = new QToolButton(this);
    vZoomOut = new QToolButton(this);

    hZoomIn->setPixmap(UserIcon("scroll_zoomin"));
    vZoomIn->setPixmap(UserIcon("scroll_zoomin"));
    hZoomOut->setPixmap(UserIcon("scroll_zoomout"));
    vZoomOut->setPixmap(UserIcon("scroll_zoomout"));

    hZoomIn->setAutoRepeat(true);
    hZoomOut->setAutoRepeat(true);
    vZoomIn->setAutoRepeat(true);
    vZoomOut->setAutoRepeat(true);

    connect(hZoomIn,  SIGNAL(clicked()), SIGNAL(zoomInHorizontal()));
    connect(hZoomOut, SIGNAL(clicked()), SIGNAL(zoomOutHorizontal()));
    connect(vZoomIn,  SIGNAL(clicked()), SIGNAL(zoomInVertical()));
    connect(vZoomOut, SIGNAL(clicked()), SIGNAL(zoomOutVertical()));

    connect(rubber, SIGNAL(dragBoxCompleted(int,int,int,int,int,bool)),
            SLOT(slotDragBoxCompleted(int,int,int,int,int,bool)));

    rubber->filter(this);
    viewport()->setFocusPolicy(StrongFocus);
    viewport()->setFocusProxy(this); // This is a disgusting hack that prevents
                                     // the viewport flickering every time
                                     // focus is lost/gained

    for (size_t n = 0; n < song->size(); n++)
    {
        attachToTrack((*song)[n]);
    }

    lastFrom = song->from();
    lastTo   = song->to();
}


PartView::~PartView()
{
    delete rubber;
}


void PartView::changeSong(TSE3::Song *s)
{
    // 1. Out with the old
    TSE3::Listener<TSE3::SongListener>::detachFrom(song);
    for (size_t n = 0; n < song->size(); n++)
    {
        TSE3::Track * const track = (*song)[n];
        TSE3::Listener<TSE3::TrackListener>::detachFrom(track);
        for (size_t p = 0; p < track->size(); p++)
        {
            TSE3::Listener<TSE3::PartListener>::detachFrom((*track)[p]);
        }
    }

    // 2. In with the new
    song = s;
    TSE3::Listener<TSE3::SongListener>::attachTo(song);
    for (size_t n = 0; n < song->size(); n++)
    {
        attachToTrack((*song)[n]);
    }
    resizeContents(contentsWidth(), itemHeight*song->size());
    lastFrom = song->from();
    lastTo   = song->to();
    update(0, 0);
}


/******************************************************************************
 * Window setup
 *****************************************************************************/

void PartView::setRMBMenu(QPopupMenu *menu)
{
    rmbMenu = menu;
}


void PartView::slotSetPointerMode(PointerMode pm)
{
    if (pm != _pointerMode)
    {
        partSelection->clear();
        _pointerMode = pm;
        emit pointerModeChanged();
    }
}


void PartView::slotSetPartInsertAction(int action)
{
    if (action != _partInsertAction)
    {
        _partInsertAction = action;
        emit partInsertActionChanged();
    }
}


/******************************************************************************
 * Drawing the viewport contents
 *****************************************************************************/

void PartView::drawContents(QPainter *p,
                            int clipx, int clipy, int clipw, int cliph)
{
    TSE3::Clock start = timeLine->pixelToClock(clipx);
    TSE3::Clock end   = timeLine->pixelToClock(clipx+clipw);
    size_t      mintrk = trackAt(clipy);
    size_t      maxtrk = trackAt(clipy+cliph);

    rubber->draw(p);

    // Draw the background things

    // Big bar lines
    if (_viewOptions.verticalRules)
    {
        TSE3::Clock min = start, max = end;
        timeLine->snapRange(min, max);
        QPen pen(p->pen());
        pen.setColor(QColor(0x66, 0x66, 0x66));
        pen.setStyle(DotLine);
        p->setPen(pen);
        while (min <= max)
        {
            if (timeLine->isBig(min))
            {
                int x = timeLine->clockToPixel(min);
                p->drawLine(x, 0, x, contentsHeight());
            }
            min = timeLine->next(min);
        }
    }

    // Repeat markers
    QPen pen(Qt::blue);
    pen.setStyle(DotLine);
    p->setPen(pen);
    int fromPos = timeLine->clockToPixel(song->from());
    int toPos   = timeLine->clockToPixel(song->to());
    p->drawLine(fromPos, 0, fromPos, contentsHeight());
    p->drawLine(toPos, 0, toPos, contentsHeight());


    // Draw the Track contents
    for (size_t trk = mintrk; trk <= maxtrk; trk++)
    {
        for (size_t prt = (*song)[trk]->index(start);
             prt < (*song)[trk]->size()
                 && (*(*song)[trk])[prt]->start() < end;
             prt++)
        {
            drawPart(p, (*(*song)[trk])[prt]);
        }
    }

    rubber->draw(p);
}


void PartView::drawPart(QPainter *p, TSE3::Part *part)
{
    int    startx = timeLine->clockToPixel(part->start());
    int    endx   = timeLine->clockToPixel(part->end());
    bool   sunken = partSelection->isSelected(part);
    size_t trk    = song->index(part->parent());

    TSE3::App::PartDisplay disp
        (part, Application::application()->presetColours());

    // Choose the colour
    int r = 255, g = 255, b = 255;
    disp.colour(r, g, b);

    // Draw the Part rectangle (in the appropriate style)

    if (_viewOptions.plainBoxes)
    {

        if(_viewOptions.fillBoxes)
        {
            QColor c(255, 255, 255);
            if (disp.useColour()
                && _viewOptions.colours == ViewOptions::Colour_Box)
            {
                c = QColor(r, g, b);
            }
            if (sunken)
            {
                int r, g, b;
                c.rgb(&r, &g, &b);
                c = QColor(darken(r,2), darken(g,2), darken(b,2));
            }
            p->fillRect(startx, itemHeight*trk,
                        endx - startx-1, itemHeight-1, c);
        }

        if (sunken)
        {
            p->setPen(Qt::white);
        }
        else
        {
            p->setPen(Qt::black);
        }
        p->drawRect(startx, itemHeight*trk,
                    endx - startx-1, itemHeight-1);
        p->moveTo(startx+1, itemHeight*(trk+1)-1);
        p->setPen(QColor(0x50,0x50,0x50));
        p->lineTo(endx-1, itemHeight*(trk+1)-1);
        p->lineTo(endx-1, itemHeight*trk+1);
    }
    else
    {
        QColorGroup cg = colorGroup();
        if (disp.useColour() && _viewOptions.colours == ViewOptions::Colour_Box)
        {
            cg.setColor(QColorGroup::Background, QColor(r, g, b));
            QColor light(lighten(r,3), lighten(g,3), lighten(b,3));
            QColor midlight(lighten(r,2), lighten(g,2), lighten(b,2));
            QColor dark(darken(r,3), darken(g,3), darken(b,3));
            QColor mid(lighten(r,2), lighten(g,2), lighten(b,2));
            cg.setColor(QColorGroup::Light,    light);
            cg.setColor(QColorGroup::Midlight, midlight);
            cg.setColor(QColorGroup::Mid,      mid);
            cg.setColor(QColorGroup::Dark,     dark);
            cg.setColor(QColorGroup::Shadow,   dark);
        }

        if (sunken)
        {
            for (int n = QColorGroup::Foreground;
                 n < QColorGroup::NColorRoles;
                 n++)
            {
                int r, b, g;
                cg.color(QColorGroup::ColorRole(n)).rgb(&r, &g, &b);
                cg.setColor(QColorGroup::ColorRole(n),
                            QColor(darken(r,2), darken(g,2), darken(b,2)));
            }
        }
        const QBrush *brush = _viewOptions.fillBoxes
                            ? &(cg.brush(QColorGroup::Background))
                            : 0;
#if KDE_VERSION < 300
        style().drawPanel(p,
                          startx, itemHeight*trk,
                          endx - startx, itemHeight,
                          cg, sunken, 2, brush);
#else
        p->setBrush(*brush);
        style().drawPrimitive(QStyle::PE_PanelPopup, p,
                              QRect(startx, itemHeight*trk,
                                  endx - startx, itemHeight),
                              cg, sunken?QStyle::Style_Sunken:QStyle::Style_Raised); // XXX complete me KDE3
#endif
    }

    // Draw coloured dot

    int dot_offset = 4;

    if (disp.useColour() && _viewOptions.colours == ViewOptions::Colour_Dot)
    {
        int dotareawidth = (endx-startx > itemHeight)
                         ? itemHeight
                         : endx-startx;
        int diameter     = dotareawidth * 6 / 10;
        int notdiameter  = dotareawidth - diameter;

        p->setPen(QPen(QColor(r, g, b)));
        p->setBrush(QBrush(QColor(r, g, b)));
        p->drawEllipse(startx + notdiameter/2, itemHeight*trk + notdiameter/2,
                       diameter, diameter);
        p->setBrush(QBrush());

        dot_offset = dotareawidth;
    }

    // Draw Phrase events

    if (part->phrase() && _viewOptions.phraseEvents)
    {
        // Events
        if (disp.useColour()
            && _viewOptions.colours != ViewOptions::Colour_None
            && _viewOptions.colours != ViewOptions::Colour_Box)
        {
            p->setPen(QPen(QColor(r, g, b)));
        }
        else
        {
            p->setPen(QPen(QColor(0x50, 0x50, 0x50)));
        }

        TSE3::PlayableIterator *i = part->iterator(0);
        while (i->more() && (**i).time < part->end()-part->start())
        {
            TSE3::MidiEvent e = **i;
            if (e.data.status == TSE3::MidiCommand_NoteOn)
            {
                int ev_startx
                    = timeLine->clockToPixel(part->start()+e.time);
                int ev_endx
                    = timeLine->clockToPixel(part->start()+e.offTime);
                if (ev_endx > endx) ev_endx = endx;
                int y = itemHeight*trk
                      + (127-e.data.data1)*itemHeight/127;
                p->drawLine(ev_startx, y, ev_endx, y);
            }
            ++(*i);
        }
        delete i;
    }

    // Draw Phrase title

    if (part->phrase() && _viewOptions.phraseName)
    {
        // Text
        const int x  = startx + dot_offset;
        const int y  = itemHeight*trk;
        const int w  = endx-x;
        const int h  = itemHeight;
        const int tf = AlignLeft | AlignVCenter | SingleLine;

        QFont f = p->font();
        if (sunken)
        {
            QFont f2(f);
            f2.setBold(true);
            p->setFont(f2);
        }

        if (disp.useColour()
            && _viewOptions.colours == ViewOptions::Colour_TextBackground)
        {
            QRect rect
                = p->boundingRect(x, y, w, h, tf,
                                  part->phrase()->title().c_str());
            p->fillRect(rect, QColor(r, g, b));
        }

        if (disp.useColour()
            && _viewOptions.colours == ViewOptions::Colour_TextForeground)
        {
            p->setPen(QPen(QColor(r, b, g)));
        }
        else
        {
            p->setPen(QPen(QColor(0x0, 0x0, 0x0)));
        }

        p->drawText(x, y, w, h, tf,
                    part->phrase()->title().c_str());

        if (sunken)
        {
            p->setFont(f);
        }
    }
}


void PartView::redraw(int minTrack, TSE3::Clock minTime,
                      int maxTrack, TSE3::Clock maxTime)
{
    int x = timeLine->clockToPixel(minTime);
    int y = minTrack*itemHeight;
    int w = (maxTime == -1)
          ? contentsWidth()
          : timeLine->clockToPixel(maxTime) - x;
    int h = (maxTrack == -1)
          ? contentsHeight()
          : (maxTrack+1)*itemHeight - y;
    repaintContents(x, y, w, h);
}


void PartView::update(int minTrack, TSE3::Clock minTime,
                      int maxTrack, TSE3::Clock maxTime)
{
    int x = timeLine->clockToPixel(minTime);
    int y = minTrack*itemHeight;
    int w = (maxTime == -1)
          ? contentsWidth()                 - x
          : timeLine->clockToPixel(maxTime) - x;
    int h = (maxTrack == -1)
          ? contentsHeight()        - y
          : (maxTrack+1)*itemHeight - y;
    updateContents(x, y, w, h);
}


void PartView::updateSelection()
{
    TSE3::App::PartSelection::iterator_t i = partSelection->begin();
    while (i != partSelection->end())
    {
        int index = song->index((*i)->parent());
        update(index, (*i)->start(), index, (*i)->end());
        i++;
    }
}


void PartView::updatePhrase(TSE3::Phrase *phrase)
{
    for (size_t trk = 0; trk < song->size(); trk++)
    {
        for (size_t prt = 0; prt < (*song)[trk]->size(); prt++)
        {
            if ((*(*song)[trk])[prt]->phrase() == phrase)
            {
                updatePart((*(*song)[trk])[prt]);
            }
        }
    }
}


void PartView::updatePart(TSE3::Part *part)
{
    update(song->index(part->parent()), part->start(),
           song->index(part->parent()), part->end());
}


/******************************************************************************
 * Handling the mouse
 *****************************************************************************/

void PartView::contentsMousePressEvent(QMouseEvent *e)
{
    if (Application::application()->transport()->status()
            == TSE3::Transport::Resting)
    {
        emit setClock(timeLine->pixelToClock(e->x()), e->button());
    }

    if (e->button() == RightButton && rmbMenu)
    {
        emit rmbMenuOpened(trackAt(e->y()), partAt(e->x(), e->y()));
        rmbMenu->popup(viewport()->mapToGlobal(contentsToViewport(e->pos())));
        return;
    }

    lastMousePressPos = e->pos();

    switch (_pointerMode)
    {
        case Select:
        {
            selectMousePressEvent(e);
            break;
        }
        case Snip:
        {
            snipMousePressEvent(e);
            break;
        }
        case Glue:
        {
            glueMousePressEvent(e);
            break;
        }
        case Insert:
        {
            // does nothing
        }
    }
}


void PartView::selectMousePressEvent(QMouseEvent *e)
{
    TSE3::Part *part = partAt(e->x(), e->y());
    if (part)
    {
        // redraw all old Parts
        bool add = e->state() & ControlButton;

        partSelection->select(part, add);
    }
    else
    {
        // redraw all old Parts
        partSelection->clear();
        trackSelection->clear();
    }

    emit partSelected(part);
}


void PartView::snipMousePressEvent(QMouseEvent *e)
{
    TSE3::Clock x = snap(timeLine->pixelToClock(e->x()));
    TSE3::Part *part = partAt(e->x(), e->y());
    if (part)
    {
        TSE3::Cmd::Track_Snip *command
            = new TSE3::Cmd::Track_Snip((*song)[trackAt(e->y())], x);
        command->execute();
        Application::application()->history(song)->add(command);
    }
}


void PartView::glueMousePressEvent(QMouseEvent *e)
{
    if (TSE3::Cmd::Track_Glue::valid((*song)[trackAt(e->y())],
                                     timeLine->pixelToClock(e->x())))
    {
        TSE3::Cmd::Track_Glue *command
            = new TSE3::Cmd::Track_Glue((*song)[trackAt(e->y())],
                                        timeLine->pixelToClock(e->x()));
        command->execute();
        Application::application()->history(song)->add(command);
    }
}


void PartView::contentsMouseReleaseEvent(QMouseEvent *e)
{
    rubber->handleMouseReleaseEvent(e);
}


void PartView::contentsMouseMoveEvent(QMouseEvent *e)
{
    if (rubber->dragging())
    {
        rubber->handleMouseMoveEvent(e);
        if (_viewOptions.dragSetsMarkers)
        {
            song->setTo(snap(timeLine->pixelToClock(e->x())));
        }
        return;
    }

    // Work out whether a drag has been started

    if (!(e->state() & LeftButton) || rubber->dragging()) return;

    QPoint length  = e->pos() - lastMousePressPos;
    int    mlength = length.manhattanLength();
    int    delay   = KGlobalSettings::dndEventDelay();

    if (mlength > delay)
    {
        // Some generally useful values
        int    left    = timeLine->clockToPixel(0);
        int    right   = contentsWidth();
        size_t trackno = trackAt(e->y());

        // A drag has been started
        switch (_pointerMode)
        {
            case Select:
            {
                TSE3::Part *part = partAt(lastMousePressPos.x(),
                                          lastMousePressPos.y());
                if (!part)
                {
                    // The mouse is over no Part, therefore start a selection
                    // drag box
                    rubber->startAnchoredDrag(lastMousePressPos.x(),
                                              lastMousePressPos.y(),
                                              length.x(), length.y(),
                                              trackno, true, false);
                    dragType = DragSelection;
                }
                else
                {
                    // The mouse was over a Part. It must have been selected.
                    // Now figure whether it's a start/end drag or copy/move

                    if (lastMousePressPos.x()
                        < timeLine->clockToPixel(part->start())
                          + PART_STARTEND_DRAG_AREA)
                    {
                        // A Part start drag
                        if (e->state() & ShiftButton)
                        {
                            dragType = DragPartStartShift;
                        }
                        else
                        {
                            dragType = DragPartStart;
                        }
                    }
                    else if (lastMousePressPos.x()
                             > timeLine->clockToPixel(part->end())
                               - PART_STARTEND_DRAG_AREA)
                    {
                        dragType = DragPartEnd;
                    }
                    else
                    {
                        // A Part move drag
                        if (e->state() & ShiftButton)
                        {
                            dragType = DragPartCopy;
                        }
                        else
                        {
                            dragType = DragPartMove;
                        }
                    }

                    int boxx
                        = timeLine->clockToPixel(partSelection->earliest());
                    int boxy
                        = partSelection->minTrack()*itemHeight;
                    int boxwidth
                        = timeLine->clockToPixel(partSelection->latest())
                        - timeLine->clockToPixel(partSelection->earliest());
                    int boxheight
                        = (partSelection->maxTrack()
                           - partSelection->minTrack()+1)
                        * itemHeight;
                    switch (dragType)
                    {
                        case DragPartCopy:
                        case DragPartMove:
                        {
                            rubber->startDragBox(lastMousePressPos.x(),
                                                 lastMousePressPos.y(),
                                                 0,
                                                 boxx, boxy,
                                                 boxwidth, boxheight,
                                                 true, true,
                                                 left, 0,
                                                 right, contentsHeight());
                            break;
                        }
                        case DragPartStart:
                        case DragPartStartShift:
                        {
                            rubber->startAnchoredDrag(boxx + boxwidth, boxy,
                                                      -boxwidth, boxheight,
                                                      0,
                                                      true, true,
                                                      boxx + boxwidth, boxy,
                                                      0, boxheight,
                                                      left, boxy,
                                                      (boxx + boxwidth) - left,
                                                      boxheight
                                                      );
                            setCursor(sizeHorCursor);
                            break;
                        }
                        case DragPartEnd:
                        {
                            rubber->startAnchoredDrag(boxx, boxy,
                                                      boxwidth, boxheight,
                                                      0,
                                                      true, true,
                                                      boxx, boxy,
                                                      1, boxheight,
                                                      boxx, boxy,
                                                      right, boxheight);
                            setCursor(sizeHorCursor);
                            break;
                        }
                        case DragNone:
                        case DragSelection:
                        case DragInsertPart:
                        {
                            // All either invalid, or handled elsewhere.
                            // Included here to silence a warning.
                        }
                    }
                }
                break;
            }
            case Insert:
            {
                TSE3::Clock start
                    = snap(timeLine->pixelToClock(lastMousePressPos.x()));

                int    bottom  = trackno*itemHeight;
                int    top     = (trackno+1)*itemHeight;
                rubber->startAnchoredDrag(timeLine->clockToPixel(start),
                                          lastMousePressPos.y(),
                                          length.x(), length.y(), trackno,
                                          true, true,
                                          lastMousePressPos.x(), bottom,
                                          0, top-bottom,
                                          left, bottom, right, top-bottom);
                dragType = DragInsertPart;
                break;
            }
            case Snip:
            case Glue:
            {
                // No drag action for these modes
                break;
            }
        }
        if (_viewOptions.dragSetsMarkers)
        {
            song->setFrom(snap(timeLine->pixelToClock(e->x())));
        }
    }
}


void PartView::slotDragBoxCompleted(int x, int y, int width, int height,
                                    int private_word, bool anchored)
{
    filter(x, y, width, height, private_word, anchored);
    unsetCursor();

    // Work out some generally useful values
    if (width < 0)
    {
        x += width;
        width = -width;
    }
    if (height < 0)
    {
        y += height;
        height = -height;
    }
    size_t mintrack = trackAt(y);
    size_t maxtrack = trackAt(y+height);
    if (mintrack > song->size()) mintrack = song->size();
    if (maxtrack > song->size()) maxtrack = song->size();
    TSE3::Clock start = timeLine->pixelToClock(x);
    TSE3::Clock end   = timeLine->pixelToClock(x+width);

    if (_viewOptions.dragSetsMarkers)
    {
        song->setFrom(snap(start));
        song->setTo(snap(end));
    }

    switch (dragType)
    {
        case DragSelection:
        {
            // Select every Part that falls inside the drag box
            partSelection->clear();
            emit partSelected(0);
            for (size_t n = mintrack; n <= maxtrack; n++)
            {
                size_t pos = (*song)[n]->index(start);
                while (pos < (*song)[n]->size()
                       && (*(*song)[n])[pos]->start() <= end)
                {
                    TSE3::Part *part = (*(*song)[n])[pos];
                    partSelection->select(part, true);
                    emit partSelected(part);
                    pos++;
                }
            }
            updateSelection();
            break;
        }
        case DragPartMove:
        {
            TSE3::Clock minTime    = timeLine->pixelToClock(x);
            if (minTime < 0) minTime = 0; // can be with filtering errors
            int         deltaTrack = mintrack - partSelection->minTrack();
            TSE3::Clock deltaTime  = start - partSelection->earliest();

            if (0)
            {
                std::cout << "DragPartMove: track delta " << deltaTrack
                          << ", clock delta " << deltaTime << "\n";
            }

            TSE3::App::PartSelection  temp;
            TSE3::Cmd::CommandGroup  *cg   = new TSE3::Cmd::CommandGroup();
            bool                      msg  = false;

            while (partSelection->size())
            {
                TSE3::Part *part = partSelection->front();
                partSelection->deselect(part);

                size_t trk = song->index(part->parent())
                           + deltaTrack;
                if (trk >= 0 && trk < song->size())
                {
                    try
                    {
                        TSE3::Cmd::Command *command
                            = new TSE3::Cmd::Part_Move(_partInsertAction,
                                                       part, (*song)[trk],
                                                       minTime);
                        command->execute();
                        cg->add(command);
                    }
                    catch (const TSE3::Error &e)
                    {
                        // XXX delete the command ?
                        if (!msg)
                        {
                            KMessageBox::sorry(this,
                                               "This will cause one or more "
                                               "part to overlap in a track.\n"
                                               "No part has been moved.");
                            msg = true;
                        }
                    }
                }
                temp.select(part, true);
            }
            if (cg->size())
            {
                cg->execute();
                Application::application()->history(song)->add(cg);
            }
            else
            {
                delete cg;
            }
            *partSelection = temp;
            break;
        }
        case DragPartStart:
        {
            TSE3::Clock delta   = timeLine->pixelToClock(x)
                                - partSelection->earliest();

            if (1)
            {
                std::cout << "DragPartStart: clock delta " << delta << "\n";
            }

            TSE3::App::PartSelection  temp;
            TSE3::Cmd::CommandGroup  *cg   = new TSE3::Cmd::CommandGroup();
            bool                      msg  = false;

            while (partSelection->size())
            {
                TSE3::Part *part = partSelection->front();
                partSelection->deselect(part);

                try
                {
                    TSE3::Cmd::Command *command
                        = new TSE3::Cmd::Part_Move(_partInsertAction,
                                                   part, part->parent(),
                                                   part->start() + delta,
                                                   part->end());
                    command->execute();
                    cg->add(command);
                }
                catch (const TSE3::Error &e)
                {
                    // XXX delete the command ?
                    if (!msg)
                    {
                        KMessageBox::sorry(this,
                                           "This will cause one or more "
                                           "part to overlap in a track.\n"
                                           "No part has been moved.");
                        msg = true;
                    }
                }
                temp.select(part, true);
            }
            if (cg->size())
            {
                cg->execute();
                Application::application()->history(song)->add(cg);
            }
            else
            {
                delete cg;
            }
            *partSelection = temp;
            break;
        }
        case DragPartStartShift:
        {
            std::cout << "DragPartStartShift\n";
            break;
        }
        case DragPartEnd:
        {
            TSE3::Clock delta   = timeLine->pixelToClock(x + width)
                                - partSelection->latest();

            if (1)
            {
                std::cout << "DragPartEnd: clock delta " << delta << "\n";
            }

            TSE3::App::PartSelection  temp;
            TSE3::Cmd::CommandGroup  *cg   = new TSE3::Cmd::CommandGroup();
            bool                      msg  = false;

            while (partSelection->size())
            {
                TSE3::Part *part = partSelection->front();
                partSelection->deselect(part);

                try
                {
                    TSE3::Cmd::Command *command
                        = new TSE3::Cmd::Part_Move(_partInsertAction,
                                                   part, part->parent(),
                                                   part->start(),
                                                   part->end() + delta);
                    command->execute();
                    cg->add(command);
                }
                catch (const TSE3::Error &e)
                {
                    // XXX delete the command ?
                    if (!msg)
                    {
                        KMessageBox::sorry(this,
                                           "This will cause one or more "
                                           "part to overlap in a track.\n"
                                           "No part has been moved.");
                        msg = true;
                    }
                }
                temp.select(part, true);
            }
            if (cg->size())
            {
                cg->execute();
                Application::application()->history(song)->add(cg);
            }
            else
            {
                delete cg;
            }
            *partSelection = temp;
            break;
        }
        case DragPartCopy:
        {
            std::cout << "DragPartCopy\n";
            break;
        }
        case DragInsertPart:
        {
            // Insert a part between the given times, private_word is trackno

            if (width < 0)
            {
                x += width;
                width = -width;
            }
            if (width < 10)
            {
                // This Part is so small, it was probably a user twitch ;-)
                return;
            }

            if (start < 0) start = 0;
            if (end < 0)   end   = 0;

            /*
            (*song)[private_word]->insert(start, end);

            // Make the new Part the currently selection
            updateSelection();
            TSE3::Part *newPart = partAt(x, y);
            partSelection->select(newPart, false);
            */

            TSE3::Part *part = 0;
            try
            {
                part = new TSE3::Part();
                part->setStart(start);
                part->setEnd(end);
                part->setPhrase(phraseListWindow
                                ? phraseListWindow->phrase()
                                : 0);

                TSE3::Cmd::Command *command
                    = new TSE3::Cmd::Part_Move(_partInsertAction,
                                               part, (*song)[private_word],
                                               start);
                command->execute();
                Application::application()->history(song)->add(command);
            }
            catch (const TSE3::Error &e)
            {
                // XXX delete the command ?
                KMessageBox::sorry(this,
                                   "Sorry, that would cause parts to "
                                   "overlap in a track");
                if (part)
                {
                    delete part;
                    part = 0;
                }
            }
            if (part)
            {
                updateSelection();
                partSelection->select(part, false);
            }
            break;
        }
        default:
        {
            // erk!
            std::cerr << "PartView drag code reached surreal condition\n";
        }
    }
    dragType = DragNone;
}


void PartView::contentsMouseDoubleClickEvent(QMouseEvent *e)
{
    if (_pointerMode == Select)
    {
        TSE3::Part *part = partAt(e->x(), e->y());
        if (part)
        {
            if (e->state() & ShiftButton)
            {
                if (part->phrase())
                {
                    PhraseEditor *pe
                        = new PhraseEditor(this, song,
                                           Application::application()
                                               ->history(song),
                                           part->phrase());
                    pe->show();
                }
            }
            else
            {
                new PartWindow(this, part, song,
                               Application::application()->history(song));
            }
        }
    }
}


void PartView::keyPressEvent(QKeyEvent *e)
{
    if (e->key() == Key_Escape)
    {
        std::cout << "moo\n";
        rubber->stop();
        unsetCursor();
    }
    else
    {
        e->ignore();
    }
}


void PartView::focusInEvent(QFocusEvent *e)
{
    // don't update the display - it flickers badly
    std::cout << "focus in event\n";
}


void PartView::focusOutEvent(QFocusEvent *e)
{
    // don't update the display - it flickers badly
    std::cout << "focus out event\n";
}


void PartView::filter(int &x, int &y, int &width, int &height,
                      int private_word, bool anchored)
{
    y /= itemHeight;
    y *= itemHeight;

    if (anchored)
    {
        // snap end point (x+width)
        const TSE3::Clock snapped = snap(timeLine->pixelToClock(x+width));
        const int end             = timeLine->clockToPixel(snapped);
        width = end - x;
    }
    else
    {
        // snap x, leave width
        const TSE3::Clock snapped = snap(timeLine->pixelToClock(x));
        x = timeLine->clockToPixel(snapped);
    }
}


/******************************************************************************
 * Window scrolling and resizing
 *****************************************************************************/

void PartView::resizeContents(int w, int h)
{
    QScrollView::resizeContents(w, h);
    updateContents(0, h, w, height()-h-contentsY());
}


void PartView::setHBarGeometry(QScrollBar &hbar,
                               int x, int y, int w, int h)
{
    hbar.setGeometry(x, y, w-h*2, h);
    if (hZoomIn)  hZoomIn->setGeometry(x + w-h*2, y, h, h);
    if (hZoomOut) hZoomOut->setGeometry(x + w-h, y, h, h);
}


void PartView::setVBarGeometry(QScrollBar &vbar,
                               int x, int y, int w, int h)
{
    vbar.setGeometry(x, y, w, h-w*2);
    if (vZoomIn)  vZoomIn->setGeometry(x, y + h-w*2, w, w);
    if (vZoomOut) vZoomOut->setGeometry(x, y + h-w, w, w);
}


void PartView::setItemHeight(int h)
{
    // change size
    itemHeight = h;
    resizeContents(contentsWidth(), itemHeight*song->size());
    // do a redraw
    viewport()->repaint();
}


void PartView::moveHorizontal(int x, int /*y*/)
{
    if (x != contentsX())
    {
        bool b = this->signalsBlocked();
        blockSignals(true);
        setContentsPos(x, contentsY());
        blockSignals(b);
    }
}


void PartView::moveVertical(int /*x*/, int y)
{
    if (y != contentsY())
    {
        bool b = this->signalsBlocked();
        blockSignals(true);
        setContentsPos(contentsX(), y);
        blockSignals(b);
    }
}


void PartView::slotChangeWidth(int newWidth, bool needsRedraw)
{
    resizeContents(newWidth, contentsHeight());
    if (needsRedraw)
    {
        viewport()->repaint();
    }
}


/******************************************************************************
 * Utility functions
 *****************************************************************************/

TSE3::Part *PartView::partAt(int x, int y)
{
    TSE3::Clock time  = timeLine->pixelToClock(x);
    return partAt(time, y);
}


TSE3::Part *PartView::partAt(TSE3::Clock time, int y)
{
    size_t      trkno = y/itemHeight;
    if (trkno < song->size())
    {
        TSE3::Track *track = (*song)[trkno];
        size_t       prtno = track->index(time);
        if (prtno < track->size() && (*track)[prtno]->start() <= time)
        {
            return (*track)[prtno];
        }
    }
    return 0;
}


size_t PartView::trackAt(int y)
{
    size_t trackno = y/itemHeight;
    if (trackno < 0)             trackno = 0;
    if (trackno >= song->size()) trackno = song->size()-1;
    return trackno;
}


/******************************************************************************
 * Tracking the TSE3 objects
 *****************************************************************************/

void PartView::Song_TrackInserted(TSE3::Song *song, TSE3::Track *track)
{
    resizeContents(contentsWidth(), itemHeight*song->size());
    update(0, song->index(track));
}


void PartView::Song_TrackRemoved(TSE3::Song *song, TSE3::Track *track,
                                 size_t index)
{
    resizeContents(contentsWidth(), itemHeight*song->size());
    update(0, index);
}


void PartView::Song_FromAltered(TSE3::Song *, TSE3::Clock from)
{
    const TSE3::Clock DELTA = 10;
    update(0, lastFrom - DELTA, -1, lastFrom + DELTA);
    update(0, from - DELTA, -1, from + DELTA);
    lastFrom = from;
}


void PartView::Song_ToAltered(TSE3::Song *, TSE3::Clock to)
{
    const TSE3::Clock DELTA = 10;
    update(0, lastTo - DELTA, -1, lastTo + DELTA);
    update(0, to - DELTA, -1, to + DELTA);
    lastTo = to;
}


void PartView::Track_PartInserted(TSE3::Track *track, TSE3::Part *part)
{
    TSE3::Listener<TSE3::PartListener>::attachTo(part);
    if (part->phrase())
    {
        TSE3::Listener<TSE3::PhraseListener>::attachTo(part->phrase());
    }
    size_t trackno = song->index(track);
    update(trackno, part->start(), trackno, part->end());
}


void PartView::Track_PartRemoved(TSE3::Track *track, TSE3::Part *part)
{
    TSE3::Listener<TSE3::PartListener>::detachFrom(part);
    size_t trackno = song->index(track);
    update(trackno, part->start(), trackno, part->end());
}


void PartView::Part_StartAltered(TSE3::Part *part, TSE3::Clock start)
{
    if (part->parent())
    {
        size_t trackno = song->index(part->parent());
        update(trackno, part->start(), trackno, part->end());
    }
}


void PartView::Part_EndAltered(TSE3::Part *part, TSE3::Clock end)
{
    if (part->parent())
    {
        size_t trackno = song->index(part->parent());
        update(trackno, part->start(), trackno, part->end());
    }
}


void PartView::Part_RepeatAltered(TSE3::Part *part, TSE3::Clock repeat)
{
    if (part->parent())
    {
        size_t trackno = song->index(part->parent());
        update(trackno, part->start(), trackno, part->end());
    }
}


void PartView::Part_PhraseAltered(TSE3::Part *part, TSE3::Phrase *phrase)
{
    if (part->parent())
    {
        size_t trackno = song->index(part->parent());
        update(trackno, part->start(), trackno, part->end());
        if (phrase)
        {
            TSE3::Listener<TSE3::PhraseListener>::attachTo(phrase);
        }
    }
}


void PartView::Part_MidiFilterAltered(TSE3::Part *part)
{
    if (part->parent())
    {
        size_t trackno = song->index(part->parent());
        update(trackno, part->start(), trackno, part->end());
    }
}


void PartView::Part_MidiParamsAltered(TSE3::Part *part)
{
    if (part->parent())
    {
        size_t trackno = song->index(part->parent());
        update(trackno, part->start(), trackno, part->end());
    }
}


void PartView::Part_DisplayParamsAltered(TSE3::Part *part)
{
    if (part->parent())
    {
        size_t trackno = song->index(part->parent());
        update(trackno, part->start(), trackno, part->end());
    }
}


void PartView::Phrase_TitleAltered(TSE3::Phrase *phrase)
{
    updatePhrase(phrase);
}


void PartView::Phrase_DisplayParamsAltered(TSE3::Phrase *phrase)
{
    updatePhrase(phrase);
}


void PartView::PresetColours_Altered(TSE3::PresetColours *, int)
{
    update(0, 0);
}


void PartView::PartSelection_Selected(TSE3::App::PartSelection *,
                                      TSE3::Part *part, bool)
{
    update(song->index(part->parent()), part->start(),
           song->index(part->parent()), part->end());
}


void PartView::attachToTrack(TSE3::Track *track)
{
    TSE3::Listener<TSE3::TrackListener>::attachTo(track);
    for (size_t n = 0; n < track->size(); n++)
    {
        TSE3::Listener<TSE3::PartListener>::attachTo((*track)[n]);
        if ((*track)[n]->phrase())
        {
            TSE3::Listener<TSE3::PhraseListener>::attachTo
                ((*track)[n]->phrase());
        }
    }
}


/******************************************************************************
 * Other actions
 *****************************************************************************/

void PartView::slotPhraseSelected(TSE3::Phrase *phrase)
{
    TSE3::Cmd::CommandGroup *command = new TSE3::Cmd::CommandGroup();

    TSE3::App::PartSelection::iterator_t i = partSelection->begin();
    while (i != partSelection->end())
    {
        command->add(new TSE3::Cmd::Part_SetPhrase(*i, phrase));
        i++;
    }
    if (command->size())
    {
        command->execute();
        Application::application()->history(song)->add(command);
    }
    else
    {
        delete command;
    }
}


void PartView::slotCut()
{
    toClipboard(partSelection);

    TSE3::Cmd::CommandGroup *command = new TSE3::Cmd::CommandGroup();

    TSE3::App::PartSelection::iterator_t i = partSelection->begin();
    while (i != partSelection->end())
    {
        command->add(new TSE3::Cmd::Track_RemovePart(*i));
        i++;
    }
    command->execute();
    Application::application()->history(song)->add(command);
}


void PartView::slotCopy()
{
    toClipboard(partSelection);
}


void PartView::slotPaste()
{
    std::cout << "PartView::slotPaste\n";

    if (clipboardData() == Clipboard_Part)
    {
        std::cout  << "valid data\n";
    }
}


void PartView::slotSelectAll()
{
    partSelection->selectAll(song);
}


void PartView::slotSelectNone()
{
    partSelection->clear();
}


void PartView::slotSelectInvert()
{
    partSelection->invert(song);
}


void PartView::slotSelectInside()
{
    partSelection->clear();
    partSelection->selectBetween(song, song->from(), song->to());
}


void PartView::slotSelectOutside()
{
    partSelection->clear();
    partSelection->selectBetween(song, song->from(), song->to(), false);
}


