/*
 * @(#)SongWindow_PartView.h 1.00 22 May 2000
 *
 * Copyright (c) Pete Goodliffe 2000 (pete@cthree.org)
 *
 * This file is part of anthem - the TSE3 sequencer.
 *
 * This program is modifiable/redistributable under the terms of the GNU
 * General Public License.
 *
 * You should have recieved a copy of the GNU General Public License along
 * with this program; see the file COPYING. If not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 0211-1307, USA.
 */

#ifndef ANTHEM_SONGWINDOW_PARTVIEW_H
#define ANTHEM_SONGWINDOW_PARTVIEW_H

#include <qscrollview.h>

#include "tse3/Midi.h"
#include "tse3/util/Snap.h"
#include "tse3/Notifier.h"
#include "tse3/listen/Song.h"
#include "tse3/listen/Track.h"
#include "tse3/listen/Part.h"
#include "tse3/listen/Phrase.h"
#include "tse3/listen/DisplayParams.h"
#include "tse3/listen/app/PartSelection.h"
#include "tse3/listen/app/TrackSelection.h"

#include "gadgets/RubberSelectionBox.h"

/******************************************************************************
 * TSE3 forward declarations
 *****************************************************************************/
namespace TSE3
{
    class Song;
    class Phrase;
    class Track;
}

/******************************************************************************
 * KDE/Qt forward declarations
 *****************************************************************************/
class QToolButton;
class QPopupMenu;

/******************************************************************************
 * Anthem forward declataions
 *****************************************************************************/
class PhraseListWindow;

/******************************************************************************
 * File forward declataions
 *****************************************************************************/
class TimeLine;

/**
 * This widget displays the area containing the Song's Parts.  It is held within
 * the main @ref SongWindow.
 *
 * @short   Part display area
 * @author  Pete Goodliffe
 * @version 1.0
 */
class PartView : public QScrollView,
                 public RubberSelectionBoxFilter,
                 public TSE3::Listener<TSE3::SongListener>,
                 public TSE3::Listener<TSE3::TrackListener>,
                 public TSE3::Listener<TSE3::PartListener>,
                 public TSE3::Listener<TSE3::PhraseListener>,
                 public TSE3::Listener<TSE3::PresetColoursListener>,
                 public TSE3::Listener<TSE3::App::PartSelectionListener>
{
        Q_OBJECT;

    public:

        /**
         * Ctr - specify the @ref Song to work with, the parent, and the
         * timeline (for time->clock conversion).
         *
         * This is called by the @ref SongWindow (which will be the
         * parent).
         */
        PartView(TSE3::Song *song, QWidget *parent, TimeLine *timeLine,
                 TSE3::Util::Snap &snap,
                 TSE3::App::PartSelection *ps,
                 TSE3::App::TrackSelection *ts);

        virtual ~PartView();

        /**
         * Changes which @ref TSE3::Song this window views.
         *
         * This can only be called safely by the @ref SongWindow.
         */
        void changeSong(TSE3::Song *song);

        /**
         * This method sets the height of each Track line in the PartView.
         * It should only be called by the @ref SongWindow
         * on reciept of a zoom event.
         */
        void setItemHeight(int height);

        /**
         * Sets the popup menu to show when the right menu button is
         * closed. Currently, the SongWindow passes the 'Edit' menu here.
         *
         * You can pass zero which disables this menu (and is the default
         * value).
         */
        void setRMBMenu(QPopupMenu *menu);

        /**
         * enum type describing the pointer mode. These are:
         *
         * @li @p Select - For moving/resizeing @ref TSE3::Part objects.
         * @li @p Snip   - For snipping @ref TSE3::Part objects in two
         * @li @p Glue   - For glueing @ref TSE3::Part object together
         * @li @p Insert - For adding new @ref TSE3::Parts
         *
         * @see pointerMode
         */
        enum PointerMode
        {
            Select,
            Snip,
            Glue,
            Insert
        };

        /**
         * Returns the current pointer mode.
         *
         * @see slotSetPointerMode
         */
        PointerMode pointerMode() { return _pointerMode; }

        /**
         * Returns the current @ref TSE3::Util::PartInsertAction.
         *
         * @see slotSetPartInsertAction
         */
        int partInsertAction() const { return _partInsertAction; }

        /**
         * Called by the @ref SongWindow when a @ref PhraseListWindow is
         * created. This can be used to choose which @ref TSE3::Phrase is
         * used when a @ref TSE3::Part is inserted.
         */
        void setPhraseListWindow(PhraseListWindow *plw)
        {
            phraseListWindow = plw;
        }

        /**
         * Options describing how the PartView is drawn
         */
        struct ViewOptions
        {
            enum ColourOptions
            {
                Colour_None,
                Colour_TextForeground,
                Colour_TextBackground,
                Colour_Box,
                Colour_Dot
            };

            ViewOptions()
                : verticalRules(true), phraseName(true),
                  phraseEvents(true), plainBoxes(false),
                  fillBoxes(true),
                  colours(Colour_Box), dragSetsMarkers(false)
                {}
            bool          verticalRules;
            bool          phraseName;
            bool          phraseEvents;
            bool          plainBoxes;
            bool          fillBoxes;
            ColourOptions colours;
            bool          dragSetsMarkers;
        };

        /**
         * Returns the current ViewOptions
         *
         * @see setViewOptions
         */
        const ViewOptions &viewOptions() const { return _viewOptions; }

        /**
         * Sets the view options
         */
        void setViewOptions(const ViewOptions &vo)
        {
            _viewOptions = vo;
            viewport()->repaint();
        }

        /**
         * @reimplemented
         */
        virtual void Song_TrackInserted(TSE3::Song *, TSE3::Track *);

        /**
         * @reimplemented
         */
        virtual void Song_TrackRemoved(TSE3::Song *, TSE3::Track *, size_t);

        /**
         * @reimplemented
         */
        virtual void Song_FromAltered(TSE3::Song *src, TSE3::Clock from);

        /**
         * @reimplemented
         */
        virtual void Song_ToAltered(TSE3::Song *src, TSE3::Clock to);

        /**
         * @reimplemented
         */
        virtual void Track_PartInserted(TSE3::Track *track, TSE3::Part *part);

        /**
         * @reimplemented
         */
        virtual void Track_PartRemoved(TSE3::Track *track, TSE3::Part *part);

        /**
         * @reimplemented
         */
        virtual void Part_StartAltered(TSE3::Part *, TSE3::Clock start);

        /**
         * @reimplemented
         */
        virtual void Part_EndAltered(TSE3::Part *, TSE3::Clock end);

        /**
         * @reimplemented
         */
        virtual void Part_RepeatAltered(TSE3::Part *, TSE3::Clock repeat);

        /**
         * @reimplemented
         */
        virtual void Part_PhraseAltered(TSE3::Part *, TSE3::Phrase *phrase);

        /**
         * @reimplemented
         */
        virtual void Part_MidiFilterAltered(TSE3::Part *);

        /**
         * @reimplemented
         */
        virtual void Part_MidiParamsAltered(TSE3::Part *);

        /**
         * @reimplemented
         */
        virtual void Part_DisplayParamsAltered(TSE3::Part *);

        /**
         * @reimplemented
         */
        virtual void Phrase_TitleAltered(TSE3::Phrase *);

        /**
         * @reimplemented
         */
        virtual void Phrase_DisplayParamsAltered(TSE3::Phrase *);

        /**
         * @reimplemented
         */
         virtual void PresetColours_Altered(TSE3::PresetColours *, int);

        /**
         * @reimplemented
         */
        virtual void PartSelection_Selected
            (TSE3::App::PartSelection *, TSE3::Part *, bool selected);

        /**
         * @reimplemented
         */
        virtual void filter(int &x, int &y, int &width, int &height,
                            int private_word, bool anchored);

    public slots:

        /**
         * Slot member that sets the pointer mode. This will cause the
         * @ref pointerModeChanged signal to be emitted.
         *
         * @see pointerMode
         * @see pointerModeChanged
         */
        void slotSetPointerMode(PointerMode mode);

        /**
         * Slot member that sets the @ref TSE3::Util::PartInsertAction. This
         * will cause the @ref partInsertActionChanged signal to be emitted.
         *
         * @see partInsertAction
         * @see partInsertActionChanged
         */
        void slotSetPartInsertAction(int action);

        /**
         * This slot is called when the TrackList is scrolled (although
         * it doesn't have a scroll bar, the keyboard can be used to
         * scroll it). The x value is ignored.
         */
        void moveVertical(int x, int y);

        /**
         * This slot is called when the TimeLine is scrolls (although
         * it doesn't have a scroll bar, the cursor move can be used to
         * scroll it). The y value is ignored.
         */
        void moveHorizontal(int x, int y);

        /**
         * This slot is called by the TimeLine when it's width changes.
         */
        void slotChangeWidth(int newWidth, bool needsRedraw);

        /**
         * A @ref TSE3::Phrase has been chosen. Any selected Parts will
         * be set to that Phrase.
         */
        void slotPhraseSelected(TSE3::Phrase *phrase);

        /**
         * Perform a cut operation
         */
        void slotCut();

        /**
         * Perform a copy operation
         */
        void slotCopy();

        /**
         * Perform a paste operation
         */
        void slotPaste();

        /*
         * Select operations.
         */
        void slotSelectAll();
        void slotSelectNone();
        void slotSelectInvert();
        void slotSelectOutside();
        void slotSelectInside();

        /**
         * @reimplemented
         *
         * To fix a bug in Qt. If you shrink the contents size of a
         * @ref QScrollView which is in a window much larger that this size,
         * the difference (between contentsSize and parentSize) is not updated.
         *
         * In this overriden version I ensure that this section of window is
         * updated.
         */
        virtual void resizeContents(int w, int h);

    private slots:

        void slotDragBoxCompleted(int x, int y, int width, int height,
                                  int private_word, bool anchored);

    signals:

        /**
         * This signal is emitted when the zoom button is pressed. The
         * reciever will be the @ref SongWindow which will respond by
         * opening a @ref ZoomWidget.
         */
        void zoomWidget();

        void zoomInHorizontal();
        void zoomOutHorizontal();
        void zoomInVertical();
        void zoomOutVertical();

        /**
         * This signal is emitted when a @ref TSE3::Part is selected.
         *
         * When a Part is deselected then @p part is zero.
         */
        void partSelected(TSE3::Part *part);

        /**
         * This signal is emitted when the pointer mode is changed.
         *
         * @see slotSetPointerMode
         */
         void pointerModeChanged();

        /**
         * This signal is emitted when the @ref TSE3::Util::PartInsertAction is
         * changed.
         *
         * @see slotSetPartInsertAction
         */
         void partInsertActionChanged();

         /**
          * Emitted when the RMB menu is opened.
          *
          * @param track The index of the track RMB is clicked over
          * @param part  The @ref TSE3::Part at this position, or zero
          */
         void rmbMenuOpened(size_t track,TSE3::Part*);

         /**
          * Emitted when a click occurs in the PartView.
          */
         void setClock(TSE3::Clock time, Qt::ButtonState button);

    protected:

        virtual void drawContents(QPainter *p,
                                 int clipx, int clipy, int clipw, int cliph);

        virtual void setHBarGeometry(QScrollBar &vbar,
                                     int x, int y, int w, int h);
        virtual void setVBarGeometry(QScrollBar &vbar,
                                     int x, int y, int w, int h);

        virtual void contentsMousePressEvent(QMouseEvent *e);
        virtual void contentsMouseReleaseEvent(QMouseEvent *e);
        virtual void contentsMouseMoveEvent(QMouseEvent *e);
        virtual void contentsMouseDoubleClickEvent(QMouseEvent *e);
        virtual void keyPressEvent(QKeyEvent *e);
        virtual void focusInEvent(QFocusEvent *e);
        virtual void focusOutEvent(QFocusEvent *e);

    private:

        /**
         * Returns a pointer to the @ref Part at the given contents
         * position, or zero if there is no @ref Part.
         */
        TSE3::Part *partAt(int x, int y);

        /**
         * Returns a pointer to the @ref Part at the given time with @ref
         * TSE3::Track at the given coordination, or zero if there is no
         * @ref Part.
         */
        TSE3::Part *partAt(TSE3::Clock time, int y);

        /**
         * Returns index of @ref TSE3::Track at the given coordinates,
         * checking that the track index is within bounds.
         */
        size_t trackAt(int y);

        /**
         * Redraw some of the @ref PartList.
         *
         * @param minTrack The track index to start redraw at
         * @param minTime  The time position to start redraw at
         * @param maxTrack The track to stop redraw at, -1 means until end
         * @param maxTime  The time position to stop at, -1 means until end
         */
        void redraw(int minTrack, TSE3::Clock minTime,
                    int maxTrack = -1, TSE3::Clock maxTime = -1);

        /**
         * Update some of the @ref PartList.
         *
         * The update in Qt the is kind of redraw that happens some time
         * in the future, so you can subsequently change the state of what's
         * in the udpate area, and it will happen before the window is
         * redrawn.
         *
         * @param minTrack The track index to start redraw at
         * @param minTime  The time position to start redraw at
         * @param maxTrack The track to stop redraw at, -1 means until end
         * @param maxTime  The time position to stop at, -1 means until end
         */
        void update(int minTrack, TSE3::Clock minTime,
                    int maxTrack = -1, TSE3::Clock maxTime = -1);

        /**
         * Causes an update of the selected @ref TSE3::Part objects.
         */
        void updateSelection();

        /**
         * Updates any @ref TSE3::Part that uses the specified
         * @ref TSE3::Phrase.
         */
        void updatePhrase(TSE3::Phrase *phrase);

        /**
         * Updates the specified @ref TSE3::Part.
         */
        void updatePart(TSE3::Part *part);

        /**
         * Attaches to all interesting bits of the specified Track.
         * You can call this for a Track you're already attached to
         * since multiple @ref TSE3::Notifier::attach calls don't hurt.
         */
        void attachToTrack(TSE3::Track *track);

        /**
         * enum type describing what drag operation is going on. Values are:
         *
         * @li DragNone           No drag is taking place
         * @li DragPartMove       Moving Parts
         * @li DragPartStart      Changing Part start
         * @li DragPartStartShift Changing Part start with time shift
         * @li DragPartEnd        Change Part end
         * @li DragPartCopy       Copying Parts
         * @li DragInsertPart     Inserting new Part
         */
        enum DragType
        {
            DragNone,
            DragSelection,
            DragPartMove,
            DragPartStart,
            DragPartStartShift,
            DragPartEnd,
            DragPartCopy,
            DragInsertPart
        };

        /*
         * Handlers for the contentsMousePressedEvent for each of the pointer
         * modes.
         */
        void selectMousePressEvent(QMouseEvent *e);
        void snipMousePressEvent(QMouseEvent *e);
        void glueMousePressEvent(QMouseEvent *e);

        /**
         * Draws a Part according to the current @ref ViewOptions.
         */
        void drawPart(QPainter *p, TSE3::Part *part);

        RubberSelectionBox        *rubber;
        QPoint                     lastMousePressPos;
        TSE3::Song                *song;
        TSE3::App::PartSelection  *partSelection;
        TSE3::App::TrackSelection *trackSelection;
        TimeLine                  *timeLine;
        int                        itemHeight;
        PointerMode                _pointerMode;
        int                        _partInsertAction;
        QToolButton               *hZoomIn, *hZoomOut;
        QToolButton               *vZoomIn, *vZoomOut;
        DragType                   dragType;
        PhraseListWindow          *phraseListWindow;
        ViewOptions                _viewOptions;
        TSE3::Util::Snap          &snap;
        QPopupMenu                *rmbMenu;
        TSE3::Clock                lastFrom;
        TSE3::Clock                lastTo;
};

#endif
