/*
 * @(#)SongInfo.cpp 1.00 22 May 2000
 *
 * Copyright (c) Pete Goodliffe 2000 (pete@cthree.org)
 *
 * This file is part of anthem - the TSE3 sequencer.
 *
 * This program is modifiable/redistributable under the terms of the GNU
 * General Public License.
 *
 * You should have recieved a copy of the GNU General Public License along
 * with this program; see the file COPYING. If not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 0211-1307, USA.
 */

#include "songwindow/PhraseList.h"

#include "phraseeditors/PhraseEditor.h"
#include "gadgets/ColourPixmap.h"
#include "Application.h"
#include "dialogues/Phrase.h"
#include "misc/Clipboard.h"

#include "tse3/Song.h"
#include "tse3/Part.h"
#include "tse3/Phrase.h"
#include "tse3/PhraseList.h"
#include "tse3/DisplayParams.h"
#include "tse3/cmd/Phrase.h"
#include "tse3/cmd/CommandHistory.h"

#include "misc/kde2-compat.h"
#include <qlayout.h>
#include <qpopupmenu.h>
#include <klistview.h>
#include <kaction.h>
#include <kstdaction.h>

enum MENU_ITEMS
{
    PROPERTIES,
    NEW,
    CUT,
    COPY,
    PASTE
};

/******************************************************************************
 * PhraseListBox class
 *****************************************************************************/

/**
 * A KListBox that has cleverness to handle the PhraseListWindow popup menu.
 * The state of the menu buttons are set before the menu is opened.
 *
 * This class is closed coupled to the PhraseListWindow.
 */
class PhraseListBox : public KListBox
{
    public:
        PhraseListBox(QPopupMenu *menu, QWidget *parent,
                      const char *name = 0, WFlags f = 0)
            : KListBox(parent, name, f), menu(menu) {}
    protected:
        virtual void mousePressEvent(QMouseEvent *e)
        {
            if (e->button() == RightButton)
            {
                menu->setItemEnabled(PROPERTIES, anySelected());
                menu->popup(mapToGlobal(e->pos()));
                return;
            }
            else
            {
                KListBox::mousePressEvent(e);
            }
        }
    private:
        bool anySelected()
        {
            for (int n = 0; n < numRows(); n++)
            {
                if (isSelected(n)) return true;
            }
            return false;
        }
        QPopupMenu *menu;
};


/******************************************************************************
 * PhraseListWindow class
 *****************************************************************************/

PhraseListWindow::PhraseListWindow(QWidget *parent, TSE3::Song *song)
: KDialogBase(Plain, "Phrase List", Close|User1|User2, User2, parent,
              "PhraseListWindow", false, true,
              KGuiItem("Edit"), KGuiItem("Select")),
  song(song), phraseList(song->phraseList()), parent(parent)
{
    // The popup menu
    menu = new QPopupMenu(this);
    menu->insertItem("Properties", this, SLOT(slotProperties()));
    menu->insertSeparator();
    menu->insertItem("New phrase", this, SLOT(slotNew()));
    KActionCollection *ac = new KActionCollection(this);
    KAction *action = KStdAction::cut(this, SLOT(slotCut()), ac);
    action->plug(menu);
    action = KStdAction::copy(this, SLOT(slotCopy()), ac);
    action->plug(menu);
    action = KStdAction::paste(this, SLOT(slotPaste()), ac);
    action->plug(menu);

    // The list box
    QVBoxLayout *layout = new QVBoxLayout(plainPage(), 0, spacingHint());
    listBox = new PhraseListBox(menu, plainPage());
    layout->addWidget(listBox);
    setMinimumWidth(fontMetrics().maxWidth()*20);

    connect(listBox, SIGNAL(doubleClicked(QListBoxItem*)),
            SLOT(slotKListBoxItemExecuted(QListBoxItem*)));

    setUpListBox();

    TSE3::Listener<TSE3::PhraseListListener>::attachTo(phraseList);
}


void PhraseListWindow::setUpListBox()
{
    listBox->clear();
    for (size_t n = 0; n < phraseList->size(); n++)
    {
        TSE3::Phrase *phr = (*phraseList)[n];
        TSE3::Listener<TSE3::PhraseListener>::attachTo(phr);
        listBox->insertItem(ColourPixmap::newPixmap(*phr->displayParams()),
                            phr->title().c_str());
    }
}


void PhraseListWindow::changeSong(TSE3::Song *s)
{
    // 1. Out with the old
    TSE3::Listener<TSE3::PhraseListListener>::detachFrom(phraseList);
    for (size_t n = 0; n < phraseList->size(); n++)
    {
        TSE3::Listener<TSE3::PhraseListener>::detachFrom((*phraseList)[n]);
    }
    listBox->clear();

    // 2. In with the new
    song       = s;
    phraseList = s->phraseList();
    TSE3::Listener<TSE3::PhraseListListener>::attachTo(phraseList);
    setUpListBox();
}


void PhraseListWindow::accept()
{
}


TSE3::Phrase *PhraseListWindow::phrase()
{
    for (size_t n = 0; n < listBox->count(); n++)
    {
        if (listBox->isSelected(n))
        {
            return (*phraseList)[n];
        }
    }
    return 0;
}


void PhraseListWindow::done(int reason)
{
    hide();
    emit closed();
}


void PhraseListWindow::slotPartSelected(TSE3::Part *part)
{
    if (part && part->phrase())
    {
        size_t index = phraseList->index(part->phrase());
        if (index < phraseList->size())
        {
            listBox->setSelected(index, true);
            listBox->ensureCurrentVisible();
        }
    }
}


void PhraseListWindow::slotKListBoxItemExecuted(QListBoxItem *item)
{
    emit phraseSelected(phrase());
}


void PhraseListWindow::slotUser1()
{
    PhraseEditor *pe
        = new PhraseEditor(parent, song,
                           Application::application()->history(song), phrase());
    pe->show();
}


void PhraseListWindow::slotUser2()
{
    emit phraseSelected(phrase());
}


void PhraseListWindow::PhraseList_Inserted(TSE3::PhraseList *,
                                           TSE3::Phrase *phr)
{
    TSE3::Listener<TSE3::PhraseListener>::attachTo(phr);
    setUpListBox(); // XXX hacky
    listBox->setSelected(phraseList->index(phr), true);
    listBox->ensureCurrentVisible();
}


void PhraseListWindow::PhraseList_Removed(TSE3::PhraseList *, TSE3::Phrase *phr)
{
    TSE3::Listener<TSE3::PhraseListener>::detachFrom(phr);
    setUpListBox(); // XXX hacky
}


void PhraseListWindow::Phrase_DisplayParamsAltered(TSE3::Phrase *phr)
{
    size_t n = phraseList->index(phr);

    if (n != phraseList->size())
    {
        // update Phrase in list - like code in setUpListBox

        QPixmap p;
        if (phr->displayParams()->style() == TSE3::DisplayParams::Colour)
        {
            int r, g, b;
            phr->displayParams()->colour(r, g, b);
            p = ColourPixmap::newPixmap(r, g, b);
        }
        else if (phr->displayParams()->style()
                     == TSE3::DisplayParams::PresetColour)
        {
            int r, g, b;
            Application::application()->presetColours()->colour
                (phr->displayParams()->presetColour(), r, g, b);
            p = ColourPixmap::newPixmap(r, g, b);
        }
        else
        {
            p = ColourPixmap::newPixmap();
        }
        listBox->changeItem(p, phr->title().c_str(), n);
    }
}


void PhraseListWindow::slotProperties()
{
    TSE3::Phrase              *phr     = phrase();
    TSE3::Cmd::CommandHistory *history = Application::application()->history(song);
    if (phr)
    {
        PhraseWindow *pw = new PhraseWindow(this, phr, history);
        pw->show();
    }
}


void PhraseListWindow::slotNew()
{
    PhraseEditor *pe
        = new PhraseEditor(parent, song,
                           Application::application()->history(song), 0);
    pe->show();
}


void PhraseListWindow::slotCut()
{
    toClipboard(phrase());

    TSE3::Cmd::Command *command = new TSE3::Cmd::Phrase_Erase(phrase(), song);
    command->execute();
    Application::application()->history(song)->add(command);
}


void PhraseListWindow::slotCopy()
{
    toClipboard(phrase());
}


void PhraseListWindow::slotPaste()
{
    std::cout << "PhraseListWindow::slotPaste\n";
}

