/*
 * @(#)SongWindow.cpp 1.00 22 May 2000
 *
 * Copyright (c) Pete Goodliffe 2000 (pete@cthree.org)
 *
 * This file is part of anthem - the TSE3 sequencer.
 *
 * This program is modifiable/redistributable under the terms of the GNU
 * General Public License.
 *
 * You should have recieved a copy of the GNU General Public License along
 * with this program; see the file COPYING. If not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 0211-1307, USA.
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#define QT_ALTERNATE_QTSMANIP

#include "misc/kde2-compat.h"
#include <kapp.h>
#include <kiconloader.h>
#include <klocale.h>
#include <kglobal.h>
#include <kmenubar.h>
#include <ktoolbar.h>
#include <kaccel.h>
#include <qsplitter.h>
#include <klistbox.h>
#include <qlistbox.h>
#include <qvbox.h>
#include <qheader.h>
#include <qlayout.h>
#include <kfiledialog.h>
#include <kmessagebox.h>
#include <kaction.h>
#include <kstdaction.h>
#include <qpopupmenu.h>
#include <qclipboard.h>
#include <qregexp.h>
#include <qcursor.h>
#include <ktempfile.h>
#include <kio/netaccess.h>
#include <knotifyclient.h>
#include <kkeydialog.h>
#include <kedittoolbar.h>
#include <kcombobox.h>
#include <ktip.h>
#include <kstddirs.h>

#include <fstream>

#include "tse3/Song.h"
#include "tse3/MidiScheduler.h"
#include "tse3/RepeatTrack.h"
#include "tse3/FlagTrack.h"
#include "tse3/TSE3MDL.h"
#include "tse3/TSE2MDL.h"
#include "tse3/MidiFile.h"
#include "tse3/Transport.h"
#include "tse3/Error.h"
#include "tse3/app/Record.h"
#include "tse3/app/PartSelection.h"
#include "tse3/app/TrackSelection.h"
#include "tse3/cmd/CommandHistory.h"
#include "tse3/cmd/Song.h"

#include "songwindow/SongWindow.h"
#include "songwindow/TrackList.h"
#include "songwindow/PartView.h"
#include "songwindow/TimeLine.h"
#include "songwindow/PhraseList.h"
#include "songwindow/Actions.h"

#include "tse3/kdeui/SnapWidget.h"
#include "tse3/kdeui/ClockDisplayWidget.h"
#include "tse3/kdeui/FlagComboBox.h"

#include "Application.h"
#include "dialogues/SongInfo.h"
#include "dialogues/Track.h"
#include "dialogues/About.h"
#include "dialogues/MiscWindows.h"
#include "dialogues/TSE3Progress.h"
#include "dialogues/NewFlag.h"
#include "dialogues/NewTrack.h"
#include "dialogues/NewPhrase.h"
#include "dialogues/Panic.h"
#include "dialogues/Part.h"
#include "gadgets/PartInsertActionWidget.h"
#include "gadgets/MidiSchedulerWidgets.h"
#include "gadgets/Gadgets.h"
#include "gadgets/ARecentFilesAction.h"
#include "settings/PartView.h"
#include "settings/Settings.h"
#include "misc/Clipboard.h"

namespace
{
    // Used in session management to save currently edited files.
    const char *SAVE_APPEND = "_saved.tse3";

    // Toolbar names
    const char *TOOLBAR_MAIN    = "mainToolBar";
    const char *TOOLBAR_EDIT    = "edit";
    const char *TOOLBAR_POINTER = "pointer";
    const char *TOOLBAR_OTHER   = "misc";
    const char *TOOLBAR_TX      = "transport";
    const char *TOOLBAR_TIME    = "time";
    const char *TOOLBAR_MARKER  = "marker";
    const char *TOOLBAR_FLAG    = "flag";

    const char *XMLGUIFILE      = "SongWindow.rc";
}

/******************************************************************************
 * SongWindow class
 *****************************************************************************/

SongWindow::SongWindow(TSE3::Song *s, const QString &filename, const KURL &url)
: KMainWindow(0, "SongWindow"),
  filename(filename), fileurl(url), midiFileFormat(-1),
  selectionStatus(NothingSelected),
  _viewOptions(SongWindowList::songWindowList()->swViewOptions()),
  song(s ? s : Application::application()->addSong()),
  history(Application::application()->history(song)),
  snap(song->timeSigTrack()),
  partSelection(new TSE3::App::PartSelection),
  trackSelection(new TSE3::App::TrackSelection),
  songInfoWindow(0), phraseListWindow(0), aboutWindow(0),
  newPhraseWindow(0),
  clockWindow(0), gotoWindow(0), zoomWindow(0)
{
    Application::application()->registerSongWindow(this, song);

    /**************************************************************************
     * TSE3 setup
     *************************************************************************/

    if (!history)
    {
        std::cerr << "Invalid CommandHistory - we will probably fall over\n";
    }

    /**************************************************************************
     * Cooperating child widgets
     *************************************************************************/

    splitter     = new QSplitter(this);
    splitter->setOpaqueResize(TRUE);
    setCentralWidget(splitter);
    QVBox *qvbox1 = new QVBox(splitter);
    trackList     = new TrackList(song, qvbox1, trackSelection);
    QVBox *qvbox2 = new QVBox(splitter);
    timeLine      = new TimeLine(song, qvbox2, snap);
    partView      = new PartView(song, qvbox2, timeLine, snap,
                                 partSelection, trackSelection);

    // You cann't set up the splitter sizes here (with setSizes), it just all
    // goes wrong and random widgets gets resized by mistake. The way to
    // force the splitter view sizes is to catch the show() function, and set
    // the values up in there.

    connect(partView, SIGNAL(contentsMoving(int,int)),
            trackList, SLOT(move(int,int)));
    connect(partView, SIGNAL(contentsMoving(int,int)),
            timeLine, SLOT(move(int,int)));
    connect(trackList, SIGNAL(contentsMoving(int, int)),
            partView, SLOT(moveVertical(int,int)));
    connect(timeLine, SIGNAL(contentsMoving(int,int)),
            partView, SLOT(moveHorizontal(int,int)));
    connect(trackList, SIGNAL(headerHeightSet()),
            SLOT(slotTrackListHeaderHeightSet()));
    connect(timeLine, SIGNAL(widthChanged(int,bool)),
            partView, SLOT(slotChangeWidth(int,bool)));
    connect(partView, SIGNAL(zoomWidget()),        SLOT(slotOpenZoom()));
    connect(partView, SIGNAL(zoomInHorizontal()),  SLOT(zoomInHorizontal()));
    connect(partView, SIGNAL(zoomOutHorizontal()), SLOT(zoomOutHorizontal()));
    connect(partView, SIGNAL(zoomInVertical()),    SLOT(zoomInVertical()));
    connect(partView, SIGNAL(zoomOutVertical()),   SLOT(zoomOutVertical()));
    connect(partView, SIGNAL(rmbMenuOpened(size_t,TSE3::Part*)),
            SLOT(slotRmbMenuOpened(size_t,TSE3::Part*)));
    connect(trackList, SIGNAL(rmbMenuOpened(size_t)),
            SLOT(slotRmbMenuOpened(size_t)));
    connect(partView, SIGNAL(setClock(TSE3::Clock,Qt::ButtonState)),
            SLOT(slotSetClock(TSE3::Clock,Qt::ButtonState)));
    connect(timeLine, SIGNAL(setClock(TSE3::Clock,Qt::ButtonState)),
            SLOT(slotSetClock(TSE3::Clock,Qt::ButtonState)));

    setAcceptDrops(TRUE);

    minZoomVertical = trackList->itemHeight();
    maxZoomVertical = minZoomVertical * 10;

    timeLine->recalculateWidthAndEmit();
    //setHorizontalZoom(DEFAULT_HZOOM);
    setVerticalZoom(minZoomVertical);

#if 0
    /**************************************************************************
     * Menus and toolbars
     *************************************************************************/

    /**************************************************************************
     * Transport toolbar
     *************************************************************************/

    transportToolbar = toolBar(TOOLBAR_TX);

    toolBar(TOOLBAR_TX)->insertButton(UserIcon("a_stop"), Transport_Stop,
                                      true, i18n("Stop"));
    toolBar(TOOLBAR_TX)->insertButton(UserIcon("a_play"), Transport_Play,
                                      true, i18n("Play"));
    toolBar(TOOLBAR_TX)->insertButton(UserIcon("a_record"), Transport_Record,
                                      true, i18n("Record"));
    toolBar(TOOLBAR_TX)->insertButton(UserIcon("a_gobck"), Transport_Rew,
                                      true, i18n("Rewind"));
    toolBar(TOOLBAR_TX)->setAutoRepeat(Transport_Rew, true);
    toolBar(TOOLBAR_TX)->insertButton(UserIcon("a_gofwd"), Transport_FF,
                                      true, i18n("Forward"));
    toolBar(TOOLBAR_TX)->setAutoRepeat(Transport_FF, true);

    toolBar(TOOLBAR_TX)->insertLineSeparator();

    toolBar(TOOLBAR_TX)->insertButton(UserIcon("synchro"),
                                      Transport_Synchro, true, i18n("Synchro"));
    toolBar(TOOLBAR_TX)->setToggle(Transport_Synchro, true);
    toolBar(TOOLBAR_TX)->insertButton(UserIcon("punchin"),
                                      Transport_PunchIn, true,
                                      i18n("Punch-in"));
    toolBar(TOOLBAR_TX)->setToggle(Transport_PunchIn, true);
    toolBar(TOOLBAR_TX)->setButton
        (Transport_Synchro,
         Application::application()->transport()->synchro());



    /**************************************************************************
     * Marker toolbar
     *************************************************************************/

    toolBar(TOOLBAR_MARKER)->insertButton(UserIcon("repeat"), Marker_Repeat,
                                          true, i18n("Repeat"));
    toolBar(TOOLBAR_MARKER)->setToggle(Marker_Repeat, true);
    toolBar(TOOLBAR_MARKER)->setButton(Marker_Repeat, song->repeat());

#endif

    /**************************************************************************
     * Create toolbars and menu actions for XMLGUI
     *************************************************************************/

    // File menu
    KStdAction::openNew(this, SLOT(slotOpenNew()), actionCollection());
    KStdAction::open(this, SLOT(slotOpen()), actionCollection());
    recentFilesAction
        = KStdAction::openRecent(this, SLOT(slotOpenRecent(const KURL &)),
                                 actionCollection());
    recentFilesAction->loadEntries(KGlobal::config());
    new KAction(i18n("&Merge"), 0,
                this, SLOT(slotMerge()), actionCollection(), "merge");
    new KAction(i18n("&Append"), 0,
                this, SLOT(slotAppend()), actionCollection(), "append");

    KStdAction::save(this, SLOT(slotSave()), actionCollection());
    KStdAction::saveAs(this, SLOT(slotSaveAs()), actionCollection());
    KStdAction::revert(this, SLOT(slotRevert()), actionCollection());
    new KAction(i18n("Song &info"), 0,
                this, SLOT(slotSongInfoWindow()), actionCollection(),
                "song_info");

    KStdAction::close(this, SLOT(close()), actionCollection());
    KStdAction::quit(kapp, SLOT(quit()), actionCollection());

    // Edit menu
    undoAction  = KStdAction::undo(this, SLOT(slotUndo()), actionCollection());
    redoAction  = KStdAction::redo(this, SLOT(slotRedo()), actionCollection());
    cutAction   = KStdAction::cut(this, SLOT(slotCut()), actionCollection());
    copyAction  = KStdAction::copy(this, SLOT(slotCopy()), actionCollection());
    pasteAction = KStdAction::paste(this, SLOT(slotPaste()),
                                    actionCollection());

    clearHistoryAction = new KAction(i18n("&Clear history"), 0,
                                     this, SLOT(slotClearHistory()),
                                     actionCollection(), "edit_clearhistory");

    // We must create these popup menus before we call createGUIAndPatchActions
    undoMenu = new QPopupMenu(this);
    connect(undoMenu, SIGNAL(aboutToShow()), SLOT(slotAboutToShowUndo()));
    connect(undoMenu, SIGNAL(activated(int)),
           SLOT(slotUndoActivated(int)));
    redoMenu = new QPopupMenu(this);
    connect(redoMenu, SIGNAL(aboutToShow()), SLOT(slotAboutToShowRedo()));
    connect(redoMenu, SIGNAL(activated(int)),
           SLOT(slotRedoActivated(int)));

    // Track menu
    new KAction(i18n("&Insert"), 0, this, SLOT(slotInsertTrack()),
                actionCollection(), "track_insert");
    new KAction(i18n("&Sort"), 0, this, SLOT(slotSortTracks()),
                actionCollection(), "track_sort");

    // Part menu
    new KAction(i18n("&Export"), 0, this, SLOT(slotPartExport()),
                actionCollection(), "part_export");
    new KAction(i18n("Change &position"), 0, this, SLOT(slotPartPosition()),
                actionCollection(), "part_changepos");
    fromMarkerStartAction =
        new KAction(i18n("&From marker at start"), "setleftmarker", 0,
                    this, SLOT(slotPartFromStart()),
                    actionCollection(), "part_frommarker_start");
    fromMarkerEndAction =
        new KAction(i18n("F&rom marker at end"), "setleftmarker", 0,
                    this, SLOT(slotPartFromEnd()),
                    actionCollection(), "part_frommarker_end");
    toMarkerStartAction =
        new KAction(i18n("T&o marker at start"), "setrightmarker", 0,
                    this, SLOT(slotPartToStart()),
                    actionCollection(), "part_tomarker_start");
    toMarkerEndAction =
        new KAction(i18n("&To marker at start"), "setrightmarker", 0,
                    this, SLOT(slotPartToEnd()),
                    actionCollection(), "part_tomarker_end");
    new KAction(i18n("&Quantise"), 0, this, SLOT(slotPartQuantise()),
                actionCollection(), "part_quantise");
    new KAction(i18n("&Remix"), 0, this, SLOT(slotPartRemix()),
                actionCollection(), "part_remix");
    new KAction(i18n("&Partition"), 0, this, SLOT(slotPartPartition()),
                actionCollection(), "part_partition");
    new KAction(i18n("&Extract"), 0, this, SLOT(slotPartExtract()),
                actionCollection(), "part_extract");
    new KAction(i18n("&Subdivide"), 0, this, SLOT(slotPartSubdivide()),
                actionCollection(), "part_subdivide");
    new KAction(i18n("S&tretch"), 0, this, SLOT(slotPartStretch()),
                actionCollection(), "part_stretch");

    // Gadgets menu
    new KAction(i18n("&Studio clock"), "time", 0, this, SLOT(slotClockWindow()),
                actionCollection(), "gadgets_clock");
    new KAction(i18n("&VU bars"), 0, this, SLOT(slotVuWindow()),
                actionCollection(), "gadgets_vu");
    new KAction(i18n("On screen &Keyboard"), "osmkeyboard", 0, this,
                SLOT(slotKeyboardWindow()), actionCollection(), "gadgets_keys");
    KAction *action
        = KStdAction::goTo(this, SLOT(slotGotoWindow()), actionCollection());
    action->setIcon("goto"); // we need to do this for some reason :-(
    new KAction(i18n("&Panic"), "panic", 0, this, SLOT(slotPanicWindow()),
                actionCollection(), "gadgets_panic");

    // Settings menus
    KStdAction::keyBindings(this, SLOT(slotKeyBindings()), actionCollection());
    KStdAction::configureToolbars(this, SLOT(slotConfigureToolbars()),
                                  actionCollection());
    KStdAction::preferences(this, SLOT(slotSettingsWindow()),
                            actionCollection());

    // Help menu
    new KAction(i18n("Tip of the &day"), "idea", 0,
                this, SLOT(slotShowTip()), actionCollection(),
                "tipoftheday");

    // Misc toolbar
    new KAction(i18n("&Phrase list"), "phraselist", 0,
                this, SLOT(slotPhraseListWindow()),
                actionCollection(), "phraselist");
    snapWidget = new SnapWidget(0);
    connect(snapWidget, SIGNAL(valueChanged(int)), SLOT(slotSetSnap(int)));
    new QWidgetAction("Snap", snapWidget, actionCollection(), "snap");
    piaWidget = new PartInsertActionWidget(0);
    connect(piaWidget, SIGNAL(partInsertActionChanged(int)),
            partView, SLOT(slotSetPartInsertAction(int)));
    new QWidgetAction("Part insert action", piaWidget, actionCollection(),
                      "partinsert");

    // Pointer toolbar
    selectAction = new KRadioAction(i18n("&Select mode"), "select", Key_S,
                                    this, SLOT(slotPointerSelect()),
                                    actionCollection(), "pointer_select");
    snipAction   = new KRadioAction(i18n("S&nip mode"), "snip", Key_N,
                                    this, SLOT(slotPointerSnip()),
                                    actionCollection(), "pointer_snip");
    glueAction   = new KRadioAction(i18n("&Glue mode"), "glue", Key_G,
                                    this, SLOT(slotPointerGlue()),
                                    actionCollection(), "pointer_glue");
    insertAction = new KRadioAction(i18n("&Insert mode"), "part", Key_I,
                                    this, SLOT(slotPointerInsert()),
                                    actionCollection(), "pointer_insert");
    slotPointerModeChanged();

    // Transport Toolbar
    stopAction    = new RepeatToggleAction(i18n("&Stop"), "a_stop", Key_0, this,
                                     SLOT(slotTransportStop()),
                                     actionCollection(), "transport_stop");
    playAction    = new RepeatToggleAction(i18n("&Play"), "a_play", Key_Space, this,
                                     SLOT(slotTransportPlay()),
                                     actionCollection(), "transport_play");
    recordAction  = new RepeatToggleAction(i18n("&Record"), "a_record", Key_R, this,
                                     SLOT(slotTransportRecord()),
                                     actionCollection(), "transport_record");
    rewAction     = new AutoRepeatAction(i18n("Re&wind"), "a_gobck", Key_4,
                                         this, SLOT(slotTransportRew()),
                                         actionCollection(), "transport_rew");
    ffAction      = new AutoRepeatAction(i18n("&Forward"), "a_gofwd", Key_6,
                                         this, SLOT(slotTransportFF()),
                                         actionCollection(), "transport_ff");
    synchroAction = new KRadioAction(i18n("S&ynchro"), "synchro", 0, this,
                                     SLOT(slotTransportSynchro()),
                                     actionCollection(), "transport_synchro");
    punchinAction = new KRadioAction(i18n("P&unch in"), "punchin", 0, this,
                                     SLOT(slotTransportPunchIn()),
                                     actionCollection(), "transport_punchin");
    new KAction(i18n("Go to &start"), Key_7, this, SLOT(slotGoStart()),
                actionCollection(), "go_start");
    new KAction(i18n("Go to &end"), Key_8, this, SLOT(slotGoEnd()),
                actionCollection(), "go_end");
    new KAction(i18n("Go to &from"), 0, this, SLOT(slotGoFrom()),
                actionCollection(), "go_from");
    new KAction(i18n("Go to &to"), 0, this, SLOT(slotGoTo()),
                actionCollection(), "go_to");

    // Time toolbar
    QFrame *frame = new QFrame(0);
    frame->setFrameStyle(QFrame::Panel|QFrame::Sunken);
    frame->setLineWidth(1);
    QHBoxLayout *flayout = new QHBoxLayout(frame, frame->frameWidth());
    MidiSchedulerTimeWidget *tdw
        = new MidiSchedulerTimeWidget(frame,
                                      Application::application()->scheduler());
    flayout->addWidget(tdw);
    new QWidgetAction("Time", frame, actionCollection(), "time_display");

    msTimeSlider
        = new MidiSchedulerTimeSlider(0,
                                      Application::application()->scheduler(),
                                      song, &snap);
    new QWidgetAction("Time slider", msTimeSlider, actionCollection(),
                      "time_slider");

    // Marker toolbar
    new KAction(i18n("Set &from marker"), "setleftmarker", 0,
                this, SLOT(slotMarkerFrom()),
                actionCollection(), "marker_setfrom");
    fromDisplay = new ClockDisplayWidget(song->from(), 0);
    new QWidgetAction("From marker", fromDisplay,
                      actionCollection(), "marker_displayfrom");
    toDisplay = new ClockDisplayWidget(song->to(), 0);
    new QWidgetAction("From marker", toDisplay,
                      actionCollection(), "marker_displayto");
    new KAction(i18n("Set &to marker"), "setrightmarker", 0,
                this, SLOT(slotMarkerTo()),
                actionCollection(), "marker_setto");
    bothMarkersAction =
        new KAction(i18n("Set &both markers"), "setmarkers", 0,
                    this, SLOT(slotMarkerBoth()),
                    actionCollection(), "marker_setboth");

    new KToggleAction(i18n("&Repeat"), "repeat", 0,
                this, SLOT(slotMarkerRepeat()),
                actionCollection(), "marker_repeat");

    // Flag toolbar
    flagCombo = new FlagComboBox(song->flagTrack(), 0, "flag_combo");
    flagCombo->setMidiScheduler(Application::application()->scheduler());
    connect(flagCombo, SIGNAL(activated(int)),
            SLOT(slotFlagComboActicated(int)));
    new QWidgetAction("Flag", flagCombo, actionCollection(), "flag_combo");
    new KAction(i18n("&Previous flag"), "flag_prev", 0,
                this, SLOT(slotFlagPrev()),
                actionCollection(), "flag_prev");
    new KAction(i18n("&Next flag"), "flag_next", 0,
                this, SLOT(slotFlagNext()),
                actionCollection(), "flag_next");
    new KAction(i18n("&Add flag"), "flag_add", 0,
                this, SLOT(slotFlagAdd()),
                actionCollection(), "flag_add");
    new KAction(i18n("&List flags"), "flag_list", 0,
                this, SLOT(slotFlagList()),
                actionCollection(), "flag_list");

    // RMB menu
    new KAction(i18n("&Insert track"), 0,
                this, SLOT(slotInsertTrackRMB()),
                actionCollection(), "rmb_insert");
    propertiesAction = new KAction(i18n("&Properties"), 0,
                                   this, SLOT(slotRMBProperties()),
                                   actionCollection(), "rmb_properties");
    new KAction(i18n("Select &all"), 0, partView, SLOT(slotSelectAll()),
                actionCollection(), "select_all");
    new KAction(i18n("Select &none"), 0, partView, SLOT(slotSelectNone()),
                actionCollection(), "select_none");
    new KAction(i18n("&Invert selection"), 0,
                partView, SLOT(slotSelectInvert()),
                actionCollection(), "select_invert");
    new KAction(i18n("In&side markers"), 0,
                partView, SLOT(slotSelectInside()),
                actionCollection(), "select_inside");
    new KAction(i18n("&Outside markers"), 0,
                partView, SLOT(slotSelectOutside()),
                actionCollection(), "select_outside");
    new KAction(i18n("&Outside markers"), 0,
                partView, SLOT(slotSelectOutside()),
                actionCollection(), "select_outside");
    new KAction(i18n("All &parts in track"), 0,
                this, SLOT(slotSelectAllInTrack()),
                actionCollection(), "select_allpartsintrack");
    new KAction(i18n("All parts in selected &tracks"), 0,
                this, SLOT(slotSelectAllInSelectedTracks()),
                actionCollection(), "select_allpartsinselectedtracks");

    /**************************************************************************
     * Having created the actions, create the GUI
     *************************************************************************/

    createGUIAndPatchActions(XMLGUIFILE);

    rmbMenu = static_cast<QPopupMenu*>
                  (factory()->container("context_popup", this));
    if (rmbMenu)
    {
        partView->setRMBMenu(rmbMenu);
        trackList->setRMBMenu(rmbMenu);
    }
    else
    {
        std::cerr << "Failed to create SongWindow RMB menu\n";
    }

    /**************************************************************************
     * Other stuff
     *************************************************************************/

    doSetCaption();

    connect(Application::application()->clipboard(), SIGNAL(dataChanged()),
            SLOT(slotClipboardDataChanged()));
    updateCutCopy();                                  // set up copy/cutActions
    slotClipboardDataChanged();                       // set up pasteAction

    // Set up the initial states of the toolbar/menu entries
    CommandHistory_Undo(0);
    CommandHistory_Redo(0);

    // Set up toolbar icons
    Transport_Status(Application::application()->transport(),
                     Application::application()->transport()->status());

    /**************************************************************************
     * TSE3 setup
     *************************************************************************/

    TSE3::Listener<TSE3::App::RecordListener>::attachTo
        (Application::application()->record());
    TSE3::Listener<TSE3::TransportListener>::attachTo
        (Application::application()->transport());
    TSE3::Listener<TSE3::Cmd::CommandHistoryListener>::attachTo(history);

    partSelection->TSE3::Listener<TSE3::App::TrackSelectionListener>::attachTo
        (trackSelection);
    trackSelection->TSE3::Listener<TSE3::App::PartSelectionListener>::attachTo
        (partSelection);
    TSE3::Listener<TSE3::App::TrackSelectionListener>::attachTo(trackSelection);
    TSE3::Listener<TSE3::App::PartSelectionListener>::attachTo(partSelection);
    TSE3::Listener<TSE3::App::ModifiedListener>::attachTo(&modified);
    modified.setSong(song);

    // Finally, add ourselves to the SongWindowList
    SongWindowList::songWindowList()->add(this);
}


SongWindow::~SongWindow()
{
    Application::application()->deregisterSongWindow(this);

    SongWindowList::songWindowList()->remove(this);

    recentFilesAction->saveEntries(KGlobal::config());

    delete trackList; // These two would get automatically deleted, however
    delete partView;  // I delete the two selection objects, and so must delete
                      // the widgets first to prevent them using it (unlikely,
                      // but it's best to be safe).

    delete trackSelection;
    delete partSelection;

    delete song;
}


void SongWindow::doSetCaption()
{
    setCaption(fileurl.prettyURL(), modified.modified());
}


void SongWindow::show()
{
    // You have to show first, it causes the splitter to do some good setup
    // stuff that makes setSizes work. You also then have to call refresh, or
    // it doesn't know it's down width for some weird reason.
    KMainWindow::show();
    splitter->refresh();

    QValueList<int> sizes = splitter->sizes();
    sizes[0] = trackList->preferredShowWidth();
    sizes[1] = splitter->width() - sizes[0];
    splitter->setSizes(sizes);
}


void SongWindow::changeSong(TSE3::Song *s, const QString &f, const KURL &url)
{
    // 1. Out with the old

    Application::application()->deregisterSongWindow(this);
    TSE3::Listener<TSE3::Cmd::CommandHistoryListener>::detachFrom(history);
    timeLine->changeSong(s);
    trackList->changeSong(s);
    partView->changeSong(s);
    if (gotoWindow)
    {
        gotoWindow->changeSong(s);
    }
    if (songInfoWindow)
    {
        delete songInfoWindow;
        songInfoWindow = 0;
    }
    if (phraseListWindow)
    {
        phraseListWindow->changeSong(s);
    }
    snap.setTimeSigTrack(0);

    modified.setSong(0);

    delete song;

    // 2. In with the new

    song           = s;
    history        = Application::application()->history(song);
    filename       = f;
    fileurl        = url;
    midiFileFormat = -1;
    TSE3::Listener<TSE3::Cmd::CommandHistoryListener>::attachTo(history);
    snap.setTimeSigTrack(song->timeSigTrack());

    // alter toolbar buttons and KActions
    CommandHistory_Undo(0); // share window setup code?
    CommandHistory_Redo(0);

    msTimeSlider->setSong(song);
    fromDisplay->setValue(song->from());
    toDisplay->setValue(song->to());
    updateCutCopy();
    flagCombo->setFlagTrack(song->flagTrack());

    modified.setSong(song);
    doSetCaption();
    Application::application()->registerSongWindow(this, song);
}


void SongWindow::slotTrackListHeaderHeightSet()
{
    timeLine->setMaximumHeight(trackList->headerHeight());
    timeLine->setMinimumHeight(trackList->headerHeight());
}


void SongWindow::createGUIAndPatchActions(const QString &xmlfile,
                                          bool _conserveMemory)
{
    // Remove the toolbar list
    unplugActionList("toolbarlist");

    // XMLGUI magic
    createGUI(xmlfile, _conserveMemory);

    // Patch undo action
    for (int n = 0; n < undoAction->containerCount(); ++n)
    {
        QWidget *cw = undoAction->container(n);
        if (cw->inherits("KToolBar"))
        {
            KToolBar *bar = static_cast<KToolBar*>(cw);
            bar->setDelayedPopup(undoAction->itemId(n), undoMenu);
        }
    }

    // Patch redo action
    for (int n = 0; n < redoAction->containerCount(); ++n)
    {
        QWidget *cw = redoAction->container(n);
        if (cw->inherits("KToolBar"))
        {
            KToolBar *bar = static_cast<KToolBar*>(cw);
            bar->setDelayedPopup(redoAction->itemId(n), redoMenu);
        }
    }

    // Sort out the toolbar list
std::cout << "setting up toolbars\n";
    toolbarActionList.clear();
    QListIterator<KToolBar> bar(toolBarIterator());
    for (; bar.current() ; ++bar)
    {
std::cout << "toolbar\n";
      KToggleAction *act
         = new KToggleAction(i18n("Show %1").arg(bar.current()->text()),
                             0, actionCollection(), bar.current()->name());
      std::cout << act->text().latin1() << "\n";
      connect(act, SIGNAL(toggled(bool)), this, SLOT(slotToolbarToggled(bool)));
      act->setChecked(!bar.current()->isHidden());
      toolbarActionList.append(act);
    }
    plugActionList(QString::fromLatin1("toolbarlist"), toolbarActionList);
    toolbarActionList.setAutoDelete(true);
}


QSize SongWindow::sizeHint() const
{
    return QSize(640, 480);
}


/******************************************************************************
 * User interface slots (menu and tool bars)
 *****************************************************************************/

void SongWindow::slotOpenNew()
{
    if (modified.modified())
    {
        (new SongWindow)->show();
    }
    else
    {
        Application::application()->load("", "", this);
    }
}


void SongWindow::slotOpen()
{
    KURL url = KFileDialog::getOpenURL(0, "*.tse3|TSE3MDL files\n*.mid *.midi *.MID *.MIDI|Standard MIDI files\n*.tse2|TSE2MDL files\n*|All files", this);
    QString s;

    if (!KIO::NetAccess::download(url, s))
    {
        KNotifyClient::event("cannotopenfile");
        return;
    }

    doOpen(s, url);
}


bool SongWindow::doOpen(const QString &s, const KURL &url)
{
    if (!s.isEmpty())
    {
        if (modified.modified())
        {
            // If this song empty put it in this window
            Application::application()->load(s, url);
        }
        else
        {
            Application::application()->load(s, url, this);
        }
        //XXX Application::application()->aRecentFilesAction()->addURL(url);
        recentFilesAction->addURL(url);
        return true;
    }
    return false;
}


void SongWindow::slotOpenRecent(const KURL &u)
{
    QString s = u.path();
    if (!u.isLocalFile())
    {
        KTempFile temp;
        s = temp.name();
        doSave(s, u);
        temp.unlink();
    }
    else
    {
        doOpen(s, u);
    }
}


void SongWindow::slotSave()
{
    if (!filename)
    {
        slotSaveAs();
    }
    else
    {
        doSave(filename, fileurl, midiFileFormat);
    }
}


void SongWindow::slotSaveAs()
{
    doSaveAs();
}


bool SongWindow::doSaveAs()
{
    KURL u = KFileDialog::getSaveURL(fileurl.path(), "*.tse3|TSE3MDL files\n*.mid *.midi|Standard MIDI files\n*.tse2|TSE2MDL files\n*", this);
    QString s = u.path();

    if (!u.isEmpty())
    {
        if (s.find(QRegExp("\\..+$")) == -1)
        {
            s.append(".tse3");
            u = KURL(s);
        }
    }
    else
    {
        return false;
    }

    if (!u.isLocalFile())
    {
        KTempFile temp;
        s = temp.name();
        doSave(s, u);
        temp.unlink();
    }
    else
    {
        doSave(s, u);
    }
    return true;
}


void SongWindow::slotRevert()
{
    if (filename && modified.modified())
    {
        if (messageBoxSave())
        {
            Application::application()->load(filename, fileurl, this);
        }
    }
}


void SongWindow::slotMerge()
{
    std::cout << "TODO Merge\n";
}


void SongWindow::slotAppend()
{
    std::cout << "TODO Append\n";
}


void SongWindow::slotShowTip()
{
    KTipDialog::KTipDialog::showTip(locate("appdata", "tips"), true);
}


void SongWindow::slotSongInfoWindow()
{
    if (songInfoWindow)
    {
        // raise the window!
        return;
    }

    songInfoWindow = new SongInfoWindow(this, song);
    connect(songInfoWindow, SIGNAL(closed()), SLOT(slotSongInfoWindowClosed()));

    // The SongInfoWindow is deleted either by slotSongInfoWindowClosed or by
    // Qt in the destructor.
}


void SongWindow::slotSongInfoWindowClosed()
{
    if (songInfoWindow)
    {
        songInfoWindow->delayedDestruct();
        songInfoWindow = 0;
    }
}


void SongWindow::slotUndo()
{
    history->undo();
}


void SongWindow::slotRedo()
{
    history->redo();
}


void SongWindow::slotClearHistory()
{
    history->clear();
}


void SongWindow::slotCut()
{
    switch (selectionStatus)
    {
        case NothingSelected:
        {
            break;
        }
        case PartSelected:
        {
            partView->slotCut();
            break;
        }
        case TrackSelected:
        {
            trackList->slotCut();
            break;
        }
    }
}


void SongWindow::slotCopy()
{
    switch (selectionStatus)
    {
        case NothingSelected:
        {
            break;
        }
        case PartSelected:
        {
            partView->slotCopy();
            break;
        }
        case TrackSelected:
        {
            trackList->slotCopy();
            break;
        }
    }
}


void SongWindow::slotPaste()
{
    switch (clipboardData())
    {
        case Clipboard_None:
        {
            break;
        }
        case Clipboard_Phrase:
        {
            phraseListWindow->slotPaste();
            break;
        }
        case Clipboard_Part:
        {
            partView->slotPaste();
            break;
        }
        case Clipboard_Track:
        {
            trackList->slotPaste();
            break;
        }
    }
}


void SongWindow::slotPointerSelect()
{
    partView->slotSetPointerMode(PartView::Select);
    slotPointerModeChanged();
}


void SongWindow::slotPointerSnip()
{
    partView->slotSetPointerMode(PartView::Snip);
    slotPointerModeChanged();
}


void SongWindow::slotPointerGlue()
{
    partView->slotSetPointerMode(PartView::Glue);
    slotPointerModeChanged();
}


void SongWindow::slotPointerInsert()
{
    partView->slotSetPointerMode(PartView::Insert);
    slotPointerModeChanged();
}


void SongWindow::slotTransportStop()
{
    Application::application()->transport()->stop();
    if (!toolBar(TOOLBAR_TX)->isButtonOn(Transport_Stop))
    {
        // This is a bit nasty, but stop is the only icon we have
        // to watch out for being erroneously deselected.
        toolBar(TOOLBAR_TX)->setButton(Transport_Stop, true);
    }
}


void SongWindow::slotTransportPlay()
{
    Application::application()->transport()->play
        (song, Application::application()->scheduler()->clock());
}


void SongWindow::slotTransportRecord()
{
    TSE3::Track *t = trackSelection->front();
    Application::application()->record()->start(song, t);
}


void SongWindow::slotTransportRew()
{
    Application::application()->transport()->rew(false);
}


void SongWindow::slotTransportFF()
{
    Application::application()->transport()->ff(false);
}


void SongWindow::slotTransportSynchro()
{
    Application::application()->transport()->setSynchro
        (toolBar(TOOLBAR_TX)->isButtonOn(Transport_Synchro));
}


void SongWindow::slotTransportPunchIn()
{
    Application::application()->transport()->setPunchIn
        (toolBar(TOOLBAR_TX)->isButtonOn(Transport_PunchIn));
}


void SongWindow::slotMarkerFrom()
{
    song->setFrom(Application::application()->scheduler()->clock());
}


void SongWindow::slotMarkerTo()
{
    song->setTo(Application::application()->scheduler()->clock());
}


void SongWindow::slotMarkerBoth()
{
    if (partSelection->size())
    {
        song->setFrom(partSelection->earliest());
        song->setTo(partSelection->latest());
    }
}


void SongWindow::slotMarkerRepeat()
{
//XXX    song->setRepeat(repeatAction->isChecked());
}


void SongWindow::slotFlagPrev()
{
    TSE3::Clock now  = Application::application()->scheduler()->clock();
    size_t      index= song->flagTrack()->index(now, false);

    if (index < song->flagTrack()->size())
    {
        TSE3::Clock prevClock = (*song->flagTrack())[index].time;
        if (prevClock == now && index == 0)
        {
            prevClock = 0;
        }
        else if (prevClock == now)
        {
            now -= 1;
            index     = song->flagTrack()->index(now, false);
            prevClock = (*song->flagTrack())[index].time;
        }
        Application::application()->scheduler()->moveTo(prevClock);
        flagCombo->setCurrentItem(index);
    }
    else
    {
        Application::application()->scheduler()->moveTo(0);
        flagCombo->setCurrentItem(-1);
    }
}


void SongWindow::slotFlagNext()
{
    TSE3::Clock now  = Application::application()->scheduler()->clock();
    size_t      index= song->flagTrack()->index(now, true);

    bool done = false;
    if (index < song->flagTrack()->size())
    {
        done = true;
        TSE3::Clock nextClock = (*song->flagTrack())[index].time;
        if (nextClock == now)
        {
            now += 1;
            index     = song->flagTrack()->index(now, true);
            if (index < song->flagTrack()->size())
            {
                nextClock = (*song->flagTrack())[index].time;
            }
            else
            {
                done = false;
            }
        }
        if (done)
        {
            Application::application()->scheduler()->moveTo(nextClock);
            flagCombo->setCurrentItem(index);
        }
    }
    if (!done && now < song->lastClock())
    {
        Application::application()->scheduler()
            ->moveTo(song->lastClock());
        flagCombo->setCurrentItem(-1);
    }
}


void SongWindow::slotFlagAdd()
{
    NewFlagWindow *nfw
        = new NewFlagWindow(this, song,
            Application::application()->scheduler()->clock());
    nfw->show();
}


void SongWindow::slotFlagList()
{
    std::cout << "TODO: Edit flags\n";
}


void SongWindow::dropEvent(QDropEvent *event)
{
/*  // the user dropped something on our window.
  // So we simply use KIO slaves to download the stuff wherever it is.
  // The following code respects Qt's DnD tutorial

    QImage im;
    QString txt, u, tgt;
    QStrList uri;

    if (QImageDrag::decode(event, im))
    {
        QMessageBox::critical (this, i18n("KLess: error"),
            i18n("This application accepts only URL drops"),
            QMessageBox::Ok, 0);
    }

    if (QTextDrag::decode(event, txt))
    {
        setCaption("The KDE text viewer");
    }

    if(QUriDrag::decode(event, uri))
    {
        u = uri.first();
        if (KIO::NetAccess::download(u, tgt))
        {
            setCaption(u);
            loadFile(tgt);
            KIO::NetAccess::removeTempFile(tgt);
        }
    }
*/
}


void SongWindow::slotPhraseListWindow()
{
    if (!phraseListWindow)
    {
        phraseListWindow = new PhraseListWindow(this, song);
        connect(partView, SIGNAL(partSelected(TSE3::Part*)),
                phraseListWindow, SLOT(slotPartSelected(TSE3::Part*)));
        connect(phraseListWindow, SIGNAL(phraseSelected(TSE3::Phrase*)),
                partView, SLOT(slotPhraseSelected(TSE3::Phrase*)));
        partView->setPhraseListWindow(phraseListWindow);
        // The PhraseListWindow will be deleted by Qt in the destructor.
    }
    phraseListWindow->show();
}


void SongWindow::slotClockWindow()
{
    if (!clockWindow)
    {
        clockWindow
            = new ClockWindow(this);
        // The ClockWindow will be deleted by Qt in the destructor.
    }
    clockWindow->show();
}


void SongWindow::slotVuWindow()
{
    Application::application()->vuWindow()->show();
}


void SongWindow::slotKeyboardWindow()
{
    Application::application()->keyboardWindow()->show();
}


void SongWindow::slotPanicWindow()
{
    Application::application()->panicWindow()->show();
}


void SongWindow::slotGotoWindow()
{
    if (!gotoWindow)
    {
        gotoWindow = new GotoWindow(this, song);
        // The GotoWindow will be deleted by Qt in the destructor.
    }
    gotoWindow->show();
}


void SongWindow::slotKeyBindings()
{
    KKeyDialog::configureKeys(actionCollection(), xmlFile());
    /*
    if (KKeyDialog::configureKeys(accel(), false, this))
    {
    }
    */
}


void SongWindow::slotConfigureToolbars()
{
    KEditToolbar dlg(actionCollection(), XMLGUIFILE);
    if (dlg.exec())
    {
        createGUIAndPatchActions(XMLGUIFILE);
    }
}


void SongWindow::slotSettingsWindow()
{
    Application::application()->settingsWindow()->show();
}


void SongWindow::slotPartExport()
{
    std::cout << "TODO\n";
}


void SongWindow::slotPartPosition()
{
    std::cout << "TODO\n";
}


void SongWindow::slotPartFromStart()
{
    if (partSelection->size())
    {
        song->setFrom(partSelection->earliest());
    }
}


void SongWindow::slotPartFromEnd()
{
    if (partSelection->size())
    {
        song->setFrom(partSelection->latest());
    }
}


void SongWindow::slotPartToStart()
{
    if (partSelection->size())
    {
        song->setTo(partSelection->earliest());
    }
}


void SongWindow::slotPartToEnd()
{
    if (partSelection->size())
    {
        song->setTo(partSelection->latest());
    }
}


void SongWindow::slotPartQuantise()
{
    std::cout << "TODO\n";
}


void SongWindow::slotPartRemix()
{
    std::cout << "TODO\n";
}


void SongWindow::slotPartPartition()
{
    std::cout << "TODO\n";
}


void SongWindow::slotPartExtract()
{
    std::cout << "TODO\n";
}


void SongWindow::slotPartSubdivide()
{
    std::cout << "TODO\n";
}


void SongWindow::slotPartStretch()
{
    std::cout << "TODO\n";
}


void SongWindow::slotGoStart()
{
    Application::application()->scheduler()->moveTo(0);
}


void SongWindow::slotGoEnd()
{
    Application::application()->scheduler()->moveTo(song->lastClock());
}


void SongWindow::slotGoFrom()
{
    Application::application()->scheduler()->moveTo(song->from());
}


void SongWindow::slotGoTo()
{
    Application::application()->scheduler()->moveTo(song->to());
}


/******************************************************************************
 * Other SongWindow slots
 *****************************************************************************/

void SongWindow::slotPointerModeChanged()
{
    selectAction->setChecked(partView->pointerMode() == PartView::Select);
    snipAction->setChecked(partView->pointerMode() == PartView::Snip);
    glueAction->setChecked(partView->pointerMode() == PartView::Glue);
    insertAction->setChecked(partView->pointerMode() == PartView::Insert);
}


void SongWindow::slotOpenZoom()
{
    if (!zoomWindow)
    {
        zoomWindow = new ZoomWindow(this, this);
    }
    zoomWindow->popup(QCursor::pos());
}


int SongWindow::zoomHorizontal()
{
    return timeLine->zoom();
}


void SongWindow::setZoomHorizontal(int pc)
{
    timeLine->setZoom(pc);
}


int SongWindow::zoomVertical()
{
    return (maxZoomVertical - _zoomVertical)
        * 100 / (maxZoomVertical-minZoomVertical);
}


void SongWindow::setZoomVertical(int pc)
{
    //int newZoom = minZoomVertical + (maxZoomVertical-minZoomVertical)*pc/100;
    int newZoom = maxZoomVertical - (maxZoomVertical-minZoomVertical)*pc/100;
    setVerticalZoom(newZoom);
}


void SongWindow::zoomInHorizontal()
{
    timeLine->zoomIn();
}


void SongWindow::zoomOutHorizontal()
{
    timeLine->zoomOut();
}


void SongWindow::zoomInVertical()
{
    int newZoom = _zoomVertical+10;
    if (newZoom > maxZoomVertical) newZoom = maxZoomVertical;
    setVerticalZoom(newZoom);
}


void SongWindow::zoomOutVertical()
{
    int newZoom = _zoomVertical-10;
    if (newZoom < minZoomVertical) newZoom = minZoomVertical;
    setVerticalZoom(newZoom);
}


void SongWindow::setHorizontalZoom(int newValue)
{
    std::cout << "SongWindow::setHorizontalZoom\n";
//    timeLine->setZoom(newValue);
}


void SongWindow::setVerticalZoom(int pixels)
{
    trackList->setItemHeight(pixels);
    partView->setItemHeight(pixels);
    _zoomVertical = pixels;
}


void SongWindow::showAboutApplication()
{
    if (!aboutWindow)
    {
        aboutWindow = new AboutWindow(this);
    }
    aboutWindow->show();
}


void SongWindow::Record_RecordingEnded(TSE3::App::Record *src,
                                       TSE3::Song        *rec_song,
                                       TSE3::Track       *track)
{
    if (rec_song == song && !newPhraseWindow)
    {
        newPhraseWindow = new NewPhraseWindow(this, song->phraseList(),
                                              "", true, track != 0);
        connect(newPhraseWindow,
                SIGNAL(newPhraseDone(bool,QString,bool,bool,bool)),
                SLOT(slotNewPhraseDone(bool,QString,bool,bool)));
        newPhraseWindow->show();
    }
}


void SongWindow::slotNewPhraseDone(bool success, QString title,
                                   bool replacePhrase, bool insertPart)
{
    if (success)
    {
        try
        {
            Application::application()->record()
                ->insertPhrase(title.latin1(), replacePhrase, insertPart,
                               partView->partInsertAction(), history);
        }
        catch (const TSE3::PhraseListError &e)
        {
            KMessageBox::error(this, "Application error when inserting Phrase");
            return;
        }
    }
    else
    {
        Application::application()->record()->reset();
    }
    newPhraseWindow->delayedDestruct();
    newPhraseWindow = 0;
}


void SongWindow::slotAboutToShowUndo()
{
    undoMenu->clear();
    for (size_t n = 0; n < 5; n++)
    {
        TSE3::Cmd::Command *command = history->undoCommand(n);
        if (command)
        {
            QString str;
            str.sprintf("Undo %d: %s", n+1, command->title().c_str());
            undoMenu->insertItem(str);
        }
    }
}


void SongWindow::slotAboutToShowRedo()
{
    redoMenu->clear();
    for (size_t n = 0; n < 5; n++)
    {
        TSE3::Cmd::Command *command = history->redoCommand(n);
        if (command)
        {
            QString str;
            str.sprintf("Redo %d: %s", n+1, command->title().c_str());
            redoMenu->insertItem(str);
        }
    }
}


void SongWindow::slotUndoActivated(int id)
{
    id = undoMenu->indexOf(id)+1;
    while (id--)
    {
        history->undo();
    }
}


void SongWindow::slotRedoActivated(int id)
{
    id = redoMenu->indexOf(id)+1;
    while (id--)
    {
        history->redo();
    }
}


void SongWindow::slotSetSnap(int s)
{
    snap.setSnap(s);
}


void SongWindow::slotClipboardDataChanged()
{
    pasteAction->setEnabled(clipboardData() != Clipboard_None);
}


void SongWindow::slotInsertTrack()
{
    new NewTrackWindow(trackSelection, song, this);
}


void SongWindow::slotInsertTrackRMB()
{
    TSE3::Cmd::Song_InsertTrack *cmd
        = new TSE3::Cmd::Song_InsertTrack(song, rmbMenuTrack);
    cmd->execute();
    history->add(cmd);
}


void SongWindow::slotRMBProperties()
{
    if (rmbMenuFromTrack)
    {
        new TrackWindow(this, (*song)[rmbMenuTrack],
                        Application::application()->history(song));
    }
    else if (rmbMenuPart)
    {
        new PartWindow(this, rmbMenuPart, song,
                       Application::application()->history(song));
    }
}


void SongWindow::slotRmbMenuOpened(size_t track, TSE3::Part *part)
{
    rmbMenuTrack     = track;
    rmbMenuPart      = part;
    rmbMenuFromTrack = false;
    propertiesAction->setEnabled(part != 0);
    propertiesAction->setText("Part &properties");
}


void SongWindow::slotRmbMenuOpened(size_t track)
{
    rmbMenuTrack     = track;
    rmbMenuPart      = 0;
    rmbMenuFromTrack = true;
    propertiesAction->setEnabled(true);
    propertiesAction->setText("Track &properties");
}


void SongWindow::slotSelectAllInTrack()
{
    if (rmbMenuTrack < song->size())
    {
        partSelection->clear();
        partSelection->selectAll((*song)[rmbMenuTrack]);
    }
}


void SongWindow::slotSelectAllInSelectedTracks()
{
    std::vector<TSE3::Track *> tracks;
    std::copy(trackSelection->begin(), trackSelection->end(),
              std::back_inserter(tracks));

    std::vector<TSE3::Track *>::iterator i = tracks.begin();
    while (i != tracks.end())
    {
        partSelection->selectAll(*i);
        ++i;
    }
}


void SongWindow::slotToolbarToggled(bool toggle)
{
    KToolBar *bar = toolBar(sender()->name());
    if (bar)
    {
      if (toggle)
      {
          bar->show();
      }
      else
      {
          bar->hide();
      }
    }
}


void SongWindow::slotSortTracks()
{
    TrackSortWindow *sort = new TrackSortWindow(this, song, trackSelection);
    sort->show();
}


void SongWindow::slotSetClock(TSE3::Clock clock, Qt::ButtonState button)
{
    if ((button == LeftButton && _viewOptions.lmbSetsClock)
        || (button == RightButton && _viewOptions.rmbSetsClock))
    {
        Application::application()->scheduler()->moveTo(snap(clock));
    }
}


void SongWindow::slotFlagComboActicated(int index)
{
    if ((size_t)index < song->flagTrack()->size())
    {
        TSE3::Clock nextClock = (*song->flagTrack())[index].time;
        Application::application()->scheduler()->moveTo(nextClock);
    }
}


/******************************************************************************
 * Other methods
 *****************************************************************************/

bool SongWindow::queryClose()
{
    return messageBoxSave();
}


bool SongWindow::queryExit()
{
    return true;
}


void SongWindow::saveProperties(KConfig *config)
{
    if (!filename && !modified.modified())
    {
        return;
    }
    config->writeEntry("filename",       filename);
    config->writeEntry("midiFileFormat", midiFileFormat);
    config->writeEntry("modified",       modified.modified());
    if (modified.modified())
    {
        filename.append(SAVE_APPEND);
        QString tmplocation = kapp->tempSaveName(filename);
        doSave(tmplocation, KURL(tmplocation));
    }
}


void SongWindow::readProperties(KConfig *config)
{
    filename       = config->readEntry("filename",          "");
    midiFileFormat = config->readNumEntry("midiFileFormat", 1);
    int mod        = config->readNumEntry("modified",       0);
    if (!filename.isEmpty() && mod)
    {
        QString tmpfilename = filename;
        bool    ok;
        tmpfilename.append(SAVE_APPEND);
        QString fn = kapp->checkRecoverFile(tmpfilename, ok);
        if (ok)
        {
            // Yes, there's a tempoary file and it's 'fn'
            Application::application()->load(filename, fileurl, this);
            modified.setModified();
        }
    }
    else
    {
        if (!filename.isEmpty())
        {
            // No temp file, so we just open up the previously opened file
            Application::application()->load(filename, fileurl, this);
        }
    }
}


void SongWindow::doSave(const QString &f, const KURL &u, int format)
{
    KURL url = (u == KURL()) ? KURL(f) : u;

    class BadFormatError {};

    try
    {
        if (f.find(QRegExp(".mid$")) != -1 || f.find(QRegExp(".midi$")) != -1)
        {
            // type = .midi or .mid

            // ask type
            if (format == -1)
            {
                if (KMessageBox::questionYesNo(this, "You can save MIDI files in two formats.\nPlease choose one:\n\tFormat 0: All data in a single 'MTrk'.\n\tFormat 1: Each track in a separate 'MTrk'.\n\nIf you're not sure, choose format 1.", "Choose MIDI file format", KGuiItem("Format 1"), KGuiItem("Format 0")) == KMessageBox::Yes)
                {
                    format = 1;
                }
                else
                {
                    format = 0;
                }
            }
            TSE3ProgressDialog progress("Exporting song", QString::null);
            TSE3::MidiFileExport mfe(format, true);
            mfe.save(f.latin1(), song, &progress);
        }
        else if (f.find(QRegExp(".tse2$")) != -1) // type = .tse2
        {
            //TSE3::TSE2MDL tse2mdl(Application::application()->name());
            //tse2mdl.save(f.latin1(), song);
            KMessageBox::sorry(this, "The TSE3 library does not support saving in TSE2 file format.");
        }
        else // type = .tse3
        {
            TSE3::TSE3MDL tse3mdl(Application::application()->name());
            tse3mdl.save(f.latin1(), song);
        }
        if (!KIO::NetAccess::upload(f, u))
        {
            KNotifyClient::event("cannotopenfile");
            return;
        }
        filename       = f;
        fileurl        = u;
        midiFileFormat = format;
        modified.setModified(false);
        doSetCaption();
        //XXX Application::application()->aRecentFilesAction()->addURL(url);
        recentFilesAction->addURL(url);
    }
    catch (const BadFormatError &e)
    {
        KMessageBox::sorry(this, "Anthem does not support this file type");
    }
    catch (...)
    {
        KMessageBox::error(this, "The file save operation was not sucessful.");
    }
}


bool SongWindow::messageBoxSave()
{
    if (modified.modified())
    {
        switch (KMessageBox::warningYesNoCancel(this,
                "The song has been modified.\n"
                "If you close this window your changes will be lost.",
                "",
                KGuiItem("Save"), KGuiItem("Don't save")/*, KGuiItem("Don't close")*/)) // XXX fix this
        {
            case KMessageBox::Yes:
            {
                if (!doSaveAs())
                {
                    return false;
                }
                break;
            }
            case KMessageBox::No:
            {
                return true;
                break;
            }
            case KMessageBox::Cancel:
            {
                return false;
            }
        }
    }
    return true;
}


void SongWindow::updateCutCopy()
{
    if (trackSelection->size())
    {
        selectionStatus = TrackSelected;
    }
    else if (partSelection->size())
    {
        selectionStatus = PartSelected;
    }
    else
    {
        selectionStatus = NothingSelected;
    }
    cutAction->setEnabled(selectionStatus != NothingSelected);
    copyAction->setEnabled(selectionStatus != NothingSelected);
    bothMarkersAction->setEnabled(selectionStatus == PartSelected);
    fromMarkerStartAction->setEnabled(selectionStatus == PartSelected);
    fromMarkerEndAction->setEnabled(selectionStatus == PartSelected);
    toMarkerStartAction->setEnabled(selectionStatus == PartSelected);
    toMarkerEndAction->setEnabled(selectionStatus == PartSelected);
}


/******************************************************************************
 * TSE3 listener interfaces
 *****************************************************************************/

void SongWindow::Transport_Status(TSE3::Transport *src, int newStatus)
{
    stopAction->setChecked(newStatus == TSE3::Transport::Resting);
    playAction->setChecked
        (newStatus == TSE3::Transport::Playing
         || newStatus == TSE3::Transport::SynchroPlaying);
    recordAction->setChecked
        (newStatus == TSE3::Transport::Recording
         || newStatus == TSE3::Transport::SynchroRecording);
}


void SongWindow::CommandHistory_Undo(TSE3::Cmd::CommandHistory *src)
{
    undoAction->setEnabled(history->undos());
    clearHistoryAction->setEnabled(history->undos() || history->redos());
}


void SongWindow::CommandHistory_Redo(TSE3::Cmd::CommandHistory *src)
{
    redoAction->setEnabled(history->redos());
    clearHistoryAction->setEnabled(history->undos() || history->redos());
}


void SongWindow::Modified_Changed(TSE3::App::Modified *)
{
    doSetCaption();
}


void SongWindow::TrackSelection_Selected(TSE3::App::TrackSelection *,
                                         TSE3::Track *, bool selected)
{
    updateCutCopy();
}


void SongWindow::PartSelection_Selected(TSE3::App::PartSelection *,
                                        TSE3::Part *, bool selected)
{
    updateCutCopy();
}


void SongWindow::Song_FromAltered(TSE3::Song *, TSE3::Clock c)
{
    fromDisplay->setValue(c);
}


void SongWindow::Song_ToAltered(TSE3::Song *, TSE3::Clock c)
{
    toDisplay->setValue(c);
}

