/*
 * @(#)SongWindow_TimeLine.cpp 1.00 22 May 2000
 *
 * Copyright (c) Pete Goodliffe 2000 (pete@cthree.org)
 *
 * This file is part of anthem - the TSE3 sequencer.
 *
 * This program is modifiable/redistributable under the terms of the GNU
 * General Public License.
 *
 * You should have recieved a copy of the GNU General Public License along
 * with this program; see the file COPYING. If not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 0211-1307, USA.
 */

#include "songwindow/TimeLine.h"

#include "Application.h"

#include <kglobal.h>
#include <qlayout.h>
#include <qtimer.h>
#include <qpainter.h>

#include "tse3/Song.h"
#include "tse3/Track.h"
#include "tse3/Part.h"
#include "tse3/TimeSigTrack.h"
#include "tse3/MidiScheduler.h"
#include "tse3/Transport.h"
#include "tse3/FlagTrack.h"

namespace
{
    const int DEFAULT_HZOOM = 10000;
    const int MIN_HZOOM = 1000;
    const int MAX_HZOOM = 40000;
    const int CURSOR_HEIGHT = 5;
    const int CURSOR_WIDTH  = 10;
    const int MSECS         = 50;
}

namespace
{
    /**
     * The TimeLine is ever so slightly bigger than the PartView: this
     * is because the PartView has a vertical scroll bar that consumes some
     * screen real estate. Therefore this fiddle factor ensures that there is
     * still a little give in the TimeLine (it pretends to be smaller than
     * it is).
     */
    const int scrollBarOffset = 200;

    /**
     * The space (in pixels) to the right of the @ref Song (room to
     * grow into!).
     */
    const TSE3::Clock marginRight = TSE3::Clock::PPQN*32*4;
}

TimeLine::TimeLine(TSE3::Song *song, QWidget *parent,
                   TSE3::Util::Snap &snap)
: QScrollView(parent, "TimeLine", WStyle_NoBorder),
  snap(snap), song(song), scale(DEFAULT_HZOOM), cursor(0), _followCursor(true)
{
    viewport()->setBackgroundColor(QColor(255, 255, 255));
    setHScrollBarMode(AlwaysOff);
    setVScrollBarMode(AlwaysOff);
    recalculateWidth();

    timer = new QTimer(this);
    timer->start(MSECS, false);
    connect(timer, SIGNAL(timeout()), SLOT(slotTimeout()));

    if (0)
    {
        // XXX -- REMOVE ME!!! -- XXX --- test data -- XXX
        // Test harness
        using namespace TSE3;
        song->timeSigTrack()->insert(Event<TimeSig>(TimeSig(3, 4),
                                     Clock::PPQN*16));
        song->timeSigTrack()->insert(Event<TimeSig>(TimeSig(4, 4),
                                     Clock::PPQN*28));
    }

    TSE3::Listener<TSE3::SongListener>::attachTo(song);
    TSE3::Listener<TSE3::FlagTrackListener>::attachTo(song->flagTrack());
    for (size_t n = 0; n < song->size(); n++)
    {
        TSE3::Listener<TSE3::TrackListener>::attachTo((*song)[n]);
    }
}


void TimeLine::changeSong(TSE3::Song *s)
{
    // 1. Out with the old
    TSE3::Listener<TSE3::SongListener>::detachFrom(song);
    TSE3::Listener<TSE3::FlagTrackListener>::detachFrom(song->flagTrack());
    for (size_t n = 0; n < song->size(); n++)
    {
        TSE3::Listener<TSE3::TrackListener>::detachFrom((*song)[n]);
    }

    // 2. In with the new
    song = s;
    recalculateWidth();
    TSE3::Listener<TSE3::SongListener>::attachTo(song);
    TSE3::Listener<TSE3::FlagTrackListener>::attachTo(song->flagTrack());
    for (size_t n = 0; n < song->size(); n++)
    {
        TSE3::Listener<TSE3::TrackListener>::attachTo((*song)[n]);
    }
    viewport()->repaint();
}


/******************************************************************************
 * Calculating mark positions
 *****************************************************************************/

void TimeLine::snapRange(TSE3::Clock &start, TSE3::Clock &end)
{
    // Start
    /*
    start /= TSE3::Clock::PPQN*4*4;
    start *= TSE3::Clock::PPQN*4*4;
    if (start < 0) start = 0;
    */
    size_t                         tspos   = song->timeSigTrack()
                                                 ->index(start, false);
    TSE3::TimeSigTrack::event_type timesig = (*song->timeSigTrack())[tspos];
    const TSE3::Clock              barlen  = timesig.data.top
                                           * TSE3::Clock::PPQN
                                           * 4
                                           / timesig.data.bottom;
    TSE3::Clock                    diff    = start - timesig.time;

    diff /= barlen;
    diff *= barlen;

    start = timesig.time + diff;
    if (start < 0) start = 0;

    // End
    end /= TSE3::Clock::PPQN*4;
    end += 1;
    end *= TSE3::Clock::PPQN*4;
    if (end < 0) end = 0;
}


TSE3::Clock TimeLine::next(TSE3::Clock last)
{
    size_t                         tspos   = song->timeSigTrack()
                                                 ->index(last, false);
    TSE3::TimeSigTrack::event_type timesig = (*song->timeSigTrack())[tspos];
    const TSE3::Clock              barlen  = timesig.data.top
                                           * TSE3::Clock::PPQN
                                           * 4
                                           / timesig.data.bottom;

    if (tspos+1 < song->timeSigTrack()->size())
    {
        TSE3::Clock nexttspos = (*song->timeSigTrack())[tspos+1].time;
        //int top = (*song->timeSigTrack())[tspos+1].data.top;
        //int bottom = (*song->timeSigTrack())[tspos+1].data.bottom;
        if (last + barlen > nexttspos)
        {
            return nexttspos;
        }
    }

    return last + barlen;
}


bool TimeLine::isBig(TSE3::Clock time)
{
    int bar = 0, beat = 0, pulse = 0;
    song->timeSigTrack()->barBeatPulse(time, bar, beat, pulse);
    return !(bar % 4);
}


bool TimeLine::isBigBar(int barno)
{
    return !(barno % 4);
}


/******************************************************************************
 * Mouse events
 *****************************************************************************/

void TimeLine::contentsMousePressEvent(QMouseEvent *e)
{
    TSE3::Clock pos = snap(pixelToClock(e->x()));
    if (pos < 0) pos = 0;

    if (e->button() == LeftButton && e->state() & ControlButton)
    {
        song->setFrom(pos);
    }
    else if (e->button() == RightButton && e->state() & ControlButton)
    {
        song->setTo(pos);
    }
    else
    {
        emit setClock(pos, e->button());
    }
}


/******************************************************************************
 * Window redraw
 *****************************************************************************/

void TimeLine::drawContents(QPainter *p,
                            int clipx, int clipy, int clipw, int cliph)
{

    /*
    for (int n = (clipx/40)*40; n < clipx+clipw; n += 40)
    {
        p->drawLine(n, 0, n+40, height());
    }
    */
    using namespace TSE3;

    Clock minClock = pixelToClock(clipx);
    Clock maxClock = pixelToClock(clipx+clipw);

    //cout << "Redraw timeline, pixels(" << clipx << ", " << clipx+clipw
    //     << ") clocks (" << minClock << ", " << maxClock << ")\n";

    snapRange(minClock, maxClock);

    // Calculate min bar no
    int bar = 0, tempbeat = 0, temppulse = 0;
    song->timeSigTrack()->barBeatPulse(minClock, bar, tempbeat, temppulse);

    for (Clock clock = minClock; clock <= maxClock; clock = next(clock))
    {
        if (isBigBar(bar))
        {
            int x = clockToPixel(clock);
            p->drawLine(x, 0, x, height());
            QString string;
            string.sprintf("%i", bar+1);
            p->drawText(x+4, height()-10, string);
        }
        else
        {
            int x = clockToPixel(clock);
            p->drawLine(x, 0, x, height()/2);
        }
        bar++;
    }

    // Draw repeat markers
    {
        int x1 = clockToPixel(song->from());
        int x2 = clockToPixel(song->to());
        p->setPen(Qt::darkYellow);
        p->drawLine(x1, viewport()->height()-CURSOR_HEIGHT+1,
                    x2, viewport()->height()-CURSOR_HEIGHT+1);
        p->drawLine(x1, viewport()->height()-CURSOR_HEIGHT-1,
                    x2, viewport()->height()-CURSOR_HEIGHT-1);
        p->setBrush(Qt::yellow);
        QPointArray tri(3);
        tri.setPoint(0, x1,               viewport()->height());
        tri.setPoint(1, x1,               viewport()->height()-CURSOR_HEIGHT*2);
        tri.setPoint(2, x1+CURSOR_WIDTH/2,viewport()->height()-CURSOR_HEIGHT);
        p->drawPolygon(tri);
        tri.setPoint(0, x2,               viewport()->height());
        tri.setPoint(1, x2,               viewport()->height()-CURSOR_HEIGHT*2);
        tri.setPoint(2, x2-CURSOR_WIDTH/2,viewport()->height()-CURSOR_HEIGHT);
        p->drawPolygon(tri);
        p->setPen(Qt::yellow);
        p->drawLine(x1+2, viewport()->height()-CURSOR_HEIGHT,
                    x2-2, viewport()->height()-CURSOR_HEIGHT);
    }

    // Draw flags
    {
        TSE3::FlagTrack *flagTrack = song->flagTrack();
        p->setPen(QPen(Qt::darkGreen, 2));
        p->setBrush(Qt::green);
        size_t maxFlag = flagTrack->index(maxClock);
        for (size_t flag = flagTrack->index(minClock, false);
             flag < maxFlag;
             ++flag)
        {
            int x = clockToPixel((*flagTrack)[flag].time);
            QPointArray tri(3);
            tri.setPoint(0, x, 0);
            tri.setPoint(1, x, CURSOR_WIDTH);
            tri.setPoint(2, x+CURSOR_WIDTH, 0);
            p->drawPolygon(tri);
        }
        p->setPen(QPen(Qt::darkGreen, 2));
    }

    // Draw cursor
    QColor col = Qt::red;
    TSE3::Transport *transport = Application::application()->transport();
    if (transport->status() != TSE3::Transport::Resting
        && transport->playable() != song)
    {
        col = (Application::application()->scheduler()->msecs() % 500 < 250)
            ? QColor(0x50, 0x50, 0x50)
            : Qt::red;
    }

    int x = clockToPixel(cursor);
    QPointArray pa(3);
    pa.setPoint(0, x,                viewport()->height());
    pa.setPoint(1, x-CURSOR_WIDTH/2, viewport()->height()-CURSOR_HEIGHT);
    pa.setPoint(2, x+CURSOR_WIDTH/2, viewport()->height()-CURSOR_HEIGHT);
    p->setPen(col);
    QBrush brush(col);
    p->setBrush(brush);
    p->drawPolygon(pa);
}


/******************************************************************************
 * Size management
 *****************************************************************************/

void TimeLine::move(int x, int y)
{
    bool b = this->signalsBlocked();
    blockSignals(true);
    setContentsPos(x, 0);
    blockSignals(b);
}


void TimeLine::zoomIn()
{
    TSE3::Clock offset = pixelToClock(contentsX());
    scale -= 1000;
    if (scale < MIN_HZOOM) scale = MIN_HZOOM;
    internalRecalculateWidth(0, false, true);
    setContentsPos(clockToPixel(offset), contentsY());
}


void TimeLine::zoomOut()
{
    TSE3::Clock offset = pixelToClock(contentsX());
    scale += 1000;
    if (scale > MAX_HZOOM) scale = MAX_HZOOM;
    internalRecalculateWidth(0, false, true);
    setContentsPos(clockToPixel(offset), contentsY());
}


int TimeLine::zoom()
{
    return (scale - MIN_HZOOM) * 100 / (MAX_HZOOM-MIN_HZOOM);
}


void TimeLine::setZoom(int pc)
{
    TSE3::Clock offset = pixelToClock(contentsX());
    scale = MIN_HZOOM + (MAX_HZOOM-MIN_HZOOM)*pc/100;
    internalRecalculateWidth(0, false, true);
    setContentsPos(clockToPixel(offset), contentsY());
}


void TimeLine::internalRecalculateWidth(TSE3::Track *track,
                                        bool doEmit, bool needsRedraw)
{
    if (track)
    {
        TSE3::Clock ltc = lastTrackClock(track);
        int pixels = clockToPixel(ltc+marginRight) + scrollBarOffset;
        if (pixels > contentsWidth())
        {
            // XXX A copy of below, perhaps this needs rearrangement?
            resizeContents(pixels, contentsHeight());
            if (needsRedraw)
            {
                viewport()->repaint();
            }
            emit widthChanged(pixels-scrollBarOffset, needsRedraw);
            return;
        }
        else if (pixels == contentsWidth())
        {
            return;
        }
        // Else need to check every other Track to shrink the width...
    }

    TSE3::Clock lastClock = 0;
    for (size_t n = 0; n < song->size(); n++)
    {
        TSE3::Clock ltc = lastTrackClock((*song)[n]);
        if (ltc > lastClock) lastClock = ltc;
    }
    int pixels = clockToPixel(lastClock+marginRight) + scrollBarOffset;
    if (pixels != contentsWidth() || doEmit)
    {
        resizeContents(pixels, contentsHeight());
        if (needsRedraw)
        {
            viewport()->repaint();
        }
        emit widthChanged(pixels-scrollBarOffset, needsRedraw);
    }
}


TSE3::Clock TimeLine::lastTrackClock(TSE3::Track *track)
{
    return (track->size()) ? (*track)[track->size()-1]->end()
                           : TSE3::Clock(0);
}


void TimeLine::Track_PartInserted(TSE3::Track *track, TSE3::Part *part)
{
    internalRecalculateWidth(track);
}


void TimeLine::Track_PartRemoved(TSE3::Track *track, TSE3::Part *)
{
    internalRecalculateWidth(track);
}


void TimeLine::Song_FromAltered(TSE3::Song *, TSE3::Clock from)
{
    viewport()->repaint();
}


void TimeLine::Song_ToAltered(TSE3::Song *, TSE3::Clock to)
{
    viewport()->repaint();
}


void TimeLine::Song_TrackInserted(TSE3::Song *, TSE3::Track *track)
{
    internalRecalculateWidth(track);
}


void TimeLine::Song_TrackRemoved(TSE3::Song *src, TSE3::Track *, size_t)
{
    internalRecalculateWidth();
}


void TimeLine::EventTrack_EventAltered(TSE3::EventTrack<TSE3::Flag> *,
                                       size_t)
{
    viewport()->repaint();
}

void TimeLine::EventTrack_EventInserted(TSE3::EventTrack<TSE3::Flag> *,
                                        size_t)
{
    viewport()->repaint();
}

void TimeLine::EventTrack_EventErased(TSE3::EventTrack<TSE3::Flag> *,
                                      size_t)
{
    viewport()->repaint();
}


/******************************************************************************
 * Cursor handling
 *****************************************************************************/

void TimeLine::slotTimeout()
{
    TSE3::Clock newTime = Application::application()->scheduler()->clock();
    if (newTime != cursor)
    {
        // these must be repaints, not updates, else the scroll could make
        // the coordinates wrong - Qt is being sucky here

        TSE3::Clock oldTime = cursor;
        cursor = newTime;
        int oldx = clockToPixel(oldTime);
        repaintContents(oldx-CURSOR_WIDTH/2 - 2, 0, CURSOR_WIDTH + 4, height());
        int newx = clockToPixel(cursor);
        repaintContents(newx-CURSOR_WIDTH/2 - 2, 0, CURSOR_WIDTH + 4, height());
        int newxminusmargin = clockToPixel(newTime-offset);
        if (_followCursor)
        {
            if (contentsX() + visibleWidth() < newx)
            {
                setContentsPos(newx, 0);
            }
            else if (contentsX() > newxminusmargin)
            {
                setContentsPos(newxminusmargin, 0);
            }
        }
    }
}
