/*
 * @(#)SongWindow_TimeLine.h 1.00 22 May 2000
 *
 * Copyright (c) Pete Goodliffe 2000 (pete@cthree.org)
 *
 * This file is part of anthem - the TSE3 sequencer.
 *
 * This program is modifiable/redistributable under the terms of the GNU
 * General Public License.
 *
 * You should have recieved a copy of the GNU General Public License along
 * with this program; see the file COPYING. If not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 0211-1307, USA.
 */

#ifndef ANTHEM_SONGWINDOW_TIMELINE_H
#define ANTHEM_SONGWINDOW_TIMELINE_H

#include <qscrollview.h>

#include "tse3/util/Snap.h"
#include "tse3/Notifier.h"
#include "tse3/listen/Song.h"
#include "tse3/listen/Track.h"
#include "tse3/listen/TimeSigTrack.h"
#include "tse3/listen/FlagTrack.h"

namespace TSE3
{
    class Part;
}

class QTimer;

/**
 * This widget provides a timeline. It is scrolled along with the @ref PartView.
 * It handles the horizontal sizing of itself (and the @ref PartView) by
 * listening to the @ref Song and working out whether the size should change.
 *
 * It looks at the @ref TSE3::TimeSigTrack to ensure bar sizing is correct.
 * It can also display flag marks by looking at the @ref TSE3::FlagTrack.
 *
 * @short   Song Window timeline
 * @author  Pete Goodliffe
 * @version 1.0
 */
class TimeLine : public QScrollView,
                 public TSE3::Listener<TSE3::SongListener>,
                 public TSE3::Listener<TSE3::TrackListener>,
                 public TSE3::Listener<TSE3::FlagTrackListener>
{
        Q_OBJECT;

    public:

        /**
         * Create a timeline for the given @ref TSE3::Song.
         */
        TimeLine(TSE3::Song *song, QWidget *parent,
                 TSE3::Util::Snap &snap);

        /**
         * Changes which @ref TSE3::Song this window views.
         *
         * This can only be called safely by the @ref SongWindow.
         */
        void changeSong(TSE3::Song *song);

        /**
         * Returns whether the window will scroll to track the current
         * timeline cursor position.
         *
         * @see setFollowCursor
         */
        bool followCursor() const { return _followCursor; }

        /**
         * Alters the follow cursor setting.
         *
         * @see followCursor
         */
        void setFollowCursor(bool f) { _followCursor = f; }

        /**
         * Converts a @ref Clock offset in the @ref Song into a pixel
         * offset into the TimeLine.
         */
        int clockToPixel(TSE3::Clock mc)
        {
            // muldiv?
            return (mc+offset)*1000/scale;
        }

        /**
         * Converts a pixel offset in the TimeLine into a @ref Clock
         * value in the @ref Song.
         */
        TSE3::Clock pixelToClock(int pixel)
        {
            // muldiv?
            return (pixel*scale/1000)-offset;
        }

        /**
         * Given the start/end time range for a redraw, this method
         * 'snaps' the values to the nearest interesting values (i.e.
         * values that have a tickmark or bar number). This will account
         * for intersting redraw problems, like the width of bar number
         * displays and the clipping at the beginning and end of the
         * window.
         *
         * You can now start iterating at the 'snapped' @p start time, and
         * call @ref next repeatedly until you get to @p end.
         *
         * @see next
         */
        void snapRange(TSE3::Clock &start, TSE3::Clock &end);

        /**
         * Returns the next time to have a tickmark or bar number mark after
         * this one.
         */
        TSE3::Clock next(TSE3::Clock last);

        /**
         * Returns whether the specified time has a 'big' bar number mark
         * or just a 'little' tickmark.
         *
         * If you already know the bar number, it is quicker to call
         * @ref isBigBar.
         */
        bool isBig(TSE3::Clock time);

        /**
         * Returns whether the specified bar number has a 'big' bar number mark
         * or just a 'little' tickmark.
         *
         * If you don't know the bar number, just the time you can use
         * @ref isBig.
         */
        bool isBigBar(int barno);

        /**
         * You can cause this to force a width recalculation. The only
         * really useful time you will want to do this is when you
         * have just created the TimeLine and the @ref PartList and you
         * want to trigger a signal which will cause the @ref PartList to
         * resize.
         */
        void recalculateWidth()
        {
            internalRecalculateWidth();
        }

        /**
         * Like recalulateWidth, but forces the object to emit
         * @ref widthChanged.
         */
        void recalculateWidthAndEmit()
        {
            internalRecalculateWidth(0, true);
        }

        /**
         * @reimplemented
         */
        virtual void Track_PartInserted(TSE3::Track *src, TSE3::Part *);

        /**
         * @reimplemented
         */
        virtual void Track_PartRemoved(TSE3::Track *src, TSE3::Part *);

        /**
         * @reimplemented
         */
        virtual void Song_FromAltered(TSE3::Song *src, TSE3::Clock from);

        /**
         * @reimplemented
         */
        virtual void Song_ToAltered(TSE3::Song *src, TSE3::Clock to);

        /**
         * @reimplemented
         */
        virtual void Song_TrackInserted(TSE3::Song *src, TSE3::Track *);

        /**
         * @reimplemented
         */
        virtual void Song_TrackRemoved(TSE3::Song *src, TSE3::Track *, size_t);

        /**
         * @reimplemented
         */
        virtual void EventTrack_EventAltered(TSE3::EventTrack<TSE3::Flag> *,
                                             size_t);

        /**
         * @reimplemented
         */
        virtual void EventTrack_EventInserted(TSE3::EventTrack<TSE3::Flag> *,
                                              size_t);

        /**
         * @reimplemented
         */
        virtual void EventTrack_EventErased(TSE3::EventTrack<TSE3::Flag> *,
                                            size_t);

    public slots:

        /**
         * A parent view has been scrolled, move this timeline with it.
         * The y value will be ignored.
         */
        void move(int x, int y);

        /**
         * This slot will cause the horizontal scale to 'zoom in'.
         * A @ref zoomChanged() signal will be emitted.
         */
        void zoomIn();

        /**
         * This slot will cause the horizontal scale to 'zoom out'.
         * A @ref zoomChanged() signal will be emitted.
         */
        void zoomOut();

        /**
         * Returns current zoom percentage.
         */
        int zoom();

        /**
         * Zoom to percentage @p pc.
         */
        void setZoom(int pc);

    protected slots:

        /**
         * This timeout slot redraws the timeline cursor if it has moved.
         */
        void slotTimeout();

    protected:

        virtual void drawContents(QPainter *p,
                                 int clipx, int clipy, int clipw, int cliph);
        virtual void contentsMousePressEvent(QMouseEvent *e);

    signals:

        /**
         * This is emitted when the horizontal zoom is changed.
         */
        void zoomChanged();

        /**
         * This is emitted when the TimeLine's width changes.
         *
         * @param newWidth    The new TimeLine width (in pixels)
         * @param needsRedraw True if the entire display needs to be
         *                    drawn (for example, if the scale changes)
         */
        void widthChanged(int newWidth, bool needsRedraw);

         /**
          * Emitted when a click occurs in the PartView.
          */
         void setClock(TSE3::Clock time, Qt::ButtonState button);

    private:

        /**
         * This offset represents the 'left margin' of the TimeLine where
         * there is no @ref Song data, but a leanin cursor may be shown.
         */
        static const int offset = TSE3::Clock::PPQN*4;

        /**
         * Causes a size recalculation to occur. If track is 0 then all
         * @ref Tracks in the @ref Song will be checked. If track is
         * specified then a change will only be produced based on this
         * @ref Track's length having changed.
         *
         * Will emit a @ref widthChanged signal if the width changes.
         *
         * @param track       Hint for what's changed
         * @param doEmit      if false emit only when changes, true always emit
         * @param needsRedraw if true sends any signal with the redraw flag set
         */
        void internalRecalculateWidth(TSE3::Track *track = 0,
                                      bool doEmit = false,
                                      bool needsRedraw = false);

        /**
         * Returns the end time of the last @ref Part in the @ref Track.
         */
        TSE3::Clock lastTrackClock(TSE3::Track *track);

        TSE3::Util::Snap &snap;

        TSE3::Song *song;
        int         scale; // unit are clocks per pixel * 1000

        TSE3::Clock cursor;
        bool        _followCursor;

        QTimer *timer;
};

#endif
