/*
 * @(#)ClockWidget.cpp 1.00 1 June 2000
 *
 * Copyright (c) Pete Goodliffe 2000 (pete@cthree.org)
 *
 * This file is part of anthem - the TSE3 sequencer.
 *
 * This program is modifiable/redistributable under the terms of the GNU
 * General Public License.
 *
 * You should have recieved a copy of the GNU General Public License along
 * with this program; see the file COPYING. If not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 0211-1307, USA.
 */

#include "tse3/kdeui/ClockWidget.h"

#include <qlabel.h>
#include <qlayout.h>
#include <qspinbox.h>

#include <kdialogbase.h>

#include "tse3/Midi.h"

namespace
{
    int MAXINT = 99999;
}

/******************************************************************************
 * ClockSpinBox widget class
 *****************************************************************************/

/**
 * This is a special kind of spin box that handles values in the
 * ClockWidget class.
 *
 * @short   Clock spin box
 * @author  Pete Goodliffe
 * @version 1.0
 */
class ClockSpinBox : public QSpinBox
{
    public:
        void setSpecialValueTextReplace(bool r)
        {
            replace = r;
            if (value() == 0) updateDisplay();
        }
        void setSpecialValueText(QString s) { string  = s; }
    public:
        ClockSpinBox(int min, int max, int step = 1,
                     QWidget *parent = 0, const char *name = 0);
    protected:
        virtual QString mapValueToText(int v);
        virtual int mapTextToValue(bool *ok);
    private:
        bool    replace; // whether to replace or not
        QString string;
};


ClockSpinBox::ClockSpinBox(int min, int max, int step = 1,
                           QWidget *parent = 0, const char *name = 0)
: QSpinBox(min, max, step, parent, name), replace(false)
{
}


QString ClockSpinBox::mapValueToText(int v)
{
    if (replace && v == 0)
    {
        return string;
    }
    else
    {
        return QSpinBox::mapValueToText(v);
    }
}


int ClockSpinBox::mapTextToValue(bool *ok)
{
    if (replace && text() == string)
    {
        return 0;
    }
    return QSpinBox::mapTextToValue(ok);
}


/******************************************************************************
 * ClockWidget class
 *****************************************************************************/

ClockWidget::ClockWidget(QWidget *parent, bool negative,
                         TSE3::Clock mc, const char *name)
: QWidget(parent, name), negative(negative)
{
    QHBoxLayout *layout = new QHBoxLayout(this, 0, KDialogBase::spacingHint());

    beats = new ClockSpinBox(negative ? -MAXINT : 0, MAXINT, 1, this);
    layout->addWidget(beats);

    QLabel *label = new QLabel(":", this);
    layout->addWidget(label);

    // The pulses spin box is deliberately created so its values can
    // be 'one past' a valid value in each direction. This allows the slots
    // to make the pulses spin box update the beats one. Cunning.

    pulses = new ClockSpinBox(-1, TSE3::Clock::PPQN, 1, this);
    pulses->setMaximumSize(beats->sizeHint());
    layout->addWidget(pulses);

    setValue(mc);

    connect(beats,  SIGNAL(valueChanged(int)), SLOT(beatsChanged(int)));
    connect(pulses, SIGNAL(valueChanged(int)), SLOT(pulsesChanged(int)));
}


ClockWidget::~ClockWidget()
{
}


TSE3::Clock ClockWidget::value()
{
    return (beats->value()*TSE3::Clock::PPQN) + pulses->value();
}


void ClockWidget::setValue(TSE3::Clock mc)
{
    beats->setValue(mc / TSE3::Clock::PPQN);
    pulses->setValue(mc % TSE3::Clock::PPQN);
    if (!_specialValueText.isEmpty())
    {
        pulses->setSpecialValueTextReplace(beats->value() == 0);
        beats->setSpecialValueTextReplace(pulses->value() == 0);
    }
}


QString ClockWidget::specialValueText()
{
    return _specialValueText;
}


void ClockWidget::setSpecialValueText(QString s)
{
    _specialValueText = s;
    beats->setSpecialValueText(s);
    pulses->setSpecialValueText("");
    setValue(value());
}


void ClockWidget::beatsChanged(int b)
{
    if (!_specialValueText.isEmpty())
    {
        pulses->setSpecialValueTextReplace(beats->value() == 0);
    }
    emit valueChanged(value());
}


void ClockWidget::pulsesChanged(int b)
{
    switch (b)
    {
        case TSE3::Clock::PPQN:
        {
            beats->setValue(beats->value()+1);
            pulses->setValue(0);
            break;
        }
        case -1:
        {
            int beatval = beats->value();
            if (negative || beatval > 0)
            {
                beats->setValue(beatval-1);
                pulses->setValue(TSE3::Clock::PPQN-1);
            }
            else
            {
                pulses->setValue(0);
            }
            break;
        }
    }
    if (!_specialValueText.isEmpty())
    {
        beats->setSpecialValueTextReplace(pulses->value() == 0);
    }
    emit valueChanged(value());
}


QString ClockWidget::toText(TSE3::Clock c)
{
    QString ret;
    ret.sprintf("%d : %d", int(c/TSE3::Clock::PPQN), int(c%TSE3::Clock::PPQN));
    return ret;
}

