/*
 * @(#)FlagComboBox.cpp 1.00 9 May 2002
 *
 * Copyright (c) Pete Goodliffe 2002 (pete@cthree.org)
 *
 * This file is part of anthem - the TSE3 sequencer.
 *
 * This program is modifiable/redistributable under the terms of the GNU
 * General Public License.
 *
 * You should have recieved a copy of the GNU General Public License along
 * with this program; see the file COPYING. If not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 0211-1307, USA.
 */

#include "tse3/kdeui/FlagComboBox.h"

#include <qtimer.h>

#include "tse3/FlagTrack.h"
#include "tse3/MidiScheduler.h"

FlagComboBox::FlagComboBox(TSE3::FlagTrack *f,
                           QWidget *parent, const char *name)
: KComboBox(0, parent, name), timer(0), scheduler(0), flagTrack(f)
{
    if (flagTrack)
    {
        attachTo(flagTrack);
    }

    update();
}


void FlagComboBox::setFlagTrack(TSE3::FlagTrack *newFlagTrack)
{
    if (flagTrack)
    {
        detachFrom(flagTrack);
    }

    flagTrack = newFlagTrack;
    attachTo(flagTrack);
    update();
}


void FlagComboBox::setMidiScheduler(TSE3::MidiScheduler *s)
{
    if (s)
    {
        if (!timer)
        {
            scheduler = s;
            timer = new QTimer(this);
            timer->start(500, false);
            connect(timer, SIGNAL(timeout()), SLOT(slotTimer()));
        }
    }
    else
    {
        delete timer;
        timer = 0;
        s     = 0;
    }
}


void FlagComboBox::update()
{
    clear();

    if (flagTrack)
    {
        for (size_t n = 0; n < flagTrack->size(); ++n)
        {
            insertItem((*flagTrack)[n].data.title().c_str());
        }
    }
}


void FlagComboBox::setTime(TSE3::Clock time)
{
    if (flagTrack)
    {
        size_t index      = flagTrack->index(time, false);
        int    comboIndex = (index>=flagTrack->size()) ? -1 : (int)index;
        setCurrentItem(comboIndex);
    }
}


void FlagComboBox::slotTimer()
{
    if (scheduler)
    {
        setTime(scheduler->clock());
    }
}


void FlagComboBox::EventTrack_EventAltered(TSE3::EventTrack<TSE3::Flag> *,
                                     size_t)
{
    update();
}


void FlagComboBox::EventTrack_EventInserted(TSE3::EventTrack<TSE3::Flag> *,
                                     size_t)
{
    update();
}


void FlagComboBox::EventTrack_EventErased(TSE3::EventTrack<TSE3::Flag> *,
                                    size_t)
{
    update();
}


void FlagComboBox::Notifier_Deleted(TSE3::EventTrack<TSE3::Flag> *)
{
    flagTrack = 0;
    update();
}


void FlagComboBox::Notifier_Deleted(TSE3::MidiScheduler *)
{
    delete timer;
    scheduler = 0;
    timer     = 0;
}

