/*
 * @(#)PortChannelWidgets.cpp 1.00 29 April 2002
 *
 * Copyright (c) Pete Goodliffe 2002 (pete@cthree.org)
 *
 * This file is part of anthem - the TSE3 sequencer.
 *
 * This program is modifiable/redistributable under the terms of the GNU
 * General Public License.
 *
 * You should have recieved a copy of the GNU General Public License along
 * with this program; see the file COPYING. If not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 0211-1307, USA.
 */


#include "tse3/kdeui/PortWidget.h"
#include "tse3/kdeui/ChannelWidget.h"

#include "Application.h"

#include <qlabel.h>
#include <qlayout.h>
#include <qspinbox.h>
#include <kcombobox.h>
#include <kdialogbase.h>

#include "tse3/MidiScheduler.h"

/******************************************************************************
 * PortSpinBox internal helper class
 *****************************************************************************/

namespace
{
    const char *NONE_STR = "None";
    const char *ALL_STR  = "All";
    const char *SAME_STR = "Same";
}

/**
 * This is a special kind of spin box that handles values in the
 * PortWidget and ChannelWidget class.
 *
 * This is realy a convenience base class that can intelligently handle
 * the "magic" values.
 *
 * @short   Port value spin box
 * @author  Pete Goodliffe
 * @version 1.0
 */
class DestSpinBox : public QSpinBox
{
    public:
        DestSpinBox(bool allow_bad,
                    bool allow_no, bool allow_all, bool allow_same,
                    int value,
                    QWidget *parent = 0, const char *name = 0)
            : QSpinBox(-10, 9999 /*XXX*/, 1, parent, name),
              allow_bad(allow_bad)
        {
            bool allow[3] = { allow_no, allow_all, allow_same };
            minus_values[0] = 0; minus_values[1] = 0; minus_values[2] = 0;
            for (int n = 0, m = 0; n < 3; ++n)
            {
                if (allow[n])
                {
                    minus_values[m] = -1-n;
                    ++m;
                }
            }

            int minNumber = 0 - allow_no - allow_all - allow_same;
            setMinValue(minNumber);
        }
    protected:
        virtual QString mapValueToText(int v)
        {
            // Works equally well for "magic" channel numbers
            int portNumber = (v < 0)
                           ? minus_values[-1-v]
                           : internalMapValueToDisplayValue(v);
            switch (portNumber)
            {
                case TSE3::MidiCommand::NoPort:   return NONE_STR;
                case TSE3::MidiCommand::AllPorts: return ALL_STR;
                case TSE3::MidiCommand::SamePort: return SAME_STR;
            }
            QString ret;
            ret.setNum(portNumber);
            return ret;
        }
        virtual int mapTextToValue(bool *ok)
        {
            QString str = cleanText().lower();
            if (str == QString(NONE_STR).lower())
                return TSE3::MidiCommand::NoPort;
            if (str == QString(ALL_STR).lower())
                return TSE3::MidiCommand::AllPorts;
            if (str == QString(SAME_STR).lower())
                return TSE3::MidiCommand::SamePort;
            return str.toInt();
        }
        virtual int internalMapValueToDisplayValue(int v)
        {
            return v;
        }

        bool allow_bad;

        int minus_values[3];
};


class PortSpinBox : public DestSpinBox,
                    public TSE3::Listener<TSE3::MidiSchedulerListener>
{
    public:
        PortSpinBox(bool allow_bad,
                    bool allow_no, bool allow_all, bool allow_same,
                    int value,
                    QWidget *parent = 0, const char *name = 0)
            : DestSpinBox(allow_bad, allow_no, allow_all, allow_same,
                          value, parent, name)
        {
            attachTo(Application::application()->scheduler());
            calculateMaxValue();
        }
        virtual void calculateMaxValue()
        {
            if (!allow_bad)
            {
                TSE3::MidiScheduler *scheduler
                    = Application::application()->scheduler();
                setMaxValue(scheduler->numPorts()-1);
            }
            else
            {
                setMaxValue(9999);// XXX
            }
        }
        void setPortNumber(int num)
        {
            if (num < 0)
            {
                num = minusValueFromPortNumber(num);
            }
            setValue(num);
        }
        int minusValueFromPortNumber(int port)
        {
            for (int n = 0; n < 3; ++n)
            {
                if (minus_values[n] == port) return -1-n;
            }
            return -1;
        }
        int midiPortNumber()
        {
            return midiPortNumber(value());
        }
        int midiPortNumber(int val)
        {
            if (val < 0)
            {
                val = minus_values[-1-val];
            }
            else if (!allow_bad)
            {
                TSE3::MidiScheduler *scheduler
                    = Application::application()->scheduler();
                val = scheduler->portNumber(val);
            }
            return val;
        }
        virtual void MidiScheduler_PortAdded(TSE3::MidiScheduler *, size_t)
        {
            calculateMaxValue();
        }
        virtual void MidiScheduler_PortRemoved(TSE3::MidiScheduler *, size_t)
        {
            calculateMaxValue();
        }
    protected:
        virtual void valueChange()
        {
            if (allow_bad)
            {
                bool isValid = true;
                if (value() >= 0)
                {
                    isValid = Application::application()->scheduler()
                                  ->validPort(value());
                }
                setBackgroundColor(isValid ? Qt::white : Qt::red);
            }
            QSpinBox::valueChange();
        }
        virtual int internalMapValueToDisplayValue(int v)
        {
            return midiPortNumber(v);
        }
};

class ChannelSpinBox : public DestSpinBox
{
    public:
        ChannelSpinBox(bool allow_no, bool allow_all, bool allow_same,
                       int value,
                       QWidget *parent = 0, const char *name = 0)
            : DestSpinBox(false, allow_no, allow_all, allow_same,
                          value, parent, name)
        {
            setMaxValue(15);
        }
        virtual int internalMapValueToDisplayValue(int v)
        {
            return v+1;
        }
};

/******************************************************************************
 * PortWidget class
 *****************************************************************************/

PortWidget::PortWidget(bool allow_bad,
                       bool allow_no, bool allow_all, bool allow_same,
                       int value, QWidget *parent, const char *name)
: QWidget(parent, name), allow_bad(allow_bad),
  allow_no(allow_no), allow_all(allow_all), allow_same(allow_same),
  _value(value)
{
    attachTo(Application::application()->scheduler());

    QHBoxLayout *layout = new QHBoxLayout(this, 0, KDialogBase::spacingHint());

    combo = new KComboBox(this);
    layout->addWidget(combo);

    spin = new PortSpinBox(allow_bad, allow_no, allow_all, allow_same, 1, this);
    layout->addWidget(spin);

    updateCombo();
    spin->setPortNumber(_value);
    combo->setCurrentItem(portNumberToComboIndex(value));

    connect(spin,  SIGNAL(valueChanged(int)), SLOT(slotNumberSelected(int)));
    connect(combo, SIGNAL(activated(int)),    SLOT(slotComboSelected(int)));
}


void PortWidget::MidiScheduler_PortAdded(TSE3::MidiScheduler *, size_t)
{
    updateCombo();
}


void PortWidget::MidiScheduler_PortRemoved(TSE3::MidiScheduler *, size_t)
{
    updateCombo();
}


void PortWidget::setValue(int value)
{
    _value = value;
    spin->setPortNumber(value);
    combo->setCurrentItem(portNumberToComboIndex(value));
}


void PortWidget::updateCombo()
{
    TSE3::MidiScheduler *scheduler = Application::application()->scheduler();
    combo->clear();
    if (allow_no)   combo->insertItem(NONE_STR);
    if (allow_all)  combo->insertItem(ALL_STR);
    if (allow_same) combo->insertItem(SAME_STR);
    for (size_t p = 0; p < scheduler->numPorts(); ++p)
    {
        QString str;
        str.setNum(scheduler->portNumber(p));
        str += ": ";
        str += scheduler->portName(scheduler->portNumber(p));
        combo->insertItem(str);
    }
}


void PortWidget::slotComboSelected(int index)
{
    TSE3::MidiScheduler *scheduler = Application::application()->scheduler();
    index -= allow_no + allow_all + allow_same;
    if (index >= 0)
    {
        _value = scheduler->portNumber(index);
        if (allow_bad)
        {
            spin->setValue(_value);
        }
        else
        {
            spin->setValue(index);
        }
    }
    else
    {
        if (combo->currentText() == NONE_STR)
        {
            _value = TSE3::MidiCommand::NoPort;
        }
        if (combo->currentText() == ALL_STR)
        {
            _value = TSE3::MidiCommand::AllPorts;
        }
        if (combo->currentText() == SAME_STR)
        {
            _value = TSE3::MidiCommand::SamePort;
        }
        spin->setValue(_value);
    }
    emit valueChanged(_value);
}


void PortWidget::slotNumberSelected(int value)
{
    _value = value;
    combo->setCurrentItem(spinValueToComboIndex(value));
    emit valueChanged(_value);
}


int PortWidget::spinValueToComboIndex(int value)
{
    TSE3::MidiScheduler *scheduler = Application::application()->scheduler();
    if (value < 0)
    {
        return -1-value;
    }
    else
    {
        int offset = allow_no + allow_all + allow_same;
        for (size_t index = 0; index < scheduler->numPorts(); ++index)
        {
            if (scheduler->portNumber(index) == value) return index+offset;
        }
        return -1;
    }
}


int PortWidget::portNumberToComboIndex(int value)
{
    if (value < 0)
    {
        return -1-spin->minusValueFromPortNumber(_value);
    }
    else
    {
        return spinValueToComboIndex(value);
    }
}


/******************************************************************************
 * ChannelWidget class
 *****************************************************************************/

ChannelWidget::ChannelWidget(bool allow_no, bool allow_all, bool allow_same,
                             int value, QWidget *parent, const char *name)
: QWidget(parent, name),
  allow_no(allow_no), allow_all(allow_all), allow_same(allow_same),
  _value(value)
{
    QHBoxLayout *layout = new QHBoxLayout(this, 0, 0);

    spin = new ChannelSpinBox(allow_no, allow_all, allow_same, _value, this);
    layout->addWidget(spin);
    spin->setValue(_value);

    connect(spin, SIGNAL(valueChanged(int)), SLOT(slotNumberSelected(int)));
}


void ChannelWidget::setValue(int value)
{
    _value = value;
    spin->setValue(value);
}


void ChannelWidget::slotNumberSelected(int value)
{
    _value = value;
    emit valueChanged(value);
}


/******************************************************************************
 * PortWidget::toText class
 *****************************************************************************/

QString PortWidget::toText(int channel, int port)
{
    QString ret;
    switch (channel)
    {
        case TSE3::MidiCommand::NoChannel:   ret.append(NONE_STR);        break;
        case TSE3::MidiCommand::AllChannels: ret.append(ALL_STR);         break;
        case TSE3::MidiCommand::SameChannel: ret.append(SAME_STR);        break;
        default:                             ret.sprintf("%d", channel+1);break;
    }
    ret.append(" : ");
    switch (port)
    {
        case TSE3::MidiCommand::NoPort:   ret.append(NONE_STR);   break;
        case TSE3::MidiCommand::AllPorts: ret.append(ALL_STR);    break;
        case TSE3::MidiCommand::SamePort: ret.append(SAME_STR);   break;
        default:
        {
            QString s;
            s.sprintf("%d", port+1);
            ret.append(s);
            break;
        }
    }
    return ret;
}
