/* ScoreScrollbar.java
 *
 * created: Thu Oct 21 1999
 *
 * This file is part of Artemis
 * 
 * Copyright (C) 1999  Genome Research Limited
 * 
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 *
 * $Header: /nfs/disk222/yeastpub/Repository/powmap/diana/components/ScoreScrollbar.java,v 1.1 1999/10/22 14:17:30 kmr Exp $
 */

package diana.components;

import java.awt.*;
import java.awt.event.*;

/**
 *  This component is a Scrollbar that can only have values from 1 to 100 and
 *  generates ScoreChange events when the value changes.
 *
 *  @author Kim Rutherford
 *  @version $Id: ScoreScrollbar.java,v 1.1 1999/10/22 14:17:30 kmr Exp $
 **/

public class ScoreScrollbar extends Scrollbar
  implements AdjustmentListener {
  /**
   *  Constructs a new horizontal score scroll bar with an initial value of 0.
   **/
  public ScoreScrollbar () {
    this (Scrollbar.HORIZONTAL, 0);
  }

  /**
   *  Constructs a new score scroll bar with the specified orientation.
   *
   *  The orientation argument must take one of the two values
   *  java.awt.Scrollbar.HORIZONTAL, or java.awt.Scrollbar.VERTICAL,
   *  indicating a horizontal or vertical scroll bar, respectively.
   *  @param orientation indicates the orientation of the scroll bar.
   *  @param value The initial value of the scrollbar.
   *  @exception IllegalArgumentException when an illegal value for the
   *    orientation argument is supplied or if the value parameter is less
   *    than 0 or greater than 100.
   **/
  public ScoreScrollbar (final int orientation, final int value)
      throws IllegalArgumentException {
    super (orientation, value, 1, 0, 101);

    if (value < 0 || value > 100) {
      throw new IllegalArgumentException ("score must be >= 1 and <= 100");
    }
  }

  /**
   *  Add the given ScoreChangeListener as a listener for ScoreChange events
   *  from this components.
   **/
  public void addScoreChangeListener (final ScoreChangeListener l) {
    score_change_listeners.addElement (l);

    if (score_change_listeners.size () == 1) {
      addAdjustmentListener (this);
    }
  }

  /**
   *  Remove the given ScoreChangeListener as a listener for ScoreChange
   *  events from this components.
   **/
  public void removeScoreChangeListener (final ScoreChangeListener l) {
    score_change_listeners.addElement (l);

    if (score_change_listeners.size () == 0) {
      removeAdjustmentListener (this);
    }
  }

  /**
   *  Implementation of the AdjustmentListener interface.
   **/
  public void adjustmentValueChanged (AdjustmentEvent e) {
    for (int i = 0 ; i < score_change_listeners.size () ; ++i) {
      final ScoreChangeEvent event =
        new ScoreChangeEvent (this, getValue ());
      final ScoreChangeListener this_listener =
        (ScoreChangeListener)(score_change_listeners.elementAt (i));
      this_listener.scoreChanged (event);
    }
  }

  /**
   *  The ScoreChangeListener objects that have been added with
   *  addScoreChangeListener ().
   **/
  final java.util.Vector score_change_listeners = new java.util.Vector ();
}
