/* Splash.java
 *
 * created: Wed May 10 2000
 *
 * This file is part of Artemis
 *
 * Copyright (C) 2000  Genome Research Limited
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 *
 * $Header: /nfs/disk222/yeastpub/Repository/powmap/diana/components/Splash.java,v 1.9 2000/09/04 12:52:53 kmr Exp $
 */

package diana.components;

import AppGlobal;

import diana.*;

import uk.ac.sanger.pathogens.*;

import java.awt.*;
import java.awt.event.*;
import java.net.*;
import java.io.*;

/**
 *  Base class that creates a generic "Splash Screen"
 *
 *  @author Kim Rutherford <kmr@sanger.ac.uk>
 *  @version $Id: Splash.java,v 1.9 2000/09/04 12:52:53 kmr Exp $
 **/

abstract public class Splash extends Frame {
  /**
   *  Create a new Frame for a Splash screen.
   *  @param entry_sources The possible sources for reading Entry objects.
   **/
  public Splash (final String program_name, final String program_version) {
    super (program_name + " " + program_version);

    this.program_name    = program_name;
    this.program_version = program_version;

    addWindowListener (new WindowAdapter () {
      public void windowClosing (WindowEvent event) {
        exit ();
      }
    });

    setLayout(new BorderLayout());

    final Font font = Options.getOptions ().getFont ();

    setFont (font);

    setMenuBar (menu_bar);

    makeAllMenus ();

    final Canvas helix_canvas = makeHelixCanvas ();

    add (helix_canvas, "Center");

    add (status_line, "South");

    pack ();

    setSize (450, 230);

    final Dimension screen = Toolkit.getDefaultToolkit().getScreenSize();
    setLocation (new Point ((screen.width - getSize ().width) / 2,
                            (screen.height - getSize ().height) / 2));
  }

  /**
   *  This method will fill the background with white.
   **/
  public void paint (final Graphics g) {
    g.setColor (Color.white);
    g.fillRect (0, 0, getSize ().width, getSize ().height);
  }

  /**
   *  Return a Canvas object that will display a helix and a short copyright
   *  notice.
   **/
  private Canvas makeHelixCanvas () {
    return new Canvas () {
      public void update(Graphics g) {
        paint(g);
      }
      
      public void sangerPaint (final Graphics g) {
        final Font canvas_font = new Font ("Monospaced", Font.PLAIN, 16);
        g.setFont (canvas_font);
        FontMetrics fm = getFontMetrics (canvas_font);

        final int font_height = fm.getHeight () + 3;

        g.setColor (Color.white);
        g.fillRect (0, 0, getSize ().width, getSize ().height);

        g.setColor (Color.black);

        g.drawString (program_name,
                      5, font_height);
        g.drawString (program_version,
                      5, font_height * 2);
        g.drawString ("Copyright 1998,1999,2000",
                      5, font_height * 3);
        g.drawString ("Genome Research Limited",
                      5, font_height * 4);
      }

      public void paint (final Graphics g) {
        final boolean simple_splash_screen =
          Options.getOptions ().getPropertyTruthValue ("simple_splash_screen");

        if (simple_splash_screen) {
          // java SIGILL bug work-around
          sangerPaint (g);
          return;
        }

        if (helix == null) {
          Toolkit toolkit = Toolkit.getDefaultToolkit ();

          final URL helix_url = Splash.class.getResource ("/diana/helix.gif");
          helix = toolkit.getImage (helix_url);

          final URL sanger_url =
            Splash.class.getResource ("/diana/sanger-centre.gif");
          sanger = toolkit.getImage (sanger_url);

          tracker = new MediaTracker (this);

          tracker.addImage (helix, 0);
          tracker.addImage (sanger, 1);

          try {
            tracker.waitForAll ();
            helix_height = helix.getHeight (this);
            helix_width = helix.getWidth (this);
            sanger_height = sanger.getHeight (this);
          } catch (InterruptedException e) {
            return;
          }
        }

        final Font canvas_font = new Font ("Monospaced", Font.PLAIN, 16);

        g.setFont (canvas_font);

        FontMetrics fm = getFontMetrics (canvas_font);

        final int font_height = fm.getHeight () + 3;

        g.setColor (Color.white);
        g.fillRect (0, 0, getSize ().width, getSize ().height);

        for (int i = 0 ; i * helix_height <= getSize ().height; ++i) {
          g.drawImage (helix,
                       0,
                       i * helix_height, this);
        }

        int sanger_position = getSize ().height - sanger_height;

        if (sanger_position > font_height * 4) {
          g.drawImage (sanger,
                       helix_width + 5,
                       sanger_position, this);
        }

        g.setColor (Color.black);

        g.drawString (program_name,
                      helix_width + 5, font_height);
        g.drawString (program_version,
                      helix_width + 5, font_height * 2);
        g.drawString ("Copyright 1998,1999,2000",
                      helix_width + 5, font_height * 3);
        g.drawString ("Genome Research Limited",
                      helix_width + 5, font_height * 4);
      }

      Image helix = null;
      Image sanger = null;

      MediaTracker tracker = null;

      int sanger_height;
      int helix_height;
      int helix_width;
    };
  }

  /**
   *  Return the reference of the Label used as a status line.
   **/
  public Label getStatusLabel () {
    return status_line;
  }

  /**
   *  Return the reference of the MenuBar of this component.
   **/
  public MenuBar getMenuBar () {
    return menu_bar;
  }

  /**
   *  The possible sources for reading Entry objects.
   **/
  public EntrySourceVector getEntrySources (final Frame frame) {
    return AppGlobal.getEntrySources (frame, stream_progress_listener);
  }

  /**
   *  Return an InputStreamProgressListener which updates the error label with
   *  the current number of chars read while reading
   **/
  public InputStreamProgressListener getInputStreamProgressListener () {
    return stream_progress_listener;
  }

  /**
   *  Make all the menus and menu items for the main window.  Also sets up
   *  suitable ActionListener objects for each item.
   */
  abstract protected void makeAllMenus ();

  /**
   *  Do any necessary cleanup then exit.
   **/
  abstract protected void exit ();

  /**
   *  A label for status and error messages.
   **/
  final private Label status_line = new Label ("");

  /**
   *  The program name that was passed to the constructor.
   **/
  final private String program_name;

  /**
   *  The program version that was passed to the constructor.
   **/
  final private String program_version;

  /**
   *  Menu bar for the main window.
   **/
  final private MenuBar menu_bar = new MenuBar ();

  /**
   *  An InputStreamProgressListener used to update the error label with the
   *  current number of chars read.
   **/
  private final InputStreamProgressListener stream_progress_listener =
    new InputStreamProgressListener () {
      public void progressMade (final InputStreamProgressEvent event) {
        final int char_count = event.getCharCount ();
        if (char_count == -1) {
          getStatusLabel ().setText ("");
        } else {
          getStatusLabel ().setText ("chars read so far: " + char_count);
        }
      }
    };
}
