/*
 * Read code for Storage daemon
 *
 *     Kern Sibbald, November MM
 *
 *   Version $Id: read.c,v 1.33 2004/04/18 21:59:51 kerns Exp $
 */
/*
   Copyright (C) 2000-2003 Kern Sibbald and John Walker

   This program is free software; you can redistribute it and/or
   modify it under the terms of the GNU General Public License as
   published by the Free Software Foundation; either version 2 of
   the License, or (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
   General Public License for more details.

   You should have received a copy of the GNU General Public
   License along with this program; if not, write to the Free
   Software Foundation, Inc., 59 Temple Place - Suite 330, Boston,
   MA 02111-1307, USA.

 */

#include "bacula.h"
#include "stored.h"

/* Forward referenced subroutines */
static int record_cb(JCR *jcr, DEVICE *dev, DEV_BLOCK *block, DEV_RECORD *rec);


/* Responses sent to the File daemon */
static char OK_data[]    = "3000 OK data\n";
static char FD_error[]   = "3000 error\n";
static char rec_header[] = "rechdr %ld %ld %ld %ld %ld";

/* 
 *  Read Data and send to File Daemon
 *   Returns: 0 on failure
 *	      1 on success
 */
int do_read_data(JCR *jcr) 
{
   BSOCK *fd = jcr->file_bsock;
   int ok = TRUE;
   DEVICE *dev;
   
   Dmsg0(20, "Start read data.\n");

   dev = jcr->device->dev;

   Dmsg1(10, "bstored>filed: %s\n", fd->msg);

   if (!bnet_set_buffer_size(fd, jcr->device->max_network_buffer_size, BNET_SETBUF_WRITE)) {
      return 0;
   }


   Dmsg1(20, "Begin read device=%s\n", dev_name(dev));

   create_vol_list(jcr);
   if (jcr->NumVolumes == 0) {
      Jmsg(jcr, M_FATAL, 0, _("No Volume names found for restore.\n"));
      free_vol_list(jcr);
      bnet_fsend(fd, FD_error);
      return 0;
   }

   Dmsg2(200, "Found %d volumes names to restore. First=%s\n", jcr->NumVolumes, 
      jcr->VolList->VolumeName);

   pm_strcpy(&jcr->VolumeName, jcr->VolList->VolumeName);

   /* 
    * Ready device for reading, and read records
    */
   if (!acquire_device_for_read(jcr)) {
      free_vol_list(jcr);
      return 0;
   }

   /* Tell File daemon we will send data */
   bnet_fsend(fd, OK_data);
   ok = read_records(jcr, dev, record_cb, mount_next_read_volume);

   /* Send end of data to FD */
   bnet_sig(fd, BNET_EOD);

   if (!release_device(jcr)) {
      ok = FALSE;
   }
   
   free_vol_list(jcr);
   Dmsg0(30, "Done reading.\n");
   return ok ? 1 : 0;
}

/*
 * Called here for each record from read_records()
 */
static int record_cb(JCR *jcr, DEVICE *dev, DEV_BLOCK *block, DEV_RECORD *rec)
{
   BSOCK *fd = jcr->file_bsock;
   int ok = TRUE;
   POOLMEM *save_msg;

   if (rec->FileIndex < 0) {
      return 1;
   }
   Dmsg5(100, "Send to FD: SessId=%u SessTim=%u FI=%d Strm=%d, len=%d\n",
      rec->VolSessionId, rec->VolSessionTime, rec->FileIndex, rec->Stream,
      rec->data_len);

   /* Send record header to File daemon */
   if (!bnet_fsend(fd, rec_header, rec->VolSessionId, rec->VolSessionTime,
	  rec->FileIndex, rec->Stream, rec->data_len)) {
      Dmsg1(30, ">filed: Error Hdr=%s\n", fd->msg);
      Jmsg1(jcr, M_FATAL, 0, _("Error sending to File daemon. ERR=%s\n"),
	 bnet_strerror(fd));
      return FALSE;
   } else {
      Dmsg1(30, ">filed: Hdr=%s\n", fd->msg);
   }


   /* Send data record to File daemon */
   save_msg = fd->msg;		/* save fd message pointer */
   fd->msg = rec->data; 	/* pass data directly to bnet_send */
   fd->msglen = rec->data_len;
   Dmsg1(30, ">filed: send %d bytes data.\n", fd->msglen);
   if (!bnet_send(fd)) {
      Pmsg1(000, "Error sending to FD. ERR=%s\n", bnet_strerror(fd));
      Jmsg1(jcr, M_FATAL, 0, _("Error sending to File daemon. ERR=%s\n"),
	 bnet_strerror(fd));

      ok = FALSE;
   }
   fd->msg = save_msg;		      /* restore fd message pointer */
   return ok;
}
