/*
** Beaver's an Early AdVanced EditoR
** (C) 1999-2000 Marc Bevand, Damien Terrier and Emmanuel Turquin
**
** editor.c
**
** Author<s>:   Marc Bevand (aka "After") <bevand_m@epita.fr>
**              Michael Terry <mterry@fastmail.fm>
**
** Description: Low-level text management (syntax highlighting,
**              auto-indentation, etc) and UltraEdit's "wordfile.txt"
**              parsing
**
** This program is free software; you can redistribute it and/or modify
** it under the terms of the GNU General Public License as published by
** the Free Software Foundation; either version 2 of the License, or
** (at your option) any later version.
**
** This program is distributed in the hope that it will be useful,
** but WITHOUT ANY WARRANTY; without even the implied warranty of
** MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
** GNU General Public License for more details.
**
** You should have received a copy of the GNU General Public License
** along with this program; if not, write to the Free Software
** Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

/*
further syhi optimizations : name tags, and if we are in a comment or string tag, don't do anything?
*/

/*
** NOTE_REMARK : syhi stands for SYntax HIghlighting, the main work to
**  be accomplished by the code in this file
*/

/*
** In order to view debugging messages, define some of these macros
*/
//#define DEBUG_AUTO_INDENT
//#define DEBUG_FCN
//#define DEBUG_WORDFILE
//#define DEBUG_EXT_REC
//#define DEBUG_SYHI
//#define DEBUG_CORRECT
//#define DEBUG_HASHTABLE
//#define DEBUG_FREEZE_THAW

#include <stdio.h>
#include <gtk/gtk.h>
#include <gdk/gdkkeysyms.h>
#include <string.h>
#include <ctype.h>
#include <sys/stat.h>
#include <stdlib.h>
#include "editor.h"
#include "struct.h"
#include "msgbar.h"
#include "tools.h"
#include "main.h"
#include "filesops.h"
#include "interface.h"
#include "conf.h"
#include "languages.h"
#include "gtksourceview/gtksourcelanguagesmanager.h"
#include "gtksourceview/gtksourceview.h"

extern t_settings Settings;

/*
** To access WidgetInfo, we need some external variables, and we do :
** FPROPS(gtk_notebook_get_current_page(GTK_NOTEBOOK(MainNotebook)),
**        WidgetInfo.<an_element_of_structure>)
*/
extern GtkWidget	*MainNotebook;
extern GArray		*FileProperties;

extern GtkSourceLanguagesManager *LanguagesManager;


/*
** This func is called during beaver initialization
**
** Parameters :
**   void
**
** Return values :
**   void
*/
void		editor_init(void)
{
	START_FCN
	END_FCN
}

/* if mime_type is null, we autodetect it */
void set_mime_type (gint page, const gchar *mime_type)
{
	GtkSourceLanguage *lang;
	GtkSourceBuffer *buffer;
	gchar *our_mime = NULL;
	
	START_FCN
	
	if (mime_type == NULL)
	{
  		our_mime = determine_mime_type (FPROPS (page, Name));
  		mime_type = our_mime;
  	}
  	
	buffer = GTK_SOURCE_BUFFER ((FPROPS (page, Buffer)));
	lang = gtk_source_languages_manager_get_language_from_mime_type 
		(LanguagesManager, mime_type);
	
	if (lang)
	{
		gtk_source_buffer_set_language (buffer, lang);
		gtk_source_buffer_set_highlight (buffer, TRUE);
		languages_activate_menu_item_for_language (lang);
	}
	else
	{
		gtk_source_buffer_set_highlight (buffer, FALSE);
		languages_activate_menu_item_for_language (NULL);
	}
	
	g_free (our_mime);
	END_FCN
}

void load_file_contents_into_buffer (gint page, const gchar *filename)
{
	gboolean success;
	GError *error = NULL;
	gchar *buffer;
	
	/* Read file */
	success = g_file_get_contents (filename, &buffer, NULL, &error);
	
	if (!success)
	{
		gchar *primary;
		
		primary = g_strdup_printf (_("Document \"%s\" cannot be opened."), 
			str_get_last_part (filename, G_DIR_SEPARATOR, TRUE));
		show_error (GTK_WINDOW (MainWindow), primary, error->message);
		g_free (primary);
		g_free (buffer);
		g_error_free (error);
	}
	else
	{
		const gchar *end;
		
		/* make sure we have valid utf8 */
		if (!g_utf8_validate (buffer, -1, &end))
		{
			GtkWidget *confirm_invalid_utf8;
			
			confirm_invalid_utf8 = gtk_message_dialog_new(GTK_WINDOW(MainWindow),
		                GTK_DIALOG_DESTROY_WITH_PARENT | GTK_DIALOG_MODAL,
		                GTK_MESSAGE_WARNING,
		                GTK_BUTTONS_YES_NO,
		                _("File \"%s\" does not qualify as valid UTF8. Attempt to convert to UTF8 and load anyway?"),
		                filename);
	             
			if (gtk_dialog_run(GTK_DIALOG(confirm_invalid_utf8)) == GTK_RESPONSE_NO)
			{
				g_free(buffer);
				gtk_widget_destroy(confirm_invalid_utf8);
				return;
			}
			else
			{
				G_CONST_RETURN char *charset;
				gsize bytes_read, bytes_written;
				
				/* fix me, let the user choose the charset */
				g_get_charset(&charset);
				buffer = g_convert_with_fallback(buffer, strlen(buffer),
					"UTF-8", "ISO-8859-1", "[INVALID CHAR]",
					&bytes_read, &bytes_written, &error);
				if (buffer == NULL)
				{
					g_print("conversion to utf8 failed at byte %d. reason why: %s\ncharset: %s\n", bytes_read, error->message, charset);
				}
			}
			gtk_widget_destroy(confirm_invalid_utf8);
			
	//  		primary = g_strdup_printf (_("Document \"%s\" cannot be opened."), 
	//  			str_get_last_part (Filename, PATH_SEP, TRUE));
	//	  	show_error (GTK_WINDOW (MainWindow), primary, 
	//  			"This file is not valid UTF-8 and will not be displayed.");
		
		}
		
		/* Insert file content in the widget */
		gtk_text_buffer_set_text (FPROPS (page, Buffer), buffer, -1);
		g_free (buffer);
	}
	
	set_mime_type (page, NULL);
}

/*
** Fcn called by external source to load content of a file in the widget
**
** Note: Filename can be NULL, in this case, the text widget is not filled
** in.
**
** Parameters :
**  Editor		The GtkWidget that display text
**  Filename		The file to be displayed in the widget
**
** Return values :
**  void
*/
void		open_file_in_editor(GtkWidget *Editor,
					    const gchar *Filename, gint CurrentPage)
{
  GtkTextIter		start;
  GtkTextBuffer		*Buffer;
  
  START_FCN
  
  Buffer = gtk_text_view_get_buffer (GTK_TEXT_VIEW (Editor));
  
  if (Filename)
  {
	gtk_source_buffer_begin_not_undoable_action (GTK_SOURCE_BUFFER (Buffer));
	load_file_contents_into_buffer (CurrentPage, Filename);
	gtk_source_buffer_end_not_undoable_action (GTK_SOURCE_BUFFER (Buffer));
  }
  else
  {
    gtk_source_buffer_set_highlight (GTK_SOURCE_BUFFER (Buffer), FALSE);
  }
  
  
  /* display line numbers */
  gtk_source_view_set_show_line_numbers (GTK_SOURCE_VIEW (Editor), Settings.show_line_numbers);
  
  /* display margin */
  gtk_source_view_set_show_margin (GTK_SOURCE_VIEW (Editor), Settings.show_margin);
  
  /* tab width */
  gtk_source_view_set_tabs_width (GTK_SOURCE_VIEW (Editor), Settings.tab_width);
  
  /* connect new buffer */
  FPROPS (CurrentPage, Buffer) = Buffer;
  
  gtk_text_buffer_get_start_iter(Buffer, &start);
  gtk_text_buffer_place_cursor(Buffer, &start);
  gtk_widget_grab_focus(Editor);
  gtk_text_buffer_set_modified(Buffer, FALSE);
  FPROPS(CurrentPage, Format) = get_format_type (CurrentPage);
  FPROPS(CurrentPage, LastSave) = time (NULL);
  
  gtk_source_view_set_auto_indent (GTK_SOURCE_VIEW (Editor), Settings.auto_indent);
  
  note_saveable ();
  
  g_signal_connect_swapped (G_OBJECT (Buffer), "can-undo",
  	G_CALLBACK (note_redoundo), GINT_TO_POINTER (CurrentPage));
  g_signal_connect_swapped (G_OBJECT (Buffer), "can-redo",
  	G_CALLBACK (note_redoundo), GINT_TO_POINTER (CurrentPage));
  g_signal_connect (G_OBJECT (Editor), "key-press-event",
  	G_CALLBACK (catch_mult_line_indents), GINT_TO_POINTER (CurrentPage));
  g_signal_connect(G_OBJECT(Buffer), "modified-changed",
	  G_CALLBACK (buffer_changed), NULL);
  
  note_format ();
  
  END_FCN
  return;
}

gboolean catch_mult_line_indents (GtkWidget *widget, GdkEventKey *event, gpointer data)
{
	if (event->keyval == GDK_Tab || event->keyval == GDK_ISO_Left_Tab)
	{
		gunichar ch;
		GtkTextMark *mark;
		GtkTextIter start, end;
		gint page;
		
		page = GPOINTER_TO_INT (data);
		
		gtk_text_buffer_get_selection_bounds (
			FPROPS (page, Buffer), &start, &end);
		
		if (gtk_text_iter_get_line (&start) ==
		    gtk_text_iter_get_line (&end))
		{
			return FALSE;
		}
		
		mark = gtk_text_buffer_create_mark (FPROPS (page, Buffer),
			NULL, &end, FALSE);
		gtk_text_buffer_begin_user_action (FPROPS (page, Buffer));
		
		/* we don't want to consider the line if it isn't really
		   being selected */
		if (gtk_text_iter_ends_line (&start))
		{
			gtk_text_iter_forward_char (&start);
		}
		
		gtk_text_iter_set_line_index (&start, 0);
		
		do
		{
			if (event->keyval == GDK_Tab) /* indent */
			{
				gtk_text_buffer_insert (FPROPS (page, Buffer),
					&start, "\t", -1);
			}
			else
			{
				/* find first tab and remove it */
				do
				{
					ch = gtk_text_iter_get_char (&start);
					
					if (ch == '\t')
					{
						GtkTextIter next = start;
						
						gtk_text_iter_forward_char (&next);
						gtk_text_buffer_delete (FPROPS (page, Buffer),
							&start, &next);
						break;
					}
					
				} while (ch && ch != '\r' && ch != '\n' &&
				         g_unichar_isspace (ch));
			}
			
			if (!gtk_text_iter_forward_line (&start))
			{
				break;
			}
			
			gtk_text_buffer_get_iter_at_mark (FPROPS (page, Buffer),
				&end, mark);
			
		} while (gtk_text_iter_compare (&start, &end) < 0);
		
		gtk_text_buffer_end_user_action (FPROPS (page, Buffer));
		gtk_text_buffer_delete_mark (FPROPS (page, Buffer), mark);
		
		return TRUE;
	}
	
	return FALSE;
}


#if 0
/*
** This func produces a total refresh of the syhi of the widget
** passed in argument
**
** Parameters :
**  Editor		Pointer to the GtkWidget to refresh
**  tos			Type Of Syhi, can be:
**			tos >= 0		manually selects the language
**			tos == SYHI_AUTODETECT	autodetects the language
**			tos == SYHI_DISABLE	syhi is disabled
**
** Return Values :
**   void
*/
extern void		refresh_editor(GtkWidget *Editor, gint tos)
{
  gint			Lg;
  gint			CurrentPage;
  GtkTextBuffer		*Buffer;
  GtkTextIter start, end;

#ifdef DEBUG_FCN
  g_print(__FILE__": %s(): Begin\n", __func__);
#endif
  CurrentPage = gtk_notebook_get_current_page (GTK_NOTEBOOK(MainNotebook));
  Buffer = FPROPS(CurrentPage, Buffer);
  /* Language detection */
  if (tos >= 0)
    {
      if (tos >= MAX_LANG)
	{
	  print_msg(_("*Bug!* Selected language number is not supported"));
	  Lg = -1;
	}
      else if (!Prefs.L[tos].IsDefined)
	{
	  print_msg(_("*Bug!* Selected language not defined in the wordfile"));
	  Lg = -1;
	}
      else
	  Lg = tos;
    }
  else if (tos == SYHI_AUTODETECT)
      Lg = guess_lang(); /* Note: guess_lang() can return -1 */
  else if (tos == SYHI_DISABLE)
      Lg = -1;
  else
    {
      print_msg(_("*Bug!* Invalid type of syhi"));
      Lg = -1;
    }
  /* Save the language */
  FPROPS(CurrentPage, WidgetInfo.Lg) = Lg;
  if (FPROPS (CurrentPage, WidgetInfo.markers))
  {
  	g_list_foreach (FPROPS (CurrentPage, WidgetInfo.markers), (GFunc) g_free, NULL);
  	g_list_free (FPROPS (CurrentPage, WidgetInfo.markers));
  }
  /* Do the refesh */
  gtk_text_buffer_get_bounds (Buffer, &start, &end);
  gtk_text_buffer_remove_all_tags (Buffer, &start, &end);
  gtk_widget_queue_draw (Editor);
  if (Lg == -1)
    {
      gint i;

      gtk_text_buffer_get_bounds (Buffer, &start, &end);
      gtk_text_buffer_remove_all_tags (Buffer, &start, &end);
	FPROPS(CurrentPage, WidgetInfo.string0_tag) = NULL;
	FPROPS(CurrentPage, WidgetInfo.string1_tag) = NULL;
	FPROPS(CurrentPage, WidgetInfo.comment_tag) = NULL;
	FPROPS(CurrentPage, WidgetInfo.comment_alt_tag) = NULL;
	FPROPS(CurrentPage, WidgetInfo.number_tag) = NULL;

	for (i = 0; i < MAX_COL; i++)
	{
		FPROPS(CurrentPage, WidgetInfo.keyword_tags[i]) = NULL;
	}
	
	FPROPS(CurrentPage, WidgetInfo.markers) = NULL;
	
    }
  else
    {
        gint i;
	for (i = 0; i < MAX_COL; i++)
	{
		FPROPS(CurrentPage, WidgetInfo.keyword_tags[i]) = gtk_text_buffer_create_tag (Buffer,
		    NULL, "foreground-gdk", &Prefs.Colors.L[Lg].C[i], NULL);
	}
	
	FPROPS(CurrentPage, WidgetInfo.number_tag) = gtk_text_buffer_create_tag (Buffer,
		    NULL, "foreground-gdk", &Prefs.Colors.L[Lg].Number, NULL);
	
	FPROPS(CurrentPage, WidgetInfo.string0_tag) = gtk_text_buffer_create_tag (Buffer,
		    NULL, "foreground-gdk", &Prefs.Colors.L[Lg].String0, NULL);
	FPROPS(CurrentPage, WidgetInfo.string1_tag) = gtk_text_buffer_create_tag (Buffer,
		    NULL, "foreground-gdk", &Prefs.Colors.L[Lg].String1, NULL);
	FPROPS(CurrentPage, WidgetInfo.comment_tag) = gtk_text_buffer_create_tag (Buffer,
		    NULL, "foreground-gdk", &Prefs.Colors.L[Lg].Comment, NULL);
	FPROPS(CurrentPage, WidgetInfo.comment_alt_tag) = gtk_text_buffer_create_tag (Buffer,
		    NULL, "foreground-gdk", &Prefs.Colors.L[Lg].CommentAlt, NULL);
	
	FPROPS(CurrentPage, WidgetInfo.markers) = NULL;
      refresh_syhi_all (GTK_TEXT_VIEW (Editor));
    }
#ifdef DEBUG_FCN
  g_print(__FILE__": %s(): End\n", __func__);
#endif
  return ;
}


void auto_indent_on_insert (GtkTextBuffer *Buffer, GtkTextIter *iter, gchar *text,
	gint size, gpointer data)
{
	GtkTextIter start, end;
	gboolean ends_in_brace;
	gunichar ch = 0;
	gchar *indentation, *tmp;
	gchar *indent_str;
	gint CurrentPage;
	
	START_FCN
	
	CurrentPage = gtk_notebook_get_current_page (GTK_NOTEBOOK(MainNotebook));
	
	if (!Settings.auto_indent || FPROPS(CurrentPage, WidgetInfo.Lg) == -1 ||
	    !Prefs.L[FPROPS(CurrentPage, WidgetInfo.Lg)].IndentString)
	{
		return;
	}
	
	if (size != 1 || !(*text == '\n' || *text == '\r'))
	{
		return;
	}
	
	indent_str = Prefs.L[FPROPS(CurrentPage, WidgetInfo.Lg)].IndentString;
	
	start = *iter;
	
	/* does this line end with an indent string? */
	do
	{
		if (gtk_text_iter_get_line_offset (&start) == 0)
			break;
		
		gtk_text_iter_backward_cursor_position (&start);
	}
	while (g_unichar_isspace ((ch = gtk_text_iter_get_char (&start))));
	
	gtk_text_iter_forward_char (&start);
	end = start;
	gtk_text_iter_backward_chars (&start, strlen (indent_str));
	
	tmp = gtk_text_iter_get_text (&start, &end);
	ends_in_brace = indent_str ? starts_string (tmp, 0, indent_str, NULL) : FALSE;
	g_free (tmp);
	
	/* ok, let's get indentation amount */
	gtk_text_iter_set_line_offset (&start, 0);
	
	ch = gtk_text_iter_get_char (&start);
	if (ch == '\r' || ch == '\n')
		return;
	
	end = start;
	while (g_unichar_isspace ((ch = gtk_text_iter_get_char (&end))))
	{
		if (ch == '\r' || ch == '\n')
			break;
		
		gtk_text_iter_forward_cursor_position (&end);
	}
	
	tmp = gtk_text_iter_get_text (&start, &end);
	
	if (strcmp (tmp, "") == 0 && !ends_in_brace)
	{
		g_free (tmp);
		return;
	}
	
	indentation = g_strdup_printf ("%s%s", text, tmp);
	g_free (tmp);
	
	if (ends_in_brace)
	{
		tmp = indentation;
		indentation = g_strdup_printf ("%s%s", indentation, "\t");
		g_free (tmp);
	}
	
	start = *iter;
	
	if (strcmp (indentation, "") != 0)
	{
		g_signal_stop_emission_by_name (Buffer, "insert-text");
		gtk_text_buffer_insert (Buffer, iter, indentation, -1);
	}
	
	g_free (indentation);
	
	END_FCN
}

void auto_unindent_on_insert (GtkTextBuffer *Buffer, GtkTextIter *iter, gchar *text,
	gint size, gpointer data)
{
	GtkTextIter start, origin;
	gchar *tmp;
	gchar *unindent_str;
	gint CurrentPage;
	
	START_FCN
	
	CurrentPage = gtk_notebook_get_current_page (GTK_NOTEBOOK(MainNotebook));
	
	if (!Settings.auto_indent || FPROPS(CurrentPage, WidgetInfo.Lg) == -1 ||
	    !Prefs.L[FPROPS(CurrentPage, WidgetInfo.Lg)].UnindentString)
	{
		return;
	}
	
	unindent_str = Prefs.L[FPROPS(CurrentPage, WidgetInfo.Lg)].UnindentString;
	
	if (strncmp (text, unindent_str, size))
		return;
	
	origin = start = *iter;
	
	gtk_text_iter_backward_chars (&origin, size);
	gtk_text_iter_set_line_offset (&start, 0);
	
	tmp = gtk_text_iter_get_text (&start, &origin);
	
	g_strchug (tmp);
	/* we only want to unindent if this is the first char */
	if (strcmp (tmp, ""))
		return;
	g_free (tmp);
	
	while (!gtk_text_iter_equal (&start, iter))
	{
		if (gtk_text_iter_get_char (&start) == '\t')
		{
			GtkTextIter after = start;
			gtk_text_iter_forward_char (&after);
			
			g_signal_stop_emission_by_name (Buffer, "insert-text"); // cause we are deleting here
			g_signal_handlers_block_by_func (
				Buffer, G_CALLBACK (refresh_syhi_on_delete), NULL);
			gtk_text_buffer_delete_interactive (Buffer, &start, &after, TRUE);
			g_signal_handlers_unblock_by_func (
				Buffer, G_CALLBACK (refresh_syhi_on_delete), NULL);
			break;
		}
		
		gtk_text_iter_forward_cursor_position (&start);
	}
	
	END_FCN
}
#endif
