/*
** Beaver's an Early AdVanced EditoR
** (C) 1999-2000 Marc Bevand, Damien Terrier and Emmanuel Turquin
**
** filesops.c
**
** Author<s>:     Emmanuel Turquin (aka "Ender") <turqui_e@epita.fr>
**                Michael Terry <mterry@fastmail.fm>
** Latest update: Sat Jun  3 20:48:56 2000
** Description:   Files operations source
**
** This program is free software; you can redistribute it and/or modify
** it under the terms of the GNU General Public License as published by
** the Free Software Foundation; either version 2 of the License, or
** (at your option) any later version.
** 
** This program is distributed in the hope that it will be useful,
** but WITHOUT ANY WARRANTY; without even the implied warranty of
** MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
** GNU General Public License for more details.
**
** You should have received a copy of the GNU General Public License
** along with this program; if not, write to the Free Software
** Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

#include <gtk/gtk.h>
#include <stdio.h>
#include <string.h>
#include <sys/stat.h>
#include "main.h"
#include "editor.h"
#include "struct.h"
#include "interface.h"
#include "msgbar.h"
#include "conf.h"
#include "prefs.h"
#include "filesops.h"
#include "tools.h"


#define OPEN_FILE       0
#define SAVE_FILE_AS    1
#define QUIT_FILE       0
#define CLOSE_FILE      1
#define CLOSE_ALL_FILE  2


extern GtkWidget *MainNotebook;
extern GArray *FileProperties;
extern t_settings Settings;
extern gint OpenedFilesCnt;
extern gint NewFilesCnt;
static gboolean FileSelectorIsVisible = FALSE;
static gboolean QuestionIsVisible = FALSE;
static gint AutosaveSig = 0;

/* This function returns 'string' with all instances
   of 'obj' replaced with instances of 'replacement'
   It modifies string and re-allocs it.
   */
gchar *str_replace_tokens (gchar **string, gchar obj, gchar *replacement)
{
	gchar *p;
	gint rsize = strlen (replacement);
	gint osize = 1;
	gint diff = rsize - osize;
	
	p = *string;
	while ((p = strchr (p, obj)))
	{
		*string = g_realloc (*string, strlen (*string) + diff + 1);
		g_memmove (p + rsize, p + osize, strlen (p + osize) + 1);
		
		memcpy (p, replacement, rsize);
		
		p = p + rsize;
	}
	
	return *string;
}

/* This function is used by 'init_file_properties' to set the base filename
   and the type of the file. */

const gchar *str_get_last_part (const gchar *String, gchar Separator,
			  gboolean ReturnIfSeparatorNotFound)
{
  const gchar *LastPart;
  
  LastPart = strrchr (String, Separator);
  if (LastPart)
  	LastPart += 1;
  else
  	LastPart = String;
  
  if (!ReturnIfSeparatorNotFound && !LastPart) return "";
  return LastPart;
}


/* Return the absolute path of the file 'FileName' */

gchar *get_absolute_path (const gchar *FileName)
{
  gchar *TempFileName, *AbsolutePath;
  gchar **Tab;
  gint i = 0;

  if (g_path_is_absolute (FileName))
    return (strdup (FileName));
  TempFileName = g_build_filename (g_get_current_dir (), FileName, NULL);
  Tab = g_strsplit (TempFileName, G_DIR_SEPARATOR_S, 0);
  g_free (TempFileName);
  while (Tab[i] != NULL)
    {
      if (!strcmp (Tab[i], ".."))
	{
	  gint j;
	  
	  for (j = i; Tab[j] != NULL; j++)
	    {
	      Tab[j-1] = Tab[j+1];
	    }
	  i--;
	}
      else i++;
    }
  AbsolutePath = g_strjoinv (G_DIR_SEPARATOR_S, Tab);
  g_strfreev (Tab);
  return (AbsolutePath);
}


/* This function is called when a text has changed (since last save) */

void buffer_changed (GtkTextBuffer *Buffer, gpointer user_data)
{
  gint CurrentPage;
  gboolean saveable;

#ifdef DEBUG_FCN
  g_print(__FILE__": %s(): Begin\n", __func__);
#endif
  CurrentPage = gtk_notebook_get_current_page (GTK_NOTEBOOK(MainNotebook));
  saveable = gtk_text_buffer_get_modified (FPROPS(CurrentPage, Buffer));
  set_label (GTK_NOTEBOOK(MainNotebook), CurrentPage);
  
  note_saveable ();
  
  print_msg(_("File has been modified."));
#ifdef DEBUG_FCN
  g_print(__FILE__": %s(): End\n", __func__);
#endif
}


/* This function is called when you want a text to be readonly (or not) */

void toggle_readonly (void)
{
  gint CurrentPage;

  if (!OpenedFilesCnt) return;
  CurrentPage = gtk_notebook_get_current_page (GTK_NOTEBOOK(MainNotebook));
  if (FPROPS(CurrentPage, ReadOnly) == 0)
    {
      gtk_text_view_set_editable (GTK_TEXT_VIEW(FPROPS(CurrentPage, Text)), FALSE);
      FPROPS(CurrentPage, ReadOnly) = 1;
      set_label (GTK_NOTEBOOK(MainNotebook), CurrentPage);
      print_msg (_("Read-only mode activated"));
    }
  else if (FPROPS(CurrentPage, ReadOnly) == 1)
    {
      gtk_text_view_set_editable (GTK_TEXT_VIEW(FPROPS(CurrentPage, Text)), TRUE);
      FPROPS(CurrentPage, ReadOnly) = 0;
      set_label (GTK_NOTEBOOK(MainNotebook), CurrentPage);
      print_msg (_("Read-only mode deactivated"));      
    }
  else print_msg (g_strdup_printf(_("Readonly mode cannot be toggled - \"%s\" is write protected."), FPROPS(CurrentPage, BaseName)));
}


/* Init the selected file properties */

void init_file_properties (const gchar *FileName, gint CurrentPage)
{
  struct stat Stats;
  
  START_FCN
  
  FPROPS(CurrentPage, Name) = g_strdup (FileName);
  FPROPS(CurrentPage, BaseName) = g_path_get_basename (FileName);
  FPROPS(CurrentPage, Type) =
    g_strdup (str_get_last_part (FPROPS(CurrentPage, BaseName), '.', FALSE));
  if (stat (FileName, &Stats) != -1)
    {
      FILE *File;

      if (((File = fopen (FPROPS(CurrentPage, Name), "a")) != NULL))
	{ 
	  FPROPS(CurrentPage, ReadOnly) = 0;
	  fclose (File);
	}
      else
	{
	  FPROPS(CurrentPage, ReadOnly) = -1;
	}
    }
  else FPROPS(CurrentPage, ReadOnly) = 0;
  stat (FileName, &FPROPS(CurrentPage, Stats));
  FPROPS (CurrentPage, Format) = UNIX; /* defaults to unix */
  
  END_FCN
}


/* Return the rwx permissions of a file in a string */

gchar *get_file_mode (struct stat Stats)
{
  static gchar Mode[10];
  
  g_snprintf (Mode, 10, "%c%c%c%c%c%c%c%c%c",
	      (Stats.st_mode & S_IRUSR) ? 'r' : '-',
	      (Stats.st_mode & S_IWUSR) ? 'w' : '-',
	      (Stats.st_mode & S_IXUSR) ? 'x' : '-',
	      (Stats.st_mode & S_IRGRP) ? 'r' : '-',
	      (Stats.st_mode & S_IWGRP) ? 'w' : '-',
	      (Stats.st_mode & S_IXGRP) ? 'x' : '-',
	      (Stats.st_mode & S_IROTH) ? 'r' : '-',
	      (Stats.st_mode & S_IWOTH) ? 'w' : '-',
	      (Stats.st_mode & S_IXOTH) ? 'x' : '-');
  return (Mode);
}


/* Update the recent files by putting a new file in the list */

/* these entries represent the entries after the recent files section that we modify */
static GtkItemFactoryEntry EndingEntries[] = {
  {"/File/sep_end", NULL, NULL, 0, "<Separator>"},
  {"/File/_Close", "<control>W", menu_items_treatment, CLOSE, "<StockItem>", GTK_STOCK_CLOSE},
  {"/File/_Quit", "<control>Q", menu_items_treatment, QUIT, "<StockItem>", GTK_STOCK_QUIT}
};

void init_recent_files (void)
{
#ifdef DEBUG_FCN
  g_print(__FILE__": %s(): Begin\n", __func__);
#endif
	display_recent_files ();
#ifdef DEBUG_FCN
  g_print(__FILE__": %s(): End\n", __func__);
#endif
}


void put_recent_file (const gchar *FileName)
{
  gint i;
  gchar *last_val;
  
#ifdef DEBUG_FCN
  g_print(__FILE__": %s(): Begin\n", __func__);
#endif
  last_val = g_strdup (FileName);
  for (i = 1; i <= RECENT_FILES_MAX_NB; i++)
    {
    	gchar *this_key;
    	gchar *this_val;
    	
    	this_key = g_strdup_printf ("General/RecentFiles/File%d", i);
    	this_val = get_string_conf (this_key);
    	
    	set_string_conf (this_key, last_val);
    	g_free (last_val);
    	
    	if (strcmp (FileName, this_val))
    		last_val = this_val;
    	else
    		break;
    }
    
  display_recent_files ();
  
#ifdef DEBUG_FCN
  g_print(__FILE__": %s(): End\n", __func__);
#endif
}


/* Used to put the most recently opened files in a menu */

void display_recent_files (void)
{
  gint i = 0, displayed = 0;
  gboolean done = FALSE;
  gboolean close_sensitive;
  GtkWidget *item;
  
  START_FCN
  
  /* first, get rid of old entries (plus close and quit) */
  i = RECENT_FILES_OFFSET;
  while ((item = gtk_item_factory_get_item_by_action (MainFactory, ++i)))
  {
  	const gchar *path = gtk_item_factory_path_from_widget (item);
  	gtk_item_factory_delete_item (MainFactory, path);
  }
  
  close_sensitive = GTK_WIDGET_SENSITIVE (
  	gtk_item_factory_get_item (MainFactory, "/File/Close"));
  gtk_item_factory_delete_entries (MainFactory, 3, EndingEntries);
  
  /* now add new entries */
  for (i = 1; i <= RECENT_FILES_MAX_NB && !done; i++)
  {
	gchar *conf_val, *conf_key;
	
	conf_key = g_strdup_printf ("General/RecentFiles/File%d", i);
	conf_val = get_string_conf (conf_key);
	
	str_replace_tokens (&conf_val, '_', "__");
	
	if (strcmp (conf_val, ""))
	{
		gchar *base = g_path_get_basename (conf_val); 
		/* FIXME 1: make it work - how to escape those '/' ? -- Leslie */
		//gchar *base = g_strdup(conf_val);
		/* FIXME 2: then make display configurable (Glade prefs) -- Leslie */
		gchar *menu_title = g_strdup_printf ("/File/_%d %s", i,
			base);
		GtkItemFactoryEntry NewEntry  = {
			menu_title, NULL,
			(GtkItemFactoryCallback) open_recent_file,
			RECENT_FILES_OFFSET + i, "<Item>"};
		gtk_item_factory_create_items
			(MainFactory, 1, &NewEntry, NULL);
		
		g_free (base);
		g_free (menu_title);
		
		displayed++;
	}
	else
		done = TRUE;
	
	g_free (conf_key);
	g_free (conf_val);
    }
    
    /* and add back the close and quit items */
    if (displayed > 0)
	gtk_item_factory_create_items (MainFactory, 3, EndingEntries, NULL);
    else
    	gtk_item_factory_create_items (MainFactory, 2, &EndingEntries[1], NULL);
    
    gtk_widget_set_sensitive (
    	gtk_item_factory_get_item (MainFactory, "/File/Close"),
    	close_sensitive);
    
    END_FCN
}


/* Callback function for the Recent files menu */

void open_recent_file (GtkWidget *DummyWidget, guint i)
{
  struct stat Stats;
  gchar *FileName, *conf_key;
  
  START_FCN
  
  i = i - RECENT_FILES_OFFSET;

  conf_key = g_strdup_printf ("General/RecentFiles/File%d", i);
  FileName = get_string_conf (conf_key);

  if (stat (FileName, &Stats) == -1)
    {
      print_msg (_("This file does not exist anymore."));      
      return;
    }
  else
  {
  	open_filename (FileName);
  }
  (void)DummyWidget; /* avoid the "unused parameter" warning */
  g_free (conf_key);
  g_free (FileName);
  
  END_FCN
}


/* Enable/Disable Autosave */

void autosave (gint Delay)
{
  if (Delay)
    {
      if (AutosaveSig)
	{
	  gtk_timeout_remove (AutosaveSig);
	}
      AutosaveSig = gtk_timeout_add (1000 * Delay,
				     (GtkFunction)save_all,
				     NULL);
    }
  else
    if (AutosaveSig)
      {
	gtk_timeout_remove (AutosaveSig);
	AutosaveSig = 0;
      }
}

gboolean save_a_copy_window_new (void)
{
  GtkWidget *FileSelector;
  gchar *Title, *Directory;
  gint response;
  const gchar *file;
  gboolean rv = TRUE;

  Title = g_strdup_printf(_("Save a Copy of \"%s\" As..."),
  	FPROPS(gtk_notebook_get_current_page(GTK_NOTEBOOK(MainNotebook)), BaseName));
  FileSelector = gtk_file_selection_new (Title);
  Directory = DIRECTORY;
  gtk_file_selection_set_filename (GTK_FILE_SELECTION(FileSelector),
                                   Directory);
  gtk_window_set_modal (GTK_WINDOW(FileSelector), TRUE);
  response = gtk_dialog_run (GTK_DIALOG (FileSelector));
  
  file = gtk_file_selection_get_filename (GTK_FILE_SELECTION (FileSelector));
  
  gtk_widget_destroy (FileSelector);
  switch (response)
  {
  case GTK_RESPONSE_OK:
	save_a_copy_func (file);
	break;
  default:
	rv = FALSE;
	break;
  }
  
  g_free (Title);
  
  return rv;
}

void save_a_copy_func (const gchar *FileName)
{
  FILE *File;
  struct stat Stats;
  gint CurrentPage;

  CurrentPage = gtk_notebook_get_current_page (GTK_NOTEBOOK(MainNotebook));
  if (stat (FileName, &Stats) != -1)
    if ((Stats.st_mode & S_IFMT) == S_IFDIR) return;
  if ((File = fopen (FileName, "w")))
    {
      gchar *Buffer;  
      GtkWidget *CurrentText;
      gint length;
 
      CurrentText = FPROPS(CurrentPage, Text);
      Buffer = get_text (gtk_text_view_get_buffer(
                         GTK_TEXT_VIEW(CurrentText)));
      length = gtk_text_buffer_get_char_count(
                   gtk_text_view_get_buffer(GTK_TEXT_VIEW(CurrentText)));
      fwrite (Buffer, length, 1, File);
      g_free (Buffer);
      fclose (File);
      print_msg (g_strdup_printf (_("File \"%s\" saved."), FPROPS(CurrentPage, BaseName)));
    }
  else
    {
      print_msg (g_strdup_printf (_("Unable to save - \"%s\" is write-protected."), FPROPS(CurrentPage, BaseName)));
    }
 }

gboolean save_file_as_window_new (void)
{
  GtkWidget *FileSelector;
  gchar *Title, *Directory;
  gint response;
  const gchar *file;
  gboolean rv = TRUE;

  Title = g_strdup_printf(_("Save \"%s\" as..."),
  	FPROPS(gtk_notebook_get_current_page(GTK_NOTEBOOK(MainNotebook)), BaseName));
  FileSelector = gtk_file_selection_new (Title);
  Directory = DIRECTORY;
  gtk_file_selection_set_filename (GTK_FILE_SELECTION(FileSelector),
                                   Directory);
  gtk_window_set_modal (GTK_WINDOW(FileSelector), TRUE);
  response = gtk_dialog_run (GTK_DIALOG (FileSelector));
  
  file = gtk_file_selection_get_filename (GTK_FILE_SELECTION (FileSelector));
  
  gtk_widget_destroy (FileSelector);
  switch (response)
  {
  case GTK_RESPONSE_OK:
	save_file_as_func (file);
	break;
  default:
	rv = FALSE;
	break;
  }
  
  g_free (Title);
  
  return rv;
}

/* Return the appropriate File Selection window, using one of the two
   functions below (i.e 'save_file_as_func' or 'open_file_func') */

gboolean open_file_window_new (void)
{
  GtkWidget *FileSelector;
  gint response;
  gchar **files;
  gint i;
  gboolean rv = TRUE;

  FileSelector = gtk_file_selection_new (_("Open file(s)..."));
  gtk_file_selection_set_filename (GTK_FILE_SELECTION(FileSelector),
                                   DIRECTORY);
  gtk_window_set_modal (GTK_WINDOW(FileSelector), TRUE);
  gtk_file_selection_set_select_multiple (GTK_FILE_SELECTION (FileSelector),
  	TRUE);
  response = gtk_dialog_run (GTK_DIALOG (FileSelector));
  
  files = gtk_file_selection_get_selections (GTK_FILE_SELECTION (FileSelector));
  
  gtk_widget_destroy (FileSelector);
  switch (response)
  {
  case GTK_RESPONSE_OK:
  	for (i = 0; files[i]; i++)
  	{
		open_filename (files[i]);
	}
	break;
  default:
	rv = FALSE;
	break;
  }
  
  g_strfreev (files);
  
  return rv;
}


/* Used to save a file as... */

void save_file_as_func (const gchar *FileName)
{
  FILE *File;
  struct stat Stats;
  gint CurrentPage;

  CurrentPage = gtk_notebook_get_current_page (GTK_NOTEBOOK(MainNotebook));
  if (stat (FileName, &Stats) != -1)
    if ((Stats.st_mode & S_IFMT) == S_IFDIR) return;
  if ((File = fopen (FileName, "w")))
    {
      gchar *Buffer;  
      GtkWidget *CurrentText;
      gint length;
 
      put_recent_file (FileName);
      CurrentText = FPROPS(CurrentPage, Text);
      Buffer = get_text (gtk_text_view_get_buffer(
                         GTK_TEXT_VIEW(CurrentText)));
      length = gtk_text_buffer_get_char_count(
                   gtk_text_view_get_buffer(GTK_TEXT_VIEW(CurrentText)));
      fwrite (Buffer, length, 1, File);
      g_free (Buffer);
      fclose (File);
      init_file_properties (FileName, CurrentPage);
      set_mime_type (CurrentPage, NULL);
      set_label (GTK_NOTEBOOK(MainNotebook), CurrentPage);
      set_title (CurrentPage);
      print_msg (g_strdup_printf (_("File \"%s\" saved."), FPROPS(CurrentPage, BaseName)));
      FPROPS(CurrentPage, LastSave) = time (NULL);
      gtk_text_buffer_set_modified (FPROPS(CurrentPage, Buffer), FALSE);
    }
  else
    {
      print_msg (g_strdup_printf (_("Unable to save - \"%s\" is write-protected."), FPROPS(CurrentPage, BaseName)));
    }
}

/* actually does the work of opening a file */
void open_filename (const gchar *filename)
{
//  gint CurrentPage;
  struct stat Stats;
  gchar *dir;
  
  START_FCN
  
  if (stat (filename, &Stats) == -1) return;
  if ((Stats.st_mode & S_IFMT) == S_IFDIR) return;
  
  g_free (DIRECTORY);
  dir = g_path_get_dirname (filename);
  DIRECTORY = g_strconcat (dir, G_DIR_SEPARATOR_S, NULL);
  g_free (dir);
  set_string_conf ("General/RecentFiles/Directory", DIRECTORY);
  
  put_recent_file (filename);
  add_page_in_notebook (GTK_NOTEBOOK(MainNotebook), filename);
  open_file_in_editor(GTK_WIDGET(FPROPS(OpenedFilesCnt - 1, Text)), filename, OpenedFilesCnt - 1);
  if (FPROPS(OpenedFilesCnt - 1, ReadOnly))
    print_msg (g_strdup_printf (_("Opened file \"%s\" read-only."), FPROPS(OpenedFilesCnt - 1, BaseName)));
  else
    print_msg (g_strdup_printf (_("Opened file \"%s\"."), FPROPS(OpenedFilesCnt - 1, BaseName)));
  
  END_FCN
}


void new_file (void)
{
  add_page_in_notebook (GTK_NOTEBOOK(MainNotebook), NULL);
  open_file_in_editor(GTK_WIDGET(FPROPS(OpenedFilesCnt - 1, Text)), NULL, OpenedFilesCnt - 1);
  gtk_text_buffer_set_modified (FPROPS(OpenedFilesCnt - 1, Buffer), TRUE);

  note_saveable ();
}


void open_file (void)
{ 
	open_file_window_new ();
}


void save_file (void)
{
  FILE *File;
  gint CurrentPage;

#ifdef DEBUG_FCN
  g_print(__FILE__": %s(): Begin\n", __func__);
#endif
  if (!OpenedFilesCnt) return;
  CurrentPage = gtk_notebook_get_current_page (GTK_NOTEBOOK(MainNotebook));
  if (!gtk_text_buffer_get_modified (FPROPS(CurrentPage, Buffer)))
    {
      print_msg (_("File does not need to be saved"));
      return;  
    }
  if (FPROPS(CurrentPage, ReadOnly))
    save_file_as();
  else if ((stat (FPROPS(CurrentPage, Name), &FPROPS(CurrentPage, Stats)) == -1)
      && (!strncmp (FPROPS(CurrentPage, Name), _("Untitled "), 9)))
    save_file_as();
  else
    {
      if ((BACKUP) && (stat (FPROPS(CurrentPage, Name),
			     &FPROPS(CurrentPage, Stats)) != -1))
	{
	  gchar *BackupName;
	  
	  BackupName = g_strconcat (FPROPS(CurrentPage, Name),
				    BACKUP_EXT, NULL);
	  rename (FPROPS(CurrentPage, Name), BackupName);
	  g_free (BackupName);
	}
      if ((File = fopen (FPROPS(CurrentPage, Name), "w")))
	{
	  gchar *Buffer;
	  gint	length;

          Buffer = get_text (gtk_text_view_get_buffer(
                         GTK_TEXT_VIEW(FPROPS(CurrentPage, Text))));
          length = get_num_characters(GTK_TEXT_VIEW(
            FPROPS(CurrentPage, Text)));
	  fwrite (Buffer, length, 1, File);
	  g_free (Buffer);
	  fclose (File);
	  print_msg (g_strdup_printf (_("File \"%s\" saved."), FPROPS(CurrentPage, Name))); 
          FPROPS(CurrentPage, LastSave) = time (NULL);
	  gtk_text_buffer_set_modified (FPROPS(CurrentPage, Buffer), FALSE);
	}
      else
	{
          print_msg (g_strdup_printf (_("Unable to save - \"%s\" is write-protected."), FPROPS(CurrentPage, BaseName)));		
	  save_file_as();
	}
    }
#ifdef DEBUG_FCN
  g_print(__FILE__": %s(): End\n", __func__);
#endif
}

void save_file_as (void)
{ 
  if (!OpenedFilesCnt) return;
  save_file_as_window_new ();
}


void save_all (void)
{
  gint CurrentPage, i;

  if (!OpenedFilesCnt) return;
  CurrentPage = gtk_notebook_get_current_page (GTK_NOTEBOOK(MainNotebook));
  for (i = 0; i < OpenedFilesCnt; i++)
    {
      gtk_notebook_set_current_page (GTK_NOTEBOOK(MainNotebook), i);
      save_file ();
    }
  gtk_notebook_set_current_page (GTK_NOTEBOOK(MainNotebook), CurrentPage);
}

void save_a_copy (void)
{
  if (!OpenedFilesCnt) return;
  save_a_copy_window_new ();
}


void revert (void)
{
  GtkWidget *dialog;
  gint response, CurrentPage = gtk_notebook_get_current_page (GTK_NOTEBOOK(MainNotebook));
  gboolean rv = TRUE;
  gchar *primary, *secondary, *time_str;
  time_t current_time = time (NULL);
  int elapsed_time = difftime (current_time, FPROPS(CurrentPage, LastSave));

  if (!OpenedFilesCnt) return;

  gtk_notebook_set_current_page (GTK_NOTEBOOK(MainNotebook), CurrentPage);
  if (elapsed_time > 3600)
	time_str = g_strdup_printf ("%i hour%s", elapsed_time / 3600, ((elapsed_time / 3600) == 1) ? "" : "s");
  else if (elapsed_time < 60)
	time_str = g_strdup_printf ("%i second%s", elapsed_time, (elapsed_time == 1) ? "" : "s");
  else
	time_str = g_strdup_printf ("%i minute%s", elapsed_time / 60, ((elapsed_time / 60) == 1) ? "" : "s");

  primary = g_strdup_printf (_("Revert to saved copy of \"%s\"?"), 
	 FPROPS(CurrentPage, BaseName));
  secondary = g_strdup_printf (_("If you revert, changes from the last %s will be discarded."), time_str);
  dialog = alert_new (GTK_WINDOW(MainWindow), GTK_STOCK_DIALOG_WARNING,
	primary, secondary);
  gtk_dialog_add_buttons (GTK_DIALOG (dialog), 
	GTK_STOCK_CANCEL, 1, GTK_STOCK_REVERT_TO_SAVED, 2, NULL);
  gtk_dialog_set_default_response (GTK_DIALOG (dialog), 3);
  response = gtk_dialog_run (GTK_DIALOG (dialog));
  gtk_widget_destroy (dialog);

  g_free (time_str);
  g_free (secondary);
  g_free (primary);

  switch (response)
  {
  case GTK_RESPONSE_NONE: /* if the user somehow closes the dialog window, just cancel */
  case 1:
    rv = FALSE; break;
  case 2:
    revert_func (); break;
  }
}

void revert_func (void)
{
  gint CurrentPage = gtk_notebook_get_current_page (GTK_NOTEBOOK(MainNotebook));
  
  open_file_in_editor(GTK_WIDGET(FPROPS(CurrentPage, Text)),
  	FPROPS (CurrentPage, Name), CurrentPage);
}


/* Display a Dialog box which ask you wether you wanna save or not the
   modified files when you quit or when you close these files */
/* returns false if the action is cancelled, true if not */
gboolean question_window_show (gint CurrentPage)
{
  GtkWidget *dialog;
  gint response;
  gboolean rv = TRUE;
  gchar *primary, *secondary, *time_str;
  time_t current_time = time (NULL);
  int elapsed_time = difftime (current_time, FPROPS(CurrentPage, LastSave));

  gtk_notebook_set_current_page (GTK_NOTEBOOK(MainNotebook), CurrentPage);
  if (elapsed_time > 3600)
	time_str = g_strdup_printf ("%i hour%s", elapsed_time / 3600, ((elapsed_time / 3600) == 1) ? "" : "s");
  else if (elapsed_time < 60)
	time_str = g_strdup_printf ("%i second%s", elapsed_time, (elapsed_time == 1) ? "" : "s");
  else
	time_str = g_strdup_printf ("%i minute%s", elapsed_time / 60, ((elapsed_time / 60) == 1) ? "" : "s");

  primary = g_strdup_printf (_("Save changes to document \"%s\" before closing?"), 
	 FPROPS(CurrentPage, BaseName));
  secondary = g_strdup_printf (_("If you close without saving, changes from the last %s will be discarded."), time_str);
  dialog = alert_new (GTK_WINDOW(MainWindow), GTK_STOCK_DIALOG_WARNING,
	primary, secondary);
  gtk_dialog_add_buttons (GTK_DIALOG (dialog), _("Close _without saving"), 1, 
	GTK_STOCK_CANCEL, 2, GTK_STOCK_SAVE, 3, NULL);
  gtk_dialog_set_default_response (GTK_DIALOG (dialog), 3);
  response = gtk_dialog_run (GTK_DIALOG (dialog));
  gtk_widget_destroy (dialog);

  g_free (time_str);
  g_free (secondary);
  g_free (primary);

  switch (response)
  {
  case 1:
    close_file_func (CurrentPage); break;
  case GTK_RESPONSE_NONE: /* if the user somehow closes the dialog window, just cancel */
  case 2:
    rv = FALSE; break;
  case 3:
    save_file_before_close_func ();
    close_file_func (CurrentPage); break;
  }

  QuestionIsVisible = FALSE;
  return rv;
}


/*  Used to close a file */

void close_file_func (gint CurrentPage)
{
  START_FCN
  gtk_notebook_remove_page (GTK_NOTEBOOK(MainNotebook), CurrentPage);
  g_free (FPROPS(CurrentPage, Name));
  g_free (FPROPS(CurrentPage, BaseName));
  g_free (FPROPS(CurrentPage, Type));
  gtk_widget_destroyed (FPROPS(CurrentPage, Text),
			&FPROPS(CurrentPage, Text));
  g_array_remove_index (FileProperties, CurrentPage);
  OpenedFilesCnt--;
  print_msg (_("File closed."));
  menu_manage_documents ();
  if (!OpenedFilesCnt)
    {
      set_title (-1);
      NewFilesCnt = 0;
      menu_set_files_open (FALSE);
    }
  END_FCN
}


/*  Used to save a file just before closing it */
 
void save_file_before_close_func (void)
{
  gtk_notebook_set_current_page (GTK_NOTEBOOK(MainNotebook),
			 gtk_notebook_get_current_page
			 (GTK_NOTEBOOK(MainNotebook)));
  save_file ();
  while (FileSelectorIsVisible) gtk_main_iteration_do (FALSE);
}


void close_file (void)
{
  gint CurrentPage;

  if (!OpenedFilesCnt) return;
  CurrentPage = gtk_notebook_get_current_page (GTK_NOTEBOOK(MainNotebook));
  if (gtk_text_buffer_get_modified (FPROPS (CurrentPage, Buffer)))
  {
    question_window_show (CurrentPage);
  }
  else
    close_file_func (CurrentPage);
}


void close_all (void)
{
  gint i;
  
  if (!OpenedFilesCnt) return; 
  print_msg (_("Closing all files..."));

  for (i = OpenedFilesCnt-1; i >= 0; i--)
    {
      if (gtk_text_buffer_get_modified (FPROPS (i, Buffer)))
      {
        gtk_notebook_set_current_page (GTK_NOTEBOOK(MainNotebook), i);
        if (question_window_show (i))
        {
        }
        else
        {
          return;
        }
      }
    }
  for (i = OpenedFilesCnt-1; i >= 0; i--)
    {
      close_file_func (i);
    }
  if (!OpenedFilesCnt)
    {
      print_msg (_("All files closed."));
      set_title (-1);
      NewFilesCnt = 0;
    }
}


gboolean quit (void)
{
  gboolean do_quit = TRUE;

#ifdef DEBUG_FCN
  g_print(__FILE__": %s(): Begin\n", __func__);
#endif
  set_preferences_to_disk (&Settings, NULL);
  if (!OpenedFilesCnt){ gtk_main_quit ();}
  else
    {
      gint i;

      for (i = OpenedFilesCnt -1 ; i >= 0 && do_quit; i--)
      {
            if (gtk_text_buffer_get_modified (FPROPS (i, Buffer)))
	    {
	      do_quit = question_window_show (i);
              if (do_quit) // file closed
              {
                
              }
	    }
	}
      if (do_quit) gtk_main_quit ();
    }

#ifdef DEBUG_FCN
  g_print(__FILE__": %s(): End\n", __func__);
#endif

  return !do_quit;
}
