/*
** Beaver's an Early AdVanced EditoR
** (C) 1999-2000 Marc Bevand, Damien Terrier and Emmanuel Turquin
**
** interface.c
**
** Author<s>:     Emmanuel Turquin (aka "Ender") <turqui_e@epita.fr>
**                Michael Terry <mterry@fastmail.fm>
**		  Leslie Polzer <leslie.polzer@gmx.net>
** Description:   Beaver interface source
**
** This program is free software; you can redistribute it and/or modify
** it under the terms of the GNU General Public License as published by
** the Free Software Foundation; either version 2 of the License, or
** (at your option) any later version.
**
** This program is distributed in the hope that it will be useful,
** but WITHOUT ANY WARRANTY; without even the implied warranty of
** MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
** GNU General Public License for more details.
**
** You should have received a copy of the GNU General Public License
** along with this program; if not, write to the Free Software
** Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

#include <gtk/gtk.h>
#include <stdio.h>
#include <string.h>
#include <sys/stat.h>
#include <time.h>
#include "main.h"
#include "editor.h"
#include "struct.h"
#include "filesops.h"
#include "msgbar.h"
#include "toolbar.h"
#include "conf.h"
#include "prefs.h"
#include "languages.h"
#include "tools.h"
#include "search.h"
#include "completion.h"
#include "interface.h"
#include "gtksourceview/gtksourceview.h"


/* The 'about' pixmap */

#include "../pixmaps/about.xpm"
#include "../pixmaps/beaver.xpm"

/* These variables HAVE to be global ones: I know this is bad, but... */

GtkItemFactory *MainFactory;	// item factory
GtkWidget *MainWindow;
GtkWidget *MainNotebook;
GtkWidget *MainVBox;
t_settings Settings;
GArray *FileProperties;
gint NewFilesCnt = 0;
gint OpenedFilesCnt = 0;
static gboolean MsgBarToggleDisplay;
static gboolean ToolBarToggleDisplay;
static gboolean ToggleWordwrap;

/* dialogs */

/**
 * Creates an alert with 'stock' used to create an icon and parent text of 'parent',
 * secondary text of 'secondary'.  No buttons are added.
 */
GtkWidget *alert_new (GtkWindow *parent, const gchar *stock, const gchar *primary, const gchar *secondary)
{
	GtkWidget *dialog, *hbox, *image, *label;
	gchar *buf;
	
	dialog = gtk_dialog_new_with_buttons (
		"",
		parent,
		GTK_DIALOG_MODAL | GTK_DIALOG_DESTROY_WITH_PARENT | GTK_DIALOG_NO_SEPARATOR,
		NULL);
	
	hbox = gtk_hbox_new (FALSE, 12);
	image = gtk_image_new_from_stock (stock, GTK_ICON_SIZE_DIALOG);
	label = gtk_label_new (NULL);
	
	if (secondary)
		buf = g_strdup_printf ("<span weight=\"bold\" size=\"larger\">%s</span>\n\n%s", primary, secondary);
	else
		buf = g_strdup_printf ("<span weight=\"bold\" size=\"larger\">%s</span>", primary);
	
	gtk_label_set_markup (GTK_LABEL (label), buf);
	g_free (buf);
	
	gtk_container_add (GTK_CONTAINER (GTK_DIALOG (dialog)->vbox), hbox);
	gtk_container_add (GTK_CONTAINER (hbox), image);
	gtk_container_add (GTK_CONTAINER (hbox), label);
	
	gtk_misc_set_alignment (GTK_MISC (image), 0.5, 0);
	gtk_misc_set_alignment (GTK_MISC (label), 0.5, 0);
	gtk_label_set_line_wrap (GTK_LABEL (label), TRUE);
	gtk_box_set_spacing (GTK_BOX (GTK_DIALOG (dialog)->vbox), 12);
	gtk_container_set_border_width (GTK_CONTAINER (hbox), 6);
	gtk_container_set_border_width (GTK_CONTAINER (dialog), 6);
	
	gtk_widget_realize (dialog);
	
	gtk_window_set_type_hint (GTK_WINDOW (dialog), GDK_WINDOW_TYPE_HINT_DIALOG);
	gdk_window_set_functions (dialog->window, GDK_FUNC_MOVE);
	gtk_window_set_resizable (GTK_WINDOW (dialog), FALSE);
	
	gtk_widget_show_all (hbox);
	
	return dialog;
}

/* must be at least primary */
void show_error (GtkWindow *parent, const gchar *primary, const gchar *secondary)
{
	GtkWidget *dialog;
	
	dialog = alert_new (parent, GTK_STOCK_DIALOG_ERROR,
		primary,
		secondary);
	
	gtk_dialog_add_buttons (GTK_DIALOG (dialog), GTK_STOCK_OK, 1, NULL);
	
	gtk_dialog_run (GTK_DIALOG (dialog));
	
	gtk_widget_destroy (dialog);
}




/***************************** MENUS MANAGEMENT ******************************/


/* What to do when you chose a menu item ???
   The answer is there... */

void menu_items_treatment (GtkWidget *Widget, gpointer Op_in)
{
  gint CurrentPage, i;
  gint Op = GPOINTER_TO_INT (Op_in);

  CurrentPage = gtk_notebook_get_current_page (GTK_NOTEBOOK(MainNotebook));
  switch(Op)
    {
    case NEW                  : new_file ();                   break;
    case OPEN                 : open_file ();                  break;
    case SAVE                 : save_file ();                  break;
    case SAVE_ALL             : save_all ();                   break;
    case SAVE_AS              : save_file_as ();               break;
    case SAVE_A_COPY          : save_a_copy ();                break;
    case REVERT               : revert ();                     break;
    case PRINT                : print_buffer ();               break;
    case CLOSE                : close_file ();                 break;
    case CLOSE_ALL            : close_all ();                  break;
    case QUIT                 : quit ();                       break;
    case UNDO                 :
      if (OpenedFilesCnt)
	{
	  gtk_source_buffer_undo (GTK_SOURCE_BUFFER (FPROPS (CurrentPage, Buffer)));
	  break;
	 } 
    case REDO                 :
      if (OpenedFilesCnt)
	{
	  gtk_source_buffer_redo (GTK_SOURCE_BUFFER (FPROPS (CurrentPage, Buffer)));
	  break;
	}
    case CUT                  :
      if ((OpenedFilesCnt) && !FPROPS(CurrentPage, ReadOnly))
	{
          GtkClipboard *clipboard = gtk_clipboard_get (GDK_SELECTION_CLIPBOARD);
	  gtk_text_buffer_cut_clipboard (gtk_text_view_get_buffer (
	           GTK_TEXT_VIEW(FPROPS(CurrentPage, Text))),
	           clipboard, TRUE);
	  print_msg (_("Selection cut to clipboard."));
	}
      break;
    case COPY                 :
      if (OpenedFilesCnt)
	{
          GtkClipboard *clipboard = gtk_clipboard_get (GDK_SELECTION_CLIPBOARD);
	  gtk_text_buffer_copy_clipboard (gtk_text_view_get_buffer (
	           GTK_TEXT_VIEW(FPROPS(CurrentPage, Text))),
	           clipboard);
	  print_msg (_("Selection copied to clipboard."));
	}
	break;
    case PASTE                :
      if ((OpenedFilesCnt) && !FPROPS(CurrentPage, ReadOnly))
	{
          GtkClipboard *clipboard = gtk_clipboard_get (GDK_SELECTION_CLIPBOARD);
	  gtk_text_buffer_paste_clipboard (gtk_text_view_get_buffer (
	           GTK_TEXT_VIEW(FPROPS(CurrentPage, Text))),
	           clipboard, NULL, TRUE);
	  print_msg (_("Text pasted from clipboard..."));
	}
      break;
    case CLEAR                :
      if ((OpenedFilesCnt) && !FPROPS(CurrentPage, ReadOnly))
	{
	  gtk_text_buffer_set_text (gtk_text_view_get_buffer
	    (GTK_TEXT_VIEW(FPROPS(CurrentPage, Text))), "", -1);
	  print_msg (_("Buffer cleared."));
	}
      break;
    case SELECT_ALL           :
      if (OpenedFilesCnt)
	{
	  GtkTextIter start, end;
	  GtkTextBuffer *Buffer;
	  
	  Buffer = gtk_text_view_get_buffer (
	           GTK_TEXT_VIEW(FPROPS(CurrentPage, Text)));
	  gtk_text_buffer_get_bounds (Buffer, &start, &end);
	  gtk_text_buffer_place_cursor (Buffer, &start);
	  gtk_text_buffer_move_mark_by_name (Buffer, "selection_bound", &end);
	  print_msg (_("Selected whole text."));
	}
      break;
    case COMPLETE             :
      if (OpenedFilesCnt)
	auto_completion (GTK_TEXT_VIEW(FPROPS(CurrentPage, Text)));
      break;
    case FIND                 :
      if (OpenedFilesCnt)
	search (FALSE);
      break;
    case FIND_IN_FILES        :
      search_in_files (FALSE);
      break;
    case REPLACE              :
      if (OpenedFilesCnt)
	search (TRUE);
      break;
    case REPLACE_IN_FILES     :
      search_in_files (TRUE);
      break;
    case LINE                 :
      if (OpenedFilesCnt)
	goto_line ();
      break;
    case READONLY             : toggle_readonly ();            break;
    case CONVERTER            : converter ();                  break;
    case COLOR                : color_picker ();               break;
    case INSERT_TIME          : insert_time (CurrentPage);     break;
    case TO_UPPER             :
      if ((OpenedFilesCnt) && !FPROPS(CurrentPage, ReadOnly))
	{
	  change_case (1);
	  print_msg (_("Selection converted to uppercase."));
	}
      break;
    case TO_LOWER             :
      if ((OpenedFilesCnt) && !FPROPS(CurrentPage, ReadOnly))
	{
	  change_case (2);
	  print_msg (_("Selection converted to lowercase."));
	}
      break;
    case CAPITALIZE           :
      if ((OpenedFilesCnt) && !FPROPS(CurrentPage, ReadOnly))
	{
	  change_case (3);
	  print_msg (_("Selection capitalized."));
	}
      break; 
    case INVERT_CASE          :
      if ((OpenedFilesCnt) && !FPROPS(CurrentPage, ReadOnly))
	{
	  change_case (4);
	  print_msg (_("Inverted case."));
	}
      break;
    case TO_UNIX             :
      if ((OpenedFilesCnt) && !FPROPS(CurrentPage, ReadOnly))
	{
	  convert_this_to_unix ();
	}
      break;
    case TO_DOS             :
      if ((OpenedFilesCnt) && !FPROPS(CurrentPage, ReadOnly))
	{
	  convert_this_to_dos ();
	}
      break;
    case TO_MAC             :
      if ((OpenedFilesCnt) && !FPROPS(CurrentPage, ReadOnly))
	{
	  convert_this_to_mac ();
	}
      break;
    case TO_UNIX_ALL             :
      if (OpenedFilesCnt)
	{
   	  convert_all(convert_this_to_unix);
	}
      break;
    case TO_DOS_ALL             :
      if (OpenedFilesCnt)
	{
   	  convert_all(convert_this_to_dos);
	}
      break;
    case TO_MAC_ALL             :
      if (OpenedFilesCnt)
	{
   	  convert_all(convert_this_to_mac);
	}
      break;
    case UNIX_DOS             :
      if ((OpenedFilesCnt) && !FPROPS(CurrentPage, ReadOnly))
	{
	  conv_unix_to_dos ();
	}
      break;
    case UNIX_MAC             :
      if ((OpenedFilesCnt) && !FPROPS(CurrentPage, ReadOnly))
	{
	  conv_unix_to_mac ();
	}
      break;
    case DOS_UNIX             :
      if ((OpenedFilesCnt) && !FPROPS(CurrentPage, ReadOnly))
	{
	  conv_dos_to_unix ();
	}
      break;
    case DOS_MAC              :
      if ((OpenedFilesCnt) && !FPROPS(CurrentPage, ReadOnly))
	{
	  conv_dos_to_mac ();
	}
      break;
    case MAC_DOS              :
      if ((OpenedFilesCnt) && !FPROPS(CurrentPage, ReadOnly))
	{
	  conv_mac_to_dos ();
	}
      break;
    case MAC_UNIX             :
      if ((OpenedFilesCnt) && !FPROPS(CurrentPage, ReadOnly))
	{
	  conv_mac_to_unix ();
	}
      break;
    case UNIX_DOS_ALL             :
      convert_all(conv_unix_to_dos);
      break;
    case UNIX_MAC_ALL             :
      convert_all(conv_unix_to_mac);
      break;
    case DOS_UNIX_ALL             :
      convert_all(conv_dos_to_unix);
      break;
    case DOS_MAC_ALL              :
      convert_all(conv_dos_to_mac);
      break;
    case MAC_DOS_ALL              :
      convert_all(conv_mac_to_dos);
      break;
    case MAC_UNIX_ALL             :
      convert_all(conv_mac_to_unix);
      break;
    case FILE_INFO            : file_info (CurrentPage);       break;
    case TOOLBAR              :
      if ((!ToolBarToggleDisplay) && (TOOLBAR_DISPLAY))
	ToolBarToggleDisplay = TRUE;
      else
	{
	  if (TOOLBAR_DISPLAY)
	    {
	      TOOLBAR_DISPLAY = FALSE;
	      hide_toolbar ();
	      set_bool_conf ("General/ToolBar/Display", FALSE);
	      print_msg (_("Hid toolbar."));
	    }
	  else
	    {
	      TOOLBAR_DISPLAY = TRUE;
	      show_toolbar ();
	      set_bool_conf ("General/ToolBar/Display", TRUE);
	      if (!ToolBarToggleDisplay) ToolBarToggleDisplay = TRUE;
	      print_msg (_("Unhid toolbar..."));
	    }
	}
      break;
    case MSGBAR               :
      if ((!MsgBarToggleDisplay) && (MSGBAR_DISPLAY))
	MsgBarToggleDisplay = TRUE;
      else
	{
	  if (MSGBAR_DISPLAY)
	    {
	      MSGBAR_DISPLAY = FALSE;
	      hide_msgbar ();
	      set_bool_conf ("General/MsgBar/Display", FALSE);
	    }
	  else
	    {
	      MSGBAR_DISPLAY = TRUE;
	      show_msgbar ();
	      set_bool_conf ("General/MsgBar/Display", TRUE);
	      if (!MsgBarToggleDisplay) MsgBarToggleDisplay = TRUE;
	      print_msg (_("Unhid Msg Bar..."));
	    }
	}
      break;
    case WORDWRAP        :
     if ((!ToggleWordwrap) && (TOGGLE_WORDWRAP))
	ToggleWordwrap = TRUE;
      else
	{
	  if (TOGGLE_WORDWRAP)
	    {
	      TOGGLE_WORDWRAP = FALSE;
	      for (i = 0; i < OpenedFilesCnt; i++)
		gtk_text_view_set_wrap_mode (
		            GTK_TEXT_VIEW(FPROPS(CurrentPage, Text)),
		            GTK_WRAP_NONE);
              set_bool_conf ("General/Editor/Wordwrap", FALSE);
	      print_msg (_("Disabled word-wrap."));
	    }
	  else
	    {
	      TOGGLE_WORDWRAP = TRUE;
	      for (i = 0; i < OpenedFilesCnt; i++)
		gtk_text_view_set_wrap_mode (
		            GTK_TEXT_VIEW(FPROPS(CurrentPage, Text)),
		            GTK_WRAP_WORD);
              set_bool_conf ("General/Editor/Wordwrap", TRUE);
	      if (!ToggleWordwrap) ToggleWordwrap = TRUE;
	      print_msg (_("Enabled word-wrap."));
	    }
	}
      break;
    case TAB_POS_TOP          :
      TAB_POSITION = 1;
      set_int_conf ("General/Tabs/Position", 1);
      gtk_notebook_set_tab_pos (GTK_NOTEBOOK(MainNotebook), GTK_POS_TOP);
      break;
    case TAB_POS_BOTTOM       :
      TAB_POSITION = 2;
      set_int_conf ("General/Tabs/Position", 2);
      gtk_notebook_set_tab_pos (GTK_NOTEBOOK(MainNotebook), GTK_POS_BOTTOM);
      break;
    case TAB_POS_LEFT         :
      TAB_POSITION = 3;
      set_int_conf ("General/Tabs/Position", 3);
      gtk_notebook_set_tab_pos (GTK_NOTEBOOK(MainNotebook), GTK_POS_LEFT);
      break;
    case TAB_POS_RIGHT        :
      TAB_POSITION = 4;
      set_int_conf ("General/Tabs/Position", 4);
      gtk_notebook_set_tab_pos (GTK_NOTEBOOK(MainNotebook), GTK_POS_RIGHT);  
      break;
    case SCROLLBAR_POS_LEFT   :
      SCROLLBAR_POSITION = 1;
      set_int_conf ("General/ScrollBar/Position", 1);
      for (i = 0; i < OpenedFilesCnt; i++)
	gtk_scrolled_window_set_placement (GTK_SCROLLED_WINDOW
					   (gtk_notebook_get_nth_page
					    (GTK_NOTEBOOK(MainNotebook), i)),
					   GTK_CORNER_TOP_RIGHT);
      break;
    case SCROLLBAR_POS_RIGHT  :
      SCROLLBAR_POSITION = 2;
      set_int_conf ("General/ScrollBar/Position", 2);
      for (i = 0; i < OpenedFilesCnt; i++)
	gtk_scrolled_window_set_placement (GTK_SCROLLED_WINDOW
					   (gtk_notebook_get_nth_page
					    (GTK_NOTEBOOK(MainNotebook), i)),
					   GTK_CORNER_TOP_LEFT);
      break;
    case TAB_WIDTH_1          : set_tab_width (1); break;
    case TAB_WIDTH_2          : set_tab_width (2); break;
    case TAB_WIDTH_3          : set_tab_width (3); break;
    case TAB_WIDTH_4          : set_tab_width (4); break;
    case TAB_WIDTH_8          : set_tab_width (8); break;
    case PREFS                : display_prefs (&Settings);     break;
    case HELP                 :
      print_msg (_("No help available yet."));
      break;
    case ABOUT                : about ();                      break;
    case SHOW_LINE_NUMBERS    :
    {
    	gboolean show = gtk_check_menu_item_get_active (GTK_CHECK_MENU_ITEM (
    		gtk_item_factory_get_item (MainFactory, _("/View/Line Numbers"))));
    	set_bool_conf ("General/Editor/ShowLineNumbers", show);
    	for (i = 0; i < OpenedFilesCnt; i++)
    	{
    		gtk_source_view_set_show_line_numbers (
			GTK_SOURCE_VIEW (FPROPS (i, Text)), show);
    	}
    	break;
    }
    case SHOW_MARGIN          :
    {
    	gboolean show = gtk_check_menu_item_get_active (GTK_CHECK_MENU_ITEM (
    		gtk_item_factory_get_item (MainFactory, _("/View/Margin"))));
    	set_bool_conf ("General/Editor/ShowMargin", show);
    	for (i = 0; i < OpenedFilesCnt; i++)
    	{
    		gtk_source_view_set_show_margin (
			GTK_SOURCE_VIEW (FPROPS (i, Text)), show);
    	}
    	break;
    }
    }
  (void)Widget; /* avoid the "unused parameter" warning */
}


/* Menu entries */

static GtkItemFactoryEntry MenuEntries[] = {
  {N_("/_File"), NULL, NULL, 0, "<Branch>"},
  {N_("/File/_New"), "<control>N", menu_items_treatment, NEW, "<StockItem>", GTK_STOCK_NEW},
  {N_("/File/_Open..."), "<control>O", menu_items_treatment, OPEN, "<StockItem>", GTK_STOCK_OPEN},
  {N_("/File/sep"), NULL, NULL, 0, "<Separator>"},
  {N_("/File/_Save"), "<control>S", menu_items_treatment, SAVE, "<StockItem>", GTK_STOCK_SAVE},
  {N_("/File/Save _As..."), "<shift><control>S", menu_items_treatment, SAVE_AS, "<StockItem>", GTK_STOCK_SAVE_AS},
  {N_("/File/Sa_ve a Copy..."), NULL, menu_items_treatment, SAVE_A_COPY, "<Item>"},
  {N_("/File/_Revert"), NULL, menu_items_treatment, REVERT, "<StockItem>", GTK_STOCK_REVERT_TO_SAVED},
  {N_("/File/sep"), NULL, NULL, 0, "<Separator>"},
  {N_("/File/_Print..."), "<control>P", menu_items_treatment, PRINT, "<StockItem>", GTK_STOCK_PRINT},
  {N_("/File/sep"), NULL, NULL, 0, "<Separator>"},
  {N_("/File/Proper_ties"), NULL, menu_items_treatment, FILE_INFO, "<StockItem>", GTK_STOCK_PROPERTIES},
  {N_("/File/sep"), NULL, NULL, 0, "<Separator>"},
  {N_("/File/sep_end"), NULL, NULL, 0, "<Separator>"},
  {N_("/File/_Close"), "<control>W", menu_items_treatment, CLOSE, "<StockItem>", GTK_STOCK_CLOSE},
  {N_("/File/_Quit"), "<control>Q", menu_items_treatment, QUIT, "<StockItem>", GTK_STOCK_QUIT},
  {N_("/_Edit"), NULL, NULL, 0, "<Branch>"},
  {N_("/Edit/_Undo"), "<control>Z", menu_items_treatment, UNDO, "<StockItem>", GTK_STOCK_UNDO},
  {N_("/Edit/_Redo"), "<shift><control>Z", menu_items_treatment, REDO, "<StockItem>", GTK_STOCK_REDO},
  {N_("/Edit/sep"), NULL, NULL, 0, "<Separator>"},
  {N_("/Edit/Cu_t"), "<control>X", menu_items_treatment, CUT, "<StockItem>", GTK_STOCK_CUT},
  {N_("/Edit/_Copy"), "<control>C", menu_items_treatment, COPY, "<StockItem>", GTK_STOCK_COPY},
  {N_("/Edit/_Paste"), "<control>V", menu_items_treatment, PASTE, "<StockItem>", GTK_STOCK_PASTE},
  {N_("/Edit/Select _All"), "<control>A", menu_items_treatment, SELECT_ALL, "<Item>"},
/* Removed because auto-completion is totally borked right now (.
  {N_("/Edit/Co_mplete"), "<control>space", menu_items_treatment, COMPLETE, "<Item>"},
*/
  {N_("/Edit/sep"), NULL, NULL, 0, "<Separator>"},
  {N_("/Edit/Toggle Rea_donly"), NULL, menu_items_treatment, READONLY, "<Item>"},
  {N_("/Edit/sep"), NULL, NULL, 0, "<Separator>"},
  {N_("/Edit/_Search..."), "<control>F", menu_items_treatment, FIND, "<StockItem>", GTK_STOCK_FIND},
  {N_("/Edit/Searc_h in Files..."), "<shift><control>F", menu_items_treatment, FIND_IN_FILES, "<Item>"},
  {N_("/Edit/R_eplace..."), "<control>H", menu_items_treatment, REPLACE, "<StockItem>", GTK_STOCK_FIND_AND_REPLACE},
  {N_("/Edit/Rep_lace in Files..."), "<shift><control>H", menu_items_treatment, REPLACE_IN_FILES, "<Item>"},
  {N_("/Edit/sep"), NULL, NULL, 0, "<Separator>"},
  {N_("/Edit/_Go to Line..."), "<control>L", menu_items_treatment, LINE, "<StockItem>", GTK_STOCK_JUMP_TO}, 
  {N_("/Edit/sep"), NULL, NULL, 0, "<Separator>"},
  {N_("/Edit/Prefere_nces"), NULL, menu_items_treatment, PREFS, "<StockItem>", GTK_STOCK_PREFERENCES},
  {N_("/_View"), NULL, NULL, 0, "<Branch>"},
  {N_("/View/_Toolbar"), NULL, menu_items_treatment, TOOLBAR, "<CheckItem>"},
  {N_("/View/_Statusbar"), NULL, menu_items_treatment, MSGBAR, "<CheckItem>"},
  {N_("/View/sep"), NULL, NULL, 0, "<Separator>"},
  {N_("/View/_Line Numbers"), NULL, menu_items_treatment, SHOW_LINE_NUMBERS, "<CheckItem>"},
  {N_("/View/_Margin"), NULL, menu_items_treatment, SHOW_MARGIN, "<CheckItem>"},
  {N_("/View/_Word-wrap"), NULL, menu_items_treatment, WORDWRAP, "<CheckItem>"},
  {N_("/View/sep"), NULL, NULL, 0, "<Separator>"},
  {N_("/View/Tab Wi_dth"), NULL, NULL, 0, "<Branch>"},
  {N_("/View/Tab Width/_1"), NULL, menu_items_treatment, TAB_WIDTH_1, "<RadioItem>"},
  {N_("/View/Tab Width/_2"), NULL, menu_items_treatment, TAB_WIDTH_2, N_("/View/Tab Width/1")},
  {N_("/View/Tab Width/_3"), NULL, menu_items_treatment, TAB_WIDTH_3, N_("/View/Tab Width/1")},
  {N_("/View/Tab Width/_4"), NULL, menu_items_treatment, TAB_WIDTH_4, N_("/View/Tab Width/1")},
  {N_("/View/Tab Width/_8"), NULL, menu_items_treatment, TAB_WIDTH_8, N_("/View/Tab Width/1")},
  {N_("/View/Ta_b Position"), NULL, NULL, 0, "<Branch>"},
  {N_("/View/Tab Position/_Top"), NULL, menu_items_treatment,
   TAB_POS_TOP, "<RadioItem>"},
  {N_("/View/Tab Position/_Bottom"), NULL, menu_items_treatment,
   TAB_POS_BOTTOM, N_("/View/Tab Position/Top")},
  {N_("/View/Tab Position/_Left"), NULL, menu_items_treatment,
   TAB_POS_LEFT, N_("/View/Tab Position/Top")},
  {N_("/View/Tab Position/_Right"), NULL, menu_items_treatment,
   TAB_POS_RIGHT, N_("/View/Tab Position/Top")},
  {N_("/View/Sc_rollbar Position"), NULL, NULL, 0, "<Branch>"},
  {N_("/View/Scrollbar Position/_Left"), NULL, menu_items_treatment,
   SCROLLBAR_POS_LEFT, "<RadioItem>"},
  {N_("/View/Scrollbar Position/_Right"), NULL, menu_items_treatment,
   SCROLLBAR_POS_RIGHT, N_("/View/Scrollbar Position/Left")},
  {N_("/_Tools"), NULL, NULL, 0, "<Branch>"},
  {N_("/Tools/_Format"), NULL, NULL, 0, "<Branch>"},
  {N_("/Tools/Format/Convert This to _DOS"), NULL, menu_items_treatment,
   TO_DOS, "<StockItem>", GTK_STOCK_CONVERT},
  {N_("/Tools/Format/Convert This to _MAC"), NULL, menu_items_treatment,
   TO_MAC, "<StockItem>", GTK_STOCK_CONVERT},
  {N_("/Tools/Format/Convert This to _UNIX"), NULL, menu_items_treatment,
   TO_UNIX, "<StockItem>", GTK_STOCK_CONVERT},
  {N_("/Tools/Format/sep"), NULL, NULL, 0, "<Separator>"},
  {N_("/Tools/Format/Convert All to DOS"), NULL, menu_items_treatment,
   TO_DOS_ALL, "<StockItem>", GTK_STOCK_CONVERT},
  {N_("/Tools/Format/Convert All to MAC"), NULL, menu_items_treatment,
   TO_MAC_ALL, "<StockItem>", GTK_STOCK_CONVERT},
  {N_("/Tools/Format/Convert All to UNIX"), NULL, menu_items_treatment,
   TO_UNIX_ALL, "<StockItem>", GTK_STOCK_CONVERT},
  {N_("/Tools/sep"), NULL, NULL, 0, "<Separator>"},
  {N_("/Tools/_Base Converter"), NULL, menu_items_treatment, CONVERTER, "<StockItem>", GTK_STOCK_CONVERT},
  {N_("/Tools/_Color Picker"), NULL, menu_items_treatment, COLOR, "<StockItem>", GTK_STOCK_SELECT_COLOR},
  {N_("/Tools/Insert _Time"), NULL, menu_items_treatment, INSERT_TIME, "<Item>"},
  {N_("/Tools/sep"), NULL, NULL, 0, "<Separator>"},
  {N_("/Tools/To _Upper Case"), NULL, menu_items_treatment, TO_UPPER, "<Item>"},
  {N_("/Tools/To _Lower Case"), NULL, menu_items_treatment, TO_LOWER, "<Item>"},
  {N_("/Tools/Ca_pitalize"), NULL, menu_items_treatment, CAPITALIZE, "<Item>"},
  {N_("/Tools/In_vert Case"), NULL, menu_items_treatment, INVERT_CASE, "<Item>"},
  {N_("/_Language"), NULL, NULL, 0, "<Branch>"},
  {N_("/_Documents"), NULL, NULL, 0, "<Branch>"},
  {N_("/Documents/_Save All"), "", menu_items_treatment, SAVE_ALL, "<StockItem>", GTK_STOCK_SAVE},
  {N_("/Documents/_Close All"), "", menu_items_treatment, CLOSE_ALL, "<StockItem>", GTK_STOCK_CLOSE},
  {N_("/_Help"), NULL, NULL, 0, "<Branch>"},
  {N_("/Help/_Contents"), "F1", menu_items_treatment, HELP, "<StockItem>", GTK_STOCK_HELP},
  {N_("/Help/_About"), NULL, menu_items_treatment, ABOUT, "<Item>"},
};


/* Return the menubar widget based on 'MenuEntries' */

GtkWidget *menubar_new (GtkWidget *Window)
{
  GtkWidget *Menu;
  GtkAccelGroup *AccelGroup;
  gint i;
  const gchar *str;

  MsgBarToggleDisplay = FALSE;
  ToolBarToggleDisplay = FALSE;
  ToggleWordwrap = FALSE;
  AccelGroup = gtk_accel_group_new();
  MainFactory = gtk_item_factory_new (GTK_TYPE_MENU_BAR, "<main>", AccelGroup);
  
  for (i = 0; i < G_N_ELEMENTS (MenuEntries); i++)
  {
  	MenuEntries[i].path = _(MenuEntries[i].path);
  	/* we also have to convert the type because radio buttons use
  	   references to other paths */
  	MenuEntries[i].item_type = _(MenuEntries[i].item_type);
  }
  
  gtk_item_factory_create_items (MainFactory, G_N_ELEMENTS (MenuEntries),
				 MenuEntries, NULL);
  gtk_window_add_accel_group (GTK_WINDOW (Window), AccelGroup);
  Menu = gtk_tearoff_menu_item_new ();
  gtk_menu_shell_prepend (GTK_MENU_SHELL (gtk_item_factory_get_widget
			     (MainFactory, _("/File"))), Menu);
  Menu = gtk_tearoff_menu_item_new ();
  gtk_menu_shell_prepend (GTK_MENU_SHELL (gtk_item_factory_get_widget
			     (MainFactory, _("/Edit"))), Menu);
  Menu = gtk_tearoff_menu_item_new ();
  gtk_menu_shell_prepend (GTK_MENU_SHELL (gtk_item_factory_get_widget
			     (MainFactory, _("/View"))), Menu);
  Menu = gtk_tearoff_menu_item_new ();
  gtk_menu_shell_prepend (GTK_MENU_SHELL (gtk_item_factory_get_widget
			     (MainFactory, _("/Tools"))), Menu);
  Menu = gtk_tearoff_menu_item_new ();
  gtk_menu_shell_prepend (GTK_MENU_SHELL (gtk_item_factory_get_widget
			     (MainFactory, _("/Language"))), Menu);
  Menu = gtk_tearoff_menu_item_new ();
  gtk_menu_shell_prepend (GTK_MENU_SHELL (gtk_item_factory_get_widget
			     (MainFactory, _("/Documents"))), Menu);
  Menu = gtk_tearoff_menu_item_new (); 
  gtk_menu_shell_prepend (GTK_MENU_SHELL (gtk_item_factory_get_widget
			     (MainFactory, _("/Help"))), Menu);
  gtk_check_menu_item_set_active (GTK_CHECK_MENU_ITEM
				    (gtk_item_factory_get_widget
				     (MainFactory,
				      _("/View/Statusbar"))),
				    Settings.msgbar_display);
  gtk_check_menu_item_set_active (GTK_CHECK_MENU_ITEM
				    (gtk_item_factory_get_widget
				     (MainFactory,
				      _("/View/Toolbar"))),
				    Settings.toolbar_display);
  gtk_check_menu_item_set_active (GTK_CHECK_MENU_ITEM
				    (gtk_item_factory_get_widget
				     (MainFactory,
				      _("/View/Word-wrap"))),
				    Settings.toggle_wordwrap);
  gtk_check_menu_item_set_active (GTK_CHECK_MENU_ITEM
				    (gtk_item_factory_get_widget
				     (MainFactory,
				      _("/View/Line Numbers"))),
				    Settings.show_line_numbers);
  gtk_check_menu_item_set_active (GTK_CHECK_MENU_ITEM
				    (gtk_item_factory_get_widget
				     (MainFactory,
				      _("/View/Margin"))),
				    Settings.show_margin);
  switch (Settings.tab_width)
  {
  case 1: str = _("/View/Tab Width/1"); break;
  case 2: str = _("/View/Tab Width/2"); break;
  case 3: str = _("/View/Tab Width/3"); break;
  case 4: str = _("/View/Tab Width/4"); break;
  default:
  case 8: str = _("/View/Tab Width/8"); break;
  }
  gtk_check_menu_item_set_active (GTK_CHECK_MENU_ITEM
				    (gtk_item_factory_get_widget
				     (MainFactory,
				      str)),
				    TRUE);
				    
  switch (Settings.tab_position)
  {
  default:
  case 1: str = _("/View/Tab Position/Top"); break;
  case 2: str = _("/View/Tab Position/Bottom"); break;
  case 3: str = _("/View/Tab Position/Left"); break;
  case 4: str = _("/View/Tab Position/Right"); break;
  }
  gtk_check_menu_item_set_active (GTK_CHECK_MENU_ITEM
				    (gtk_item_factory_get_widget
				     (MainFactory,
				      str)),
				    TRUE);

  switch (Settings.scrollbar_position)
  {
  case 1: str = _("/View/Scrollbar Position/Left"); break;
  default:
  case 2: str = _("/View/Scrollbar Position/Right"); break;
  }
  gtk_check_menu_item_set_active (GTK_CHECK_MENU_ITEM
				    (gtk_item_factory_get_widget
				     (MainFactory,
				      str)),
				    TRUE);

  Menu = gtk_item_factory_get_widget (MainFactory, "<main>");
  
  init_recent_files ();
  init_languages_menu ();
  return (Menu);
}


/* PopUp Menu Entries */

static GtkItemFactoryEntry PopUpMenuEntries[] = {
  {N_("/Cu_t"), "<control>X", menu_items_treatment, CUT, "<StockItem>", GTK_STOCK_CUT},
  {N_("/_Copy"), "<control>C", menu_items_treatment, COPY, "<StockItem>", GTK_STOCK_COPY},
  {N_("/_Paste"), "<control>V", menu_items_treatment, PASTE, "<StockItem>", GTK_STOCK_PASTE},
  {N_("/sep"), NULL, NULL, 0, "<Separator>"},
  {N_("/_Save"), "<control>S", menu_items_treatment, SAVE, "<StockItem>", GTK_STOCK_SAVE},
  {N_("/Save _As..."), "<shift><control>S", menu_items_treatment, SAVE_AS, "<StockItem>", GTK_STOCK_SAVE_AS},
  {N_("/sep"), NULL, NULL, 0, "<Separator>"},
  {N_("/C_lose"), "<control>W", menu_items_treatment, CLOSE, "<StockItem>", GTK_STOCK_CLOSE},
};


/* Return the popup menu widget based on 'PopUpMenuEntries' */

GtkWidget *popup_menu_new (void)
{
  GtkItemFactory *ItemFactory;
  GtkWidget *PopUpMenu;
  gint i;

  for (i = 0; i < G_N_ELEMENTS (PopUpMenuEntries); i++)
  {
  	PopUpMenuEntries[i].path = _(PopUpMenuEntries[i].path);
  }
  
  ItemFactory = gtk_item_factory_new (GTK_TYPE_MENU, "<popup_main>", NULL);
  gtk_item_factory_create_items (ItemFactory, G_N_ELEMENTS (PopUpMenuEntries),
				 PopUpMenuEntries, NULL);
  PopUpMenu = gtk_item_factory_get_widget (ItemFactory, "<popup_main>");
  return (PopUpMenu);
}


/* Display the popup menu widget based on 'PopUpMenuEntries' */

gboolean popup_menu_show (GtkMenu *PopUpMenu, GdkEventButton *Event)
{
  if (Event -> type == GDK_BUTTON_PRESS)
    if (Event -> button == 3)
      {
	gtk_menu_popup(PopUpMenu, NULL, NULL, NULL, NULL,
		       Event-> button, Event -> time);
	return (TRUE);
      }
      
  return (FALSE);
}


/**************************** NOTEBOOK MANAGEMENT ****************************/


/* Add a Page in a Notebook */

void add_page_in_notebook (GtkNotebook *Notebook, const gchar *FileName)
{
  GtkWidget *VScrolledWindow;
  GtkWidget *TabLabel, *MenuLabel;
  GtkWidget *PopUpMenu;
  t_fprops Storage;
  PangoFontDescription *desc;
  GdkColor color;

    
  START_FCN
  
  g_array_append_val (FileProperties, Storage);
  if (!FileName)
    {
      gchar *Untitled;
      
      Untitled = g_strdup_printf (UNTITLED " %d", ++NewFilesCnt);
      init_file_properties (Untitled, OpenedFilesCnt);
      g_free (Untitled);
    }
  else init_file_properties (FileName, OpenedFilesCnt);
  VScrolledWindow = gtk_scrolled_window_new (NULL, NULL);
  gtk_scrolled_window_set_policy (GTK_SCROLLED_WINDOW(VScrolledWindow),
				  GTK_POLICY_AUTOMATIC,
				  GTK_POLICY_ALWAYS);
  gtk_scrolled_window_set_shadow_type (GTK_SCROLLED_WINDOW(VScrolledWindow),
                                  GTK_SHADOW_IN);				  
				  
  if (SCROLLBAR_POSITION == 1)
    gtk_scrolled_window_set_placement (GTK_SCROLLED_WINDOW(VScrolledWindow),
				       GTK_CORNER_TOP_RIGHT);
  if (SCROLLBAR_POSITION == 2)
    gtk_scrolled_window_set_placement (GTK_SCROLLED_WINDOW(VScrolledWindow),
				       GTK_CORNER_TOP_LEFT);
  FPROPS(OpenedFilesCnt, Text) = gtk_source_view_new ();
  FPROPS(OpenedFilesCnt, Buffer) = gtk_text_view_get_buffer 
			(GTK_TEXT_VIEW(FPROPS(OpenedFilesCnt, Text)));
  desc = pango_font_description_from_string (FONT);
  gtk_widget_modify_font (FPROPS(OpenedFilesCnt, Text), desc);
  g_free (desc);
  color.red = SELECTED_BG_RED; color.green = SELECTED_BG_GREEN; color.blue = SELECTED_BG_BLUE;
  gtk_widget_modify_base (FPROPS(OpenedFilesCnt, Text), GTK_STATE_SELECTED, &color);
  color.red = SELECTED_FG_RED; color.green = SELECTED_FG_GREEN; color.blue = SELECTED_FG_BLUE;
  gtk_widget_modify_text (FPROPS(OpenedFilesCnt, Text), GTK_STATE_SELECTED, &color);
  color.red = BG_RED; color.green = BG_GREEN; color.blue = BG_BLUE;
  gtk_widget_modify_base (FPROPS(OpenedFilesCnt, Text), GTK_STATE_NORMAL, &color);
  color.red = FG_RED; color.green = FG_GREEN; color.blue = FG_BLUE;
  gtk_widget_modify_text (FPROPS(OpenedFilesCnt, Text), GTK_STATE_NORMAL, &color);
  if (!FPROPS(OpenedFilesCnt, ReadOnly))
    gtk_text_view_set_editable (GTK_TEXT_VIEW(FPROPS(OpenedFilesCnt, Text)), TRUE);
  if (TOGGLE_WORDWRAP)
    gtk_text_view_set_wrap_mode (GTK_TEXT_VIEW(FPROPS(OpenedFilesCnt, Text)),
			    GTK_WRAP_WORD);
			    
  gtk_text_view_set_left_margin(GTK_TEXT_VIEW(FPROPS(OpenedFilesCnt, Text)), 5); /* FIXME: make this adjustable */

  
   PopUpMenu = popup_menu_new();
  g_signal_connect_swapped (G_OBJECT(FPROPS(OpenedFilesCnt, Text)),
			     "button-press-event", G_CALLBACK (popup_menu_show),
			     PopUpMenu);
  gtk_container_add (GTK_CONTAINER(VScrolledWindow),
		     FPROPS(OpenedFilesCnt, Text));
  TabLabel = gtk_label_new ("");
  MenuLabel = gtk_label_new ("");
  gtk_notebook_append_page_menu (Notebook, VScrolledWindow,
				 TabLabel, MenuLabel);
  gtk_widget_show_all (GTK_WIDGET(Notebook));
  gtk_notebook_set_current_page (GTK_NOTEBOOK (MainNotebook), OpenedFilesCnt);
  set_label (Notebook, OpenedFilesCnt);
  
  OpenedFilesCnt++;
  
  menu_manage_documents ();
  menu_set_files_open (TRUE);
  
  END_FCN
}


void set_tab_width (gint width)
{
	gint i;
	
	for (i = 0; i < OpenedFilesCnt; i++)
	{
		gtk_source_view_set_tabs_width (
			GTK_SOURCE_VIEW (FPROPS (i, Text)), width);
	}
	
	Settings.tab_width = width;
	set_int_conf ("General/Editor/TabWidth", width);
}


/* Set the current page Tab & Menu Labels */

void  set_label (GtkNotebook *Notebook, gint CurrentPage)
{
  START_FCN

  if (gtk_text_buffer_get_modified (FPROPS (CurrentPage, Buffer)))
    {
      if (!FPROPS(CurrentPage, ReadOnly))
	{
	  gchar *Title;
	  
	  Title = g_strconcat ("*", FPROPS(CurrentPage, BaseName), NULL);
	  gtk_notebook_set_tab_label_text (Notebook, gtk_notebook_get_nth_page
					   (Notebook, CurrentPage), Title);
	  gtk_notebook_set_menu_label_text (Notebook, gtk_notebook_get_nth_page
					    (Notebook, CurrentPage), Title);
	  g_free (Title);
	}      
      else
	{
	  gchar *Title;
	  
	  Title = g_strconcat ("RO *", FPROPS(CurrentPage, BaseName), NULL);
	  gtk_notebook_set_tab_label_text (Notebook, gtk_notebook_get_nth_page
					   (Notebook, CurrentPage), Title);
	  gtk_notebook_set_menu_label_text (Notebook, gtk_notebook_get_nth_page
					    (Notebook, CurrentPage), Title);
	  g_free (Title);
	}
    }
  else
    {
      if (FPROPS(CurrentPage, ReadOnly))
	{
	  gchar *Title;
	  
	  Title = g_strconcat ("RO ", FPROPS(CurrentPage, BaseName), NULL);
	  gtk_notebook_set_tab_label_text (Notebook, gtk_notebook_get_nth_page
					   (Notebook, CurrentPage), Title);
	  gtk_notebook_set_menu_label_text (Notebook, gtk_notebook_get_nth_page
					    (Notebook, CurrentPage), Title);
	  g_free (Title);
	}
      else
	{
	  gtk_notebook_set_tab_label_text (Notebook, gtk_notebook_get_nth_page
					   (Notebook, CurrentPage), 
					   FPROPS(CurrentPage, BaseName));
	  gtk_notebook_set_menu_label_text (Notebook, gtk_notebook_get_nth_page
					    (Notebook, CurrentPage),
					    FPROPS(CurrentPage, BaseName));
	}
    }
    END_FCN
}


/* Set the Window title (used when you change the active page with your
   little mouse...) */

void set_title (gint CurrentPage)
{
  START_FCN
  if (CurrentPage == -1)
    gtk_window_set_title (GTK_WINDOW(MainWindow), PACKAGE_NAME);
  else gtk_window_set_title (GTK_WINDOW(MainWindow), 
		FPROPS(CurrentPage, BaseName));
  END_FCN
}


/***************************** INTERFACE DISPLAY *****************************/


void page_changed (GtkNotebook *notebook, GtkNotebookPage *page_widget,
	guint page, gpointer data)
{
	START_FCN
	set_title (page);
	note_redoundo (page);
	note_saveable ();
	note_format ();
	END_FCN
}

void note_redoundo (gint page)
{
	gboolean redo, undo;
	
	START_FCN
	
	redo = gtk_source_buffer_can_redo (
		GTK_SOURCE_BUFFER (FPROPS (page, Buffer)));
	undo = gtk_source_buffer_can_undo (
		GTK_SOURCE_BUFFER (FPROPS (page, Buffer)));
	
	menu_set_sensitive ("/Edit/Undo", undo);
	menu_set_sensitive ("/Edit/Redo", redo);
	toolbar_set_sensitive (UNDO, undo);
	toolbar_set_sensitive (REDO, redo);
	
	END_FCN
}

void note_saveable (void)
{
	gboolean saveable;
	gint page;
	
	START_FCN
	
	page = gtk_notebook_get_current_page (GTK_NOTEBOOK (MainNotebook));
	if (page == -1)
		saveable = FALSE;
	else
		saveable = gtk_text_buffer_get_modified (FPROPS(page, Buffer));
	
	menu_set_sensitive ("/File/Save", saveable);
	toolbar_set_sensitive (SAVE, saveable);
	
        menu_set_sensitive ("/File/Revert", saveable);
	
	END_FCN
}

void note_format (void)
{
	gint page;
	
	START_FCN
	
	page = gtk_notebook_get_current_page (GTK_NOTEBOOK (MainNotebook));
	
	switch (FPROPS (page, Format))
	{
	case UNIX:
		menu_set_sensitive ("/Tools/Format/Convert This to DOS", TRUE);
		menu_set_sensitive ("/Tools/Format/Convert This to MAC", TRUE);
		menu_set_sensitive ("/Tools/Format/Convert This to UNIX", FALSE);
		break;
	case MAC:
		menu_set_sensitive ("/Tools/Format/Convert This to DOS", TRUE);
		menu_set_sensitive ("/Tools/Format/Convert This to MAC", FALSE);
		menu_set_sensitive ("/Tools/Format/Convert This to UNIX", TRUE);
		break;
	case DOS:
		menu_set_sensitive ("/Tools/Format/Convert This to DOS", FALSE);
		menu_set_sensitive ("/Tools/Format/Convert This to MAC", TRUE);
		menu_set_sensitive ("/Tools/Format/Convert This to UNIX", TRUE);
		break;
	}
	
	END_FCN
}

void switch_to_page (gint page)
{
	gtk_notebook_set_current_page (GTK_NOTEBOOK (MainNotebook), page);
}

void switch_to_page_callback (GtkWidget *dummy, gint page)
{
	START_FCN
	switch_to_page (page - 
		(RECENT_FILES_OFFSET + MAX_POSSIBLE_RECENT_FILES + 1));
	END_FCN
}

void menu_manage_documents (void)
{
	GtkItemFactoryEntry sep  = {
		"/Documents/sep", NULL,
		NULL, 0, "<Separator>"};
	GtkWidget *item;
	gint i;
	START_FCN
	
	/* remove all current items */
	gtk_item_factory_delete_entry (MainFactory, &sep);
	i = RECENT_FILES_OFFSET + MAX_POSSIBLE_RECENT_FILES;
	while ((item = gtk_item_factory_get_item_by_action (MainFactory, ++i)))
	{
		const gchar *path = gtk_item_factory_path_from_widget (item);
		gtk_item_factory_delete_item (MainFactory, path);
	}
	
	if (OpenedFilesCnt)
	{
		/* add new documents */
		gtk_item_factory_create_items
			(MainFactory, 1, &sep, NULL);
		
		for (i = 1; i <= OpenedFilesCnt; i++)
		{
			gchar *base = g_strdup (FPROPS(i - 1, BaseName));
			gchar *name = str_replace_tokens (&base,
				'_', "__");
			gchar *menu_title = g_strdup_printf (
				"/Documents/_%i %s", i, name);
			GtkItemFactoryEntry NewEntry  = {
				menu_title, NULL,
				(GtkItemFactoryCallback) switch_to_page_callback,
				RECENT_FILES_OFFSET + 
				MAX_POSSIBLE_RECENT_FILES + i, "<Item>"};
			gtk_item_factory_create_items
				(MainFactory, 1, &NewEntry, NULL);
		}
	}
	else
	{
		menu_set_sensitive ("/Documents/Save All", FALSE);
		menu_set_sensitive ("/Documents/Close All", FALSE);
	}
	
	END_FCN
}

void menu_set_sensitive (const gchar *id, gboolean setting)
{
	GtkWidget *widget;
	
	widget = gtk_item_factory_get_widget (MainFactory, id);
	if (widget)
		gtk_widget_set_sensitive (widget, setting);
}


void menu_set_files_open (gboolean opened)
{
	GList *children, *tmp;

      menu_set_sensitive ("/File/Save As...", opened);
      menu_set_sensitive ("/File/Save a Copy...", opened);
      menu_set_sensitive ("/File/Print...", opened);
      menu_set_sensitive ("/File/Properties", opened);
      menu_set_sensitive ("/File/Close", opened);
      menu_set_sensitive ("/Edit/Select All", opened);
      menu_set_sensitive ("/Edit/Complete", opened);
      menu_set_sensitive ("/Edit/Toggle Readonly", opened);
      menu_set_sensitive ("/Edit/Search...", opened);
      menu_set_sensitive ("/Edit/Replace...", opened);
      menu_set_sensitive ("/Edit/Go to Line...", opened);
      menu_set_sensitive ("/Tools/Format/Convert This to DOS", opened);
      menu_set_sensitive ("/Tools/Format/Convert This to MAC", opened);
      menu_set_sensitive ("/Tools/Format/Convert This to UNIX", opened);
      menu_set_sensitive ("/Tools/Format/Convert All to DOS", opened);
      menu_set_sensitive ("/Tools/Format/Convert All to MAC", opened);
      menu_set_sensitive ("/Tools/Format/Convert All to UNIX", opened);
      menu_set_sensitive ("/Documents/Save All", opened);
      menu_set_sensitive ("/Documents/Close All", opened);
      menu_set_sensitive ("/Edit/Copy", opened);
      toolbar_set_sensitive (COPY, opened);
      menu_set_sensitive ("/Edit/Cut", opened);
      toolbar_set_sensitive (CUT, opened);
      menu_set_sensitive ("/Edit/Paste", opened);
      toolbar_set_sensitive (PASTE, opened);
      
      if (opened == FALSE)
      {
      	menu_set_sensitive ("/File/Save", opened);
      	toolbar_set_sensitive (SAVE, opened);
      	menu_set_sensitive ("/Edit/Undo", opened);
      	toolbar_set_sensitive (UNDO, opened);
      	menu_set_sensitive ("/Edit/Redo", opened);
      	toolbar_set_sensitive (REDO, opened);
        menu_set_sensitive ("/File/Revert", opened);
      }
      
      toolbar_set_sensitive (CLOSE, opened);
      toolbar_set_sensitive (FIND, opened);
      toolbar_set_sensitive (REPLACE, opened);
      
      children = gtk_container_get_children (GTK_CONTAINER (
		gtk_item_factory_get_widget (MainFactory, _("/Language"))));
	for (tmp = children; tmp; tmp = tmp->next)
	{
		gtk_widget_set_sensitive (tmp->data, opened);
	}
	g_list_free (children);
}


/* Parse the command line and load files 'beaver [FILE1] [FILE2] ...' */
/* returns the number of files opened */
gint command_line (gint argc, gchar *argv[])
{
  gint count = 0;
  START_FCN
  
  if (argc >= 2)
    {
      gint i;
      gboolean FirstTime = TRUE;
      
      for (i = 1; i < argc; i++)
	{
	  gchar *FileName;
          
	  if (!strcmp (str_get_last_part (argv[i], G_DIR_SEPARATOR, TRUE), ""))
	    continue;
	  if ((!strcmp (argv[i], "--")) && (FirstTime == TRUE))
	    {
	      FirstTime = FALSE;
	      continue;
	    }
	  FileName = get_absolute_path (argv[i]);
	  
	  open_filename (FileName);
	  
	  count++;
	  g_free (FileName);
	}
    }
    END_FCN
    return count;
}

void set_beaver_icon (GtkWindow *window)
{
	GdkPixbuf *icon;
	
	icon = gdk_pixbuf_new_from_xpm_data (beaver_xpm);
	gtk_window_set_icon (window, icon);
	
	g_object_unref (icon);
}


/* The main function in the shaping process of the interface */  

void interface (gint argc, gchar *argv[])
{
  GtkWidget *MenuBar;
  
#ifdef DEBUG_FCN
  g_print(__FILE__": %s(): Begin\n", __func__);
#endif
  gtk_set_locale();  
  gtk_init (&argc, &argv);
  Settings = init_settings ();
  MainWindow = gtk_window_new (GTK_WINDOW_TOPLEVEL);
  gtk_window_set_resizable (GTK_WINDOW(MainWindow), TRUE);
  gtk_window_set_default_size (GTK_WINDOW (MainWindow), MAIN_WINDOW_WIDTH, MAIN_WINDOW_HEIGHT);
  g_signal_connect (G_OBJECT(MainWindow), "delete-event",
		      G_CALLBACK (quit), NULL); 
  FileProperties = g_array_new (TRUE, FALSE, sizeof(t_fprops));
  MainVBox = gtk_vbox_new (FALSE, 0);
  gtk_container_add (GTK_CONTAINER(MainWindow), MainVBox);
  MainNotebook = gtk_notebook_new ();
  editor_init();
  
  MenuBar = menubar_new (MainWindow);
  gtk_box_pack_start (GTK_BOX(MainVBox), MenuBar, FALSE, FALSE, 0);
  
  init_toolbar (GTK_BOX(MainVBox));
  
  gtk_notebook_popup_enable (GTK_NOTEBOOK(MainNotebook));
  gtk_notebook_set_scrollable (GTK_NOTEBOOK(MainNotebook), TRUE);
  gtk_box_pack_start (GTK_BOX(MainVBox), MainNotebook, TRUE, TRUE, 0);
  g_signal_connect_after (G_OBJECT(MainNotebook), "switch-page",
		      G_CALLBACK (page_changed), NULL);
  init_msgbar (GTK_BOX(MainVBox));
  set_beaver_icon (GTK_WINDOW (MainWindow));
  menu_set_files_open (FALSE);
  command_line (argc, argv);
  autosave (AUTOSAVE_DELAY);
  set_title (gtk_notebook_get_current_page(GTK_NOTEBOOK(MainNotebook)));
  gtk_widget_show_all (MainWindow);
  if (!MSGBAR_DISPLAY) hide_msgbar ();
  if (!TOOLBAR_DISPLAY) hide_toolbar ();
  gtk_main ();
#ifdef DEBUG_FCN
  g_print(__FILE__": %s(): End\n", __func__);
#endif
}


/* Display a nice little 'About...' Dialog box :) */

void about (void)
{
  GtkWidget *AboutWindow;
  GtkWidget *PixmapWidget;
  GtkWidget *HSeparator;
  GdkPixmap *Pixmap;
  GdkBitmap *Mask;
  GtkStyle  *Style;
  GtkWidget *Button;
  GtkWidget *AboutLabel;  
  
  AboutWindow = gtk_dialog_new ();
  gtk_window_set_title (GTK_WINDOW(AboutWindow), "About " PACKAGE_NAME); 
  gtk_window_set_resizable (GTK_WINDOW(AboutWindow), FALSE);
  gtk_window_set_position (GTK_WINDOW(AboutWindow), GTK_WIN_POS_CENTER);
  gtk_window_set_modal (GTK_WINDOW(AboutWindow), TRUE);
  gtk_window_set_transient_for (GTK_WINDOW(AboutWindow),
				GTK_WINDOW(MainWindow));
  g_signal_connect_swapped (G_OBJECT(AboutWindow), "delete_event",
			     (GtkSignalFunc) gtk_widget_destroy,
			     G_OBJECT(AboutWindow));
  g_signal_connect_swapped (G_OBJECT (AboutWindow), "destroy",
			     (GtkSignalFunc) gtk_widget_destroy,
			     G_OBJECT(AboutWindow));
  gtk_widget_realize (AboutWindow);
  Style = gtk_widget_get_style (AboutWindow);
  Pixmap = gdk_pixmap_create_from_xpm_d (AboutWindow->window, &Mask,
					 &Style->bg[GTK_STATE_NORMAL],
					 about_xpm);
  PixmapWidget = gtk_image_new_from_pixmap (Pixmap, Mask);
  gtk_box_pack_start (GTK_BOX(GTK_DIALOG(AboutWindow) -> vbox),
		      PixmapWidget, FALSE, FALSE, 0);
  HSeparator = gtk_hseparator_new ();
  gtk_box_pack_start (GTK_BOX(GTK_DIALOG(AboutWindow) -> vbox),
		      HSeparator, FALSE, FALSE, 0);
  AboutLabel = gtk_label_new (PACKAGE_STRING "\n"
			      APP_MOTTO "\n\n" APP_URL);
  gtk_box_pack_start (GTK_BOX(GTK_DIALOG(AboutWindow) -> vbox),
		      AboutLabel, FALSE, FALSE, 10);
  Button = gtk_button_new_from_stock (GTK_STOCK_OK);
  g_signal_connect_swapped (G_OBJECT(Button), "clicked",
			     (GtkSignalFunc) gtk_widget_destroy,
			     G_OBJECT(AboutWindow));
  GTK_WIDGET_SET_FLAGS (Button, GTK_CAN_DEFAULT);
  gtk_box_pack_start (GTK_BOX(GTK_DIALOG(AboutWindow) -> action_area),
		      Button, TRUE, TRUE, 0);
  gtk_widget_grab_default (Button);
  gtk_widget_show_all (AboutWindow);
}
