/** Beaver's an Early AdVanced EditoR
** (C) 1999-2000 Marc Bevand, Damien Terrier and Emmanuel Turquin
**
** search.c
**
** Author<s>:     Emmanuel Turquin (aka "Ender") <turqui_e@epita.fr>
** Latest update: Wed May 31 03:24:37 2000
** Description:   Beaver search & replace functions source
**
** This program is free software; you can redistribute it and/or modify
** it under the terms of the GNU General Public License as published by
** the Free Software Foundation; either version 2 of the License, or
** (at your option) any later version.
**
** This program is distributed in the hope that it will be useful,
** but WITHOUT ANY WARRANTY; without even the implied warranty of
** MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
** GNU General Public License for more details.
**
** You should have received a copy of the GNU General Public License
** along with this program; if not, write to the Free Software
** Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

#include <gtk/gtk.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <ctype.h>
#include <sys/stat.h>
#include <sys/types.h>
#include <sys/wait.h>
#include <unistd.h>
#include <wordexp.h>
#include "tools.h"
#include "editor.h"
#include "struct.h"
#include "msgbar.h"
#include "prefs.h"
#include "main.h"
#include "search.h" 
#include "filesops.h"
#include "interface.h"

#define SEARCH_FLAG_CASE_SENSITIVE	0x1
#define SEARCH_FLAG_REGULAR_EXPRESSION	0x2
#define SEARCH_FLAG_WHOLE_WORD		0x4
#define SEARCH_FLAG_SUBDIRS		0x8
#define SEARCH_FLAG_FROM_CURSOR		0xc

extern GArray *FileProperties;
extern GtkWidget *MainNotebook;
extern GtkWidget *MainWindow;
extern GtkWidget *MainVBox;
extern gint OpenedFilesCnt;
extern t_settings	Settings;
static gboolean GotoLineIsVisible = FALSE;

static GtkWidget *SearchDisplay = NULL;
static GtkWidget *SearchTree = NULL;
static GArray *SearchResults = NULL;	/* holds file_results structures */

struct line_match
{
	gint line;
	gchar *text;
};

struct file_results
{
	GtkWidget *buffer;	// NULL if not from a buffer or buffer is closed
	gchar *path;	// Full path of file/buffer
	
	GPtrArray *results;
};

/* this is just a helper function for go_to_search_result */
static gboolean scroll_idle_helper (GtkTextView *view)
{
	show_cursor_on_screen (view);
	
	return FALSE;
}

void go_to_search_result (struct file_results *result, gint match_num)
{
	struct line_match *match;
	gint page;
	GtkTextIter iter, next;
	
	match = g_ptr_array_index (result->results, match_num);
	
	if (match->line == 0)
	{
		/* it's an invalid entry, like a binary file or something */
		return;
	}
	
	/* first, we should look for open files, starting at the
	   current page, that match the filename */
	if (result->buffer)
	{
		/* ok, this is from a buffer search */
		for (page = 0; page < OpenedFilesCnt; page++)
		{
			if (FPROPS (page, Text) == result->buffer)
				break;
		}
		
		if (page == OpenedFilesCnt)
		{
			/* what?  well, let's open it */
			page = -1;
		}
	}
	else
	{
		/* first, we want to find an appropriate buffer to use */
		/* we first look to see if one is open.  if not, open one */
		for (page = 0; page < OpenedFilesCnt; page++)
		{
			if (!strcmp (FPROPS (page, Name), result->path))
			{
				break;
			}
		}
		
		if (page == OpenedFilesCnt)
		{
			/* no find, we must open it */
			page = -1;
		}
	}
	
	if (page == -1)
	{
		open_filename (result->path);
		page = OpenedFilesCnt - 1;
	}
	
	gtk_text_buffer_get_iter_at_line (FPROPS (page, Buffer), 
		&iter, match->line - 1);
	next = iter;
	gtk_text_iter_forward_to_line_end (&next);
	
	gtk_text_buffer_place_cursor (FPROPS (page, Buffer), &iter);
	gtk_text_buffer_move_mark_by_name (FPROPS (page, Buffer), 
		"selection_bound", &next);
	
	switch_to_page (page);
	
	/* hmm..  I have to use an idle loop because it won't scroll to mark
	 or iter unless I do.
	 It is probably because of all the crazy shit 
	 the syhi engine does */
	g_idle_add ((GSourceFunc) scroll_idle_helper, 
		GTK_TEXT_VIEW (FPROPS (page, Text)));
}

void search_result_activated (GtkTreeView *tree, GtkTreePath *path,
	GtkTreeViewColumn *column, gpointer data)
{
	gint *indices;
	
	if (gtk_tree_path_get_depth (path) == 1)
	{
		return;
	}
	
	indices = gtk_tree_path_get_indices (path);
	
	go_to_search_result (
		&g_array_index (SearchResults, struct file_results, indices[0]), 
		indices[1]);
}

void ensure_search_display (void)
{
	START_FCN
	
	if (!SearchDisplay)
	{
		GtkCellRenderer *renderer;
		GtkWidget *scroller;
		SearchDisplay = gtk_vpaned_new ();
		
		g_signal_connect_swapped (G_OBJECT (SearchDisplay), "destroy",
			G_CALLBACK (g_nullify_pointer), &SearchDisplay);
		
		g_object_ref (MainNotebook);
		gtk_container_remove (GTK_CONTAINER (MainVBox), MainNotebook);
		
		gtk_paned_pack1 (GTK_PANED (SearchDisplay), MainNotebook, TRUE, TRUE);
		g_object_unref (MainNotebook);
		
		scroller = gtk_scrolled_window_new (NULL, NULL);
		gtk_scrolled_window_set_policy (GTK_SCROLLED_WINDOW (scroller),
			GTK_POLICY_AUTOMATIC, GTK_POLICY_AUTOMATIC);
		
		SearchTree = gtk_tree_view_new ();
		
		gtk_container_add (GTK_CONTAINER (scroller), SearchTree);
		gtk_paned_pack2 (GTK_PANED (SearchDisplay), scroller, FALSE, TRUE);
		
		/* add the column */
		renderer = gtk_cell_renderer_text_new ();
		gtk_tree_view_insert_column_with_attributes (
					GTK_TREE_VIEW (SearchTree),
					-1, "Match",
					renderer, "text",
					0,
					NULL);
		
		gtk_tree_view_set_headers_visible (GTK_TREE_VIEW (SearchTree), FALSE);
		gtk_tree_view_set_search_column (GTK_TREE_VIEW (SearchTree), 0);
		g_signal_connect (G_OBJECT (SearchTree), "row-activated",
			G_CALLBACK (search_result_activated), NULL);
		
		gtk_container_add (GTK_CONTAINER (MainVBox), SearchDisplay);
		gtk_widget_show_all (SearchDisplay);
	}
	
	END_FCN
}


/* Adds all the data from the array of file_results to a GtkTreeStore
   and puts them in the SearchTree. */
void display_search_results (GArray *array)
{
	gint i, j;
	GtkTreeStore *store;
	
	START_FCN
	
	ensure_search_display ();
	
	store = gtk_tree_store_new (1, G_TYPE_STRING);
	
	for (i = 0; i < array->len; i++)
	{
		GtkTreeIter iter, parent;
		gint max_size;
		struct line_match *match;
		GPtrArray *results;
		gchar *max_line;
		gchar *result_title;
		
		gtk_tree_store_append (store, &parent, NULL);
		result_title = g_strconcat (
				g_array_index (array, struct file_results, i).buffer ? _("Document: ") : _("File: "),
				str_get_last_part (
					g_array_index (array, struct file_results, i).path,
					G_DIR_SEPARATOR, TRUE),
				NULL);
		gtk_tree_store_set (store, &parent,
			0, result_title,
			-1);
		g_free (result_title);
		
		results = g_array_index (array, struct file_results, i).results;
		
		// calculate the maximum number of digits in any of the line numbers
		match = g_ptr_array_index (results, results->len - 1);
		max_line = g_strdup_printf ("%i", match->line);
		max_size = strlen (max_line);
		g_free (max_line);
		
		for (j = 0; j < results->len; j++)
		{
			gchar *output;
			
			match = g_ptr_array_index (results, j);
			
			gtk_tree_store_append (store, &iter, &parent);
			output = g_strdup_printf ("%*i %s", max_size, match->line, match->text);
			gtk_tree_store_set (store, &iter, 
				0, output,
				-1);
			g_free (output);
		}
	}
	
	gtk_tree_view_set_model (GTK_TREE_VIEW (SearchTree), GTK_TREE_MODEL (store));
	g_object_unref (store);
	gtk_tree_view_expand_all (GTK_TREE_VIEW (SearchTree));
	
	gtk_widget_show_all (SearchTree);
	
	END_FCN
}

/* A sort of 'strcmp' for the characters: it returns '0' if the characteres
   are the same */

gint char_cmp (gboolean case_sen,
	       gchar char1,
	       gchar char2)
{
  if (!case_sen)
    {
      gint diff = 'A' - 'a';
      
      if (isupper(char1) && islower(char2))
	return (char2 + diff - char1);
      else if (isupper(char2) && islower(char1))
	return (char2 - diff - char1);
    }
  return (char2 - char1);
}


/* Replace all the 'string_in' by 'string_out' */

gint replace_all (gint page,
		  gboolean case_sen,
		  gboolean reg_exp,
		  gint start_pos,
		  const gchar *string_in,
		  const gchar *string_out)
{
  gchar *buffer_in, *buffer_out;
  gint len_in, len_out, string_in_len, string_out_len;
  gint i, j, k, rep_nb = 0;
  GtkTextIter start, end;
  GtkTextBuffer *Buffer;
  
  (void)reg_exp; /* regular expressions are not yet implemented */
  buffer_in = get_text (gtk_text_view_get_buffer 
              (GTK_TEXT_VIEW(FPROPS(page, Text))));
  len_in = strlen (buffer_in);
  len_out = strlen (buffer_in) + 10;
  string_in_len = strlen (string_in);
  string_out_len = strlen (string_out);
  buffer_out = g_malloc (len_out);
  for (i = 0; i < start_pos; i++)
    buffer_out[i] = buffer_in[i];
  j = i;
  while (i <= len_in)
    {
      for (k = 0; (string_in[k] != '\0') &&
	     !char_cmp(case_sen, buffer_in[i+k], string_in[k]); k++);
      if (k == string_in_len)
	{
	  rep_nb++;
	  len_out = len_out + string_out_len - string_in_len;
	  buffer_out = g_realloc (buffer_out, len_out);
	  for (k = 0; k < string_out_len; j++, k++)
	    buffer_out[j] = string_out[k];
	  i += string_in_len;
	}
      else
	{
	  buffer_out[j] = buffer_in[i];
	  i++;
	  j++;
	}
    }
  buffer_out[j] = '\0';
  Buffer = gtk_text_view_get_buffer (GTK_TEXT_VIEW(FPROPS(page, Text)));
  gtk_text_buffer_get_start_iter (Buffer, &start);
  gtk_text_buffer_get_iter_at_offset (Buffer, &end, len_in);
  gtk_text_buffer_delete (Buffer, &start, &end);
  gtk_text_buffer_get_iter_at_offset (Buffer, &end, i);
  gtk_text_buffer_insert (Buffer, &end, buffer_out, -1);
  g_free (buffer_in);
  g_free (buffer_out);
  return (rep_nb);
}

/* expands a reg exp file pattern to a list of files */
/* this returned list is an array of strings, with |pre|
   NULL entries before the results, a bunch of results,
   and an ending NULL. */
static gchar **get_files_for_search (const gchar *directory, const gchar *regexp, gint pre)
{
	gchar **answer;
	wordexp_t word;
	
	START_FCN
	
	/* We actually change the working directory here.  I'm pretty sure this
	   is safe since we don't use working directories ever in Beaver. 
	   The reason we do this is to make wordexp work right when given
	   multiple regular expressions without doing much work on checking for
	   spaces that aren't quoted and such.
	*/
	chdir (directory);
	
	word.we_offs = pre;
	if (wordexp (regexp, &word, WRDE_DOOFFS))
	{
		gint i;
		
		/* something went wrong, fill out answer with nulls */
		
		answer = g_malloc (sizeof (gchar *) * (pre + 1));
		
		for (i = 0; i < pre; i++)
		{
			answer[i] = NULL;
		}
		answer[i] = NULL;
	}
	else
	{
		gint i;
		
		/* fill out our array with results */
		answer = g_malloc (sizeof (gchar *) * 
			(word.we_wordc + word.we_offs + 1));
		for (i = word.we_offs; i < word.we_offs + word.we_wordc; i++)
		{
			answer[i] = g_strdup (word.we_wordv[i]);
		}
		answer[i] = NULL;
	}
	
	wordfree (&word);
	
	END_FCN
	return answer;
}

static struct line_match *parse_grep_output_line (gchar *line, gint line_offset)
{
	struct line_match *rv = (struct line_match *) g_malloc (sizeof (struct line_match));
	gchar *colon;
	
	START_FCN
	
	colon = strchr (line, ':');
	if (colon)
	{
		colon[0] = '\0';
		rv->line = atoi (line) + line_offset;
		colon[0] = ':';
		colon++;
		rv->text = g_strdup (colon);
	}
	else
	{
		rv->line = 0;
		rv->text = g_strdup (line);
	}
	
	END_FCN
	
	return rv;
}

static struct file_results parse_grep_output (gchar *output, GtkWidget *widget, 
	const gchar *filename, gint line_offset)
{
	struct file_results rv;
	gchar *next;
	
	START_FCN
	
	rv.buffer = widget;
	if (widget)
		g_signal_connect_swapped (G_OBJECT (widget), "destroy", 
			G_CALLBACK (g_nullify_pointer), &rv.buffer);
	
	if (!g_path_is_absolute (filename))
	{
		gchar *current_dir = g_get_current_dir ();
		
		rv.path = g_build_filename (current_dir, filename, NULL);
		g_free (current_dir);
	}
	else
	{
		rv.path = g_strdup (filename);
	}
	rv.results = g_ptr_array_new ();
	
	while ((next = strchr (output, '\n')))
	{
		next[0] = '\0';
		g_ptr_array_add (rv.results, 
			parse_grep_output_line (output, line_offset));
		next[0] = '\n';
		output = next + 1;
	}
	
	if (rv.results->len == 0)
	{
		g_ptr_array_free (rv.results, FALSE);
		rv.results = NULL;
	}
	
	END_FCN
	
	return rv;
}

/* escapes each character in |string|, returned string must be g_free'd  */
static gchar *escape_ere_string (const gchar *string)
{
	gchar *escaped, *p;
	const gchar *q;
	glong strsize;
	
	if (!g_utf8_validate (string, -1, NULL))
	{
		return NULL;
	}
	
	strsize = g_utf8_strlen (string, -1);
	escaped = (gchar *) g_malloc (strlen (string) * 2 + 1);
	
	q = string;
	p = escaped;
	while (*q)
	{
		if (*q == '.' || *q == '[' || *q == '\\' ||
		    *q == '(' || *q == ')' || *q == '*' || 
		    *q == '+' || *q == '?' || *q == '{' ||
		    *q == '|' || *q == '^' || *q == '$') 
		{
			p[0] = '\\';
			p ++;
		}
		
		g_utf8_strncpy (p, q, 1);
		p = g_utf8_next_char (p);
		q = g_utf8_next_char (q);
	}
	
	p[0] = '\0';
	
	return escaped;
}

#if 0 /* we don't use this right now */
/* escapes each bre-special character in |string|, returned string must be g_free'd  */
static gchar *escape_bre_string (const gchar *string)
{
	gchar *escaped, *p;
	const gchar *q;
	glong strsize;
	
	if (!g_utf8_validate (string, -1, NULL))
	{
		return NULL;
	}
	
	strsize = g_utf8_strlen (string, -1);
	escaped = (gchar *) g_malloc (strlen (string) * 2 + 1);
	
	q = string;
	p = escaped;
	while (*q)
	{
		if (*q == '.' || *q == '[' || *q == '\\' || *q == '*' || 
		    *q == '^' || *q == '$') 
		{
			p[0] = '\\';
			p ++;
		}
		
		g_utf8_strncpy (p, q, 1);
		p = g_utf8_next_char (p);
		q = g_utf8_next_char (q);
	}
	
	p[0] = '\0';
	
	return escaped;
}
#endif

static struct file_results run_grep_on_file (const gchar *pattern, 
	const gchar *filename, gint flags)
{
	gchar *args[10];
	gint i = 0;
	gchar *answers;
	struct file_results rv;
	gchar *mypattern;
	gchar *filename_copy;
	
	START_FCN
	
	args[i++] = "grep";
	
	if (flags & SEARCH_FLAG_WHOLE_WORD)
	{
		if (flags & SEARCH_FLAG_REGULAR_EXPRESSION)
		{
			mypattern = g_strconcat ("(^|[^[:alnum:]])", pattern, "($|[^[:alnum:]])", NULL);
		}
		else
		{
			gchar *escaped_pattern = escape_ere_string (pattern);
			if (!escaped_pattern) escaped_pattern = g_strdup (pattern);
			mypattern = g_strconcat ("(^|[^[:alnum:]])", escaped_pattern, "($|[^[:alnum:]])", NULL);
			g_free (escaped_pattern);
		}
		
		flags |= SEARCH_FLAG_REGULAR_EXPRESSION;
	}
	else
	{
		mypattern = g_strdup (pattern);
	}
	
	if (pattern) args[i++] = mypattern;
	args[i++] = "-ns";
	if (!(flags & SEARCH_FLAG_CASE_SENSITIVE)) args[i++] = "-i";
	if (!(flags & SEARCH_FLAG_REGULAR_EXPRESSION)) args[i++] = "-F";
	else args[i++] = "-E";
	filename_copy = g_strdup (filename);
	args[i++] = filename_copy;
	args[i++] = NULL;
	
	answers = run_program_with_args ("grep", args, NULL);
	g_free (filename_copy);
	
	rv = parse_grep_output (answers, NULL, filename, 0);
	
	g_free (answers);
	g_free (mypattern);
	
	END_FCN
	
	return rv;
}

static struct file_results run_grep_on_text (const gchar *pattern, gint page, 
	gint flags)
{
	gchar *args[10];
	gint i = 0;
	gchar *answers;
	struct file_results rv;
	gchar *text;
	gchar *mypattern;
	gint offset;
	
	START_FCN
	
	if (flags & SEARCH_FLAG_FROM_CURSOR)
	{
		text = get_text_from_cursor (FPROPS (page, Buffer));
		offset = get_line_of_cursor (FPROPS (page, Buffer));
	}
	else
	{
		text = get_text (FPROPS (page, Buffer));
		offset = 0;
	}
	
	args[i++] = "grep";
	
	if (flags & SEARCH_FLAG_WHOLE_WORD)
	{
		if (flags & SEARCH_FLAG_REGULAR_EXPRESSION)
		{
			mypattern = g_strconcat ("(^|[^[:alnum:]])", pattern, "($|[^[:alnum:]])", NULL);
		}
		else
		{
			gchar *escaped_pattern = escape_ere_string (pattern);
			if (!escaped_pattern) escaped_pattern = g_strdup (pattern);
			mypattern = g_strconcat ("(^|[^[:alnum:]])", escaped_pattern, "($|[^[:alnum:]])", NULL);
			g_free (escaped_pattern);
		}
		
		flags |= SEARCH_FLAG_REGULAR_EXPRESSION;
	}
	else
	{
		mypattern = g_strdup (pattern);
	}
	
	if (pattern) args[i++] = mypattern;
	args[i++] = "-ns";
	if (!(flags & SEARCH_FLAG_CASE_SENSITIVE)) args[i++] = "-i";
	if (!(flags & SEARCH_FLAG_REGULAR_EXPRESSION)) args[i++] = "-F";
	else args[i++] = "-E";
	args[i++] = "-";
	args[i++] = NULL;
	
	answers = run_program_with_args ("grep", args, text);
	
	rv = parse_grep_output (answers, 
		FPROPS (page, Text), FPROPS (page, Name), offset);
	
	g_free (answers);
	g_free (text);
	g_free (mypattern);
	
	END_FCN
	
	return rv;
}

static void run_awk_on_file (const gchar *pattern, const gchar *replacement, 
	const gchar *filename, gint flags)
{
	gchar *args[10];
	gint i = 0;
	gchar *answers;
	gchar *mypattern;
	gchar *myawkcommand;
	gchar *myreplacement;
	gchar *filename_copy;
	FILE *tmp;
	gchar *fullpath;
	
	START_FCN
	
	mypattern = g_strdup (pattern);
	if (!(flags & SEARCH_FLAG_REGULAR_EXPRESSION))
	{
		gchar *escaped_pattern = escape_ere_string (mypattern);
		if (escaped_pattern)
		{
			g_free (mypattern);
			mypattern = escaped_pattern;
		}
	}
	else
	{
		/* just escape the forward slashes */
		str_replace_tokens (&mypattern, '/', "\\/");
	}
	
	if (flags & SEARCH_FLAG_WHOLE_WORD)
	{
		gchar *tmp;
		tmp = g_strconcat ("(^|[^[:alnum:]])", mypattern, 
			"($|[^[:alnum:]])", NULL);
		g_free (mypattern);
		mypattern = tmp;
	}
	
	myreplacement = g_strdup (replacement);
	str_replace_tokens (&myreplacement, '&', "\\&");
	myawkcommand = g_strconcat ("{gsub(/", mypattern, "/,\"", 
		myreplacement, "\"); print}", NULL);
	g_free (mypattern);
	g_free (myreplacement);
	
	args[i++] = "awk";
	args[i++] = myawkcommand;
	filename_copy = g_strdup (filename);
	args[i++] = filename_copy;
	args[i++] = NULL;
	
	answers = run_program_with_args ("awk", args, NULL);
	g_free (filename_copy);
	
	tmp = fopen (filename, "w");
	fwrite (answers, strlen (answers), 1, tmp);
	fclose (tmp);
	
	/* Now we must go through open documents and ask if the user wants to
	   update the view. */
	if (g_path_is_absolute (filename))
	{
		fullpath = g_strdup (filename);
	}
	else
	{
		fullpath = g_build_filename (DIRECTORY, filename, NULL);
	}
	for (i = 0; i < OpenedFilesCnt; i++)
	{
		if (strcmp (FPROPS (i, Name), fullpath) == 0)
		{
			GtkWidget *dialog;
			gchar *primary, *secondary;
			gint response;
			
			primary = g_strdup_printf (
			               _("Refresh the file \"%s\"?"), 
			               filename);
			secondary = g_strdup_printf (
			               _("The file \"%s\" has changed."
			               "  If you do not refresh it, you"
			               " may lose these changes."), 
			               filename);
			dialog = alert_new (GTK_WINDOW (MainWindow), 
			               GTK_STOCK_DIALOG_QUESTION,
			               primary, secondary);
			g_free (primary);
			g_free (secondary);
			gtk_dialog_add_buttons (GTK_DIALOG (dialog), 
			               GTK_STOCK_NO, 1, 
			               GTK_STOCK_REFRESH, 2, NULL);
			gtk_dialog_set_default_response (
			               GTK_DIALOG (dialog), 2);
			response = gtk_dialog_run (GTK_DIALOG (dialog));
			gtk_widget_destroy (dialog);
			
			switch (response)
			{
				default:
				case GTK_RESPONSE_NONE:
				case 1:
					break;
				case 2:
					load_file_contents_into_buffer (
						i, filename);
					break;
			}
		}
	}
	
	g_free (fullpath);
	g_free (answers);
	g_free (myawkcommand);
	
	END_FCN
}

static void run_awk_on_text (gchar *pattern, gchar *replacement, 
	gint page, gint flags)
{
	gchar *args[10];
	gint i = 0;
	gchar *answers;
	gchar *text;
	gchar *mypattern;
	gchar *myawkcommand;
	gchar *myreplacement;
	
	START_FCN
	
	if (flags & SEARCH_FLAG_FROM_CURSOR)
	{
		text = get_text_from_cursor (FPROPS (page, Buffer));
	}
	else
	{
		text = get_text (FPROPS (page, Buffer));
	}
	
	mypattern = g_strdup (pattern);
	if (!(flags & SEARCH_FLAG_REGULAR_EXPRESSION))
	{
		gchar *escaped_pattern = escape_ere_string (mypattern);
		if (escaped_pattern)
		{
			g_free (mypattern);
			mypattern = escaped_pattern;
		}
	}
	else
	{
		/* just escape the forward slashes */
		str_replace_tokens (&mypattern, '/', "\\/");
	}
	
	if (flags & SEARCH_FLAG_WHOLE_WORD)
	{
		gchar *tmp;
		tmp = g_strconcat ("(^|[^[:alnum:]])", mypattern, 
			"($|[^[:alnum:]])", NULL);
		g_free (mypattern);
		mypattern = tmp;
	}
	
	myreplacement = g_strdup (replacement);
	str_replace_tokens (&myreplacement, '&', "\\&");
	myawkcommand = g_strconcat ("{gsub(/", mypattern, "/,\"", 
		myreplacement, "\"); print}", NULL);
	g_free (mypattern);
	g_free (myreplacement);
	
	args[i++] = "awk";
	args[i++] = myawkcommand;
	args[i++] = "-";
	args[i++] = NULL;
	
	answers = run_program_with_args ("awk", args, text);
	
	if (flags & SEARCH_FLAG_FROM_CURSOR)
	{
		gchar *before = get_text_to_cursor (FPROPS (page, Buffer));
		gchar *total = g_strconcat (before, answers, NULL);
		g_free (answers);
		g_free (before);
		answers = total;
	}
	
	gtk_text_buffer_set_text (FPROPS (page, Buffer), answers, -1);
	
	g_free (answers);
	g_free (text);
	g_free (myawkcommand);
	
	END_FCN
}

void search_callback (GtkDialog *dialog, gint response, gpointer data)
{
	gint replace;
	START_FCN
	
	replace = GPOINTER_TO_INT (data);
	if (response == GTK_RESPONSE_ACCEPT)
	{
		GtkWidget *widget;
		gint page;
		gchar *pattern = NULL;
		gchar *replacement = NULL;
		gint i, start, end, flags;
		gboolean all_open_documents;
		struct file_results output;
		
		if (SearchResults)
		{
			for (i = 0; i < SearchResults->len; i++)
			{
				g_free (g_array_index (SearchResults, struct file_results, i).path);
				g_ptr_array_free (g_array_index (SearchResults, struct file_results, i).results, TRUE);
			}
			
			g_array_free (SearchResults, FALSE);
		}
		SearchResults = g_array_new (FALSE, FALSE, sizeof (struct file_results));
		
		flags = 0;
		
		if (!replace)
		{
			/* replace doesn't yet handle case insensitivity */
			widget = g_object_get_data (G_OBJECT (dialog), "case");
			if (gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (widget)))
			{
				flags |= SEARCH_FLAG_CASE_SENSITIVE;
			}
		}
		
		widget = g_object_get_data (G_OBJECT (dialog), "regexp");
		if (gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (widget)))
		{
			flags |= SEARCH_FLAG_REGULAR_EXPRESSION;
		}
		
		widget = g_object_get_data (G_OBJECT (dialog), "word");
		if (gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (widget)))
		{
			flags |= SEARCH_FLAG_WHOLE_WORD;
		}
		
		widget = g_object_get_data (G_OBJECT (dialog), "from_cursor");
		if (gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (widget)))
		{
			flags |= SEARCH_FLAG_FROM_CURSOR;
		}
		
		widget = g_object_get_data (G_OBJECT (dialog), "all");
		all_open_documents = gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (widget));
		
		widget = g_object_get_data (G_OBJECT (dialog), "text");
		pattern = g_strdup (gtk_entry_get_text (GTK_ENTRY (widget)));
		
		if (replace)
		{
			widget = g_object_get_data (G_OBJECT (dialog), "replace_text");
			replacement = g_strdup (gtk_entry_get_text (GTK_ENTRY (widget)));
		}
		
		page = gtk_notebook_get_current_page (GTK_NOTEBOOK (MainNotebook));
		
		if (all_open_documents)
		{
			start = 0;
			end = OpenedFilesCnt;
		}
		else
		{
			start = page;
			end = page + 1;
		}
		
		for (i = start; i < end; i++)
		{
			if (replace)
			{
				run_awk_on_text (pattern, replacement, i, flags);
			}
			else
			{
				output = run_grep_on_text (pattern, i, flags);
				if (output.results)
				{
					g_array_append_val (SearchResults, output);
				}
			}
		}
		
		if (!replace)
			display_search_results (SearchResults);
		g_free (pattern);
		g_free (replacement);
	}
	
	gtk_widget_destroy (GTK_WIDGET (dialog));
	
	END_FCN
}

void search_in_files_callback (GtkDialog *dialog, gint response, gpointer data)
{
	gint replace;
	START_FCN
	
	replace = GPOINTER_TO_INT (data);
	if (response == GTK_RESPONSE_ACCEPT)
	{
		GtkWidget *widget;
		gchar *pattern = NULL;
		gchar *directory = NULL;
		gchar *filepattern = NULL;
		gchar *replacement = NULL;
		gchar **expanded_files;
		gint i, flags;
		struct file_results output;
		
		if (SearchResults)
		{
			for (i = 0; i < SearchResults->len; i++)
			{
				g_free (g_array_index (SearchResults, struct file_results, i).path);
				g_ptr_array_free (g_array_index (SearchResults, struct file_results, i).results, TRUE);
			}
			
			g_array_free (SearchResults, FALSE);
		}
		SearchResults = g_array_new (FALSE, FALSE, sizeof (struct file_results));
		
		flags = 0;
		
		if (!replace)
		{
			widget = g_object_get_data (G_OBJECT (dialog), "case");
			if (gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (widget)))
			{
				flags |= SEARCH_FLAG_CASE_SENSITIVE;
			}
		}
		
		widget = g_object_get_data (G_OBJECT (dialog), "regexp");
		if (gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (widget)))
		{
			flags |= SEARCH_FLAG_REGULAR_EXPRESSION;
		}
		
		widget = g_object_get_data (G_OBJECT (dialog), "word");
		if (gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (widget)))
		{
			flags |= SEARCH_FLAG_WHOLE_WORD;
		}
		/*
		widget = g_object_get_data (G_OBJECT (dialog), "subdirs");
		if (gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (widget)))
		{
			flags |= SEARCH_FLAG_SUBDIRS;
		}*/
		
		if (replace)
		{
			widget = g_object_get_data (G_OBJECT (dialog), "replace_text");
			replacement = g_strdup (gtk_entry_get_text (GTK_ENTRY (widget)));
		}
		
		widget = g_object_get_data (G_OBJECT (dialog), "text");
		
		pattern = g_strdup (gtk_entry_get_text (GTK_ENTRY (widget)));
		
		widget = g_object_get_data (G_OBJECT (dialog), "directory");
		directory = g_strdup (gtk_entry_get_text (GTK_ENTRY (widget)));
		widget = g_object_get_data (G_OBJECT (dialog), "file-pattern");
		filepattern = g_strdup (gtk_entry_get_text (GTK_ENTRY (widget)));
		expanded_files = get_files_for_search (directory, filepattern, flags);
		
		for (i = 0; expanded_files[i]; i++)
		{
			if (replace)
			{
				run_awk_on_file (pattern, replacement, expanded_files[i], flags);
			}
			else
			{
				output = run_grep_on_file (pattern, expanded_files[i], 0);
				if (output.results)
					g_array_append_val (SearchResults, output);
			}
		}
		
		g_strfreev (expanded_files);
		g_free (filepattern);
		g_free (directory);
		g_free (replacement);
		
		if (!replace)
			display_search_results (SearchResults);
	}
	
	gtk_widget_destroy (GTK_WIDGET (dialog));
	
	END_FCN
}
/* Display a Search Dialog window */

void search (gboolean replace)
{
  static GtkWidget *SearchWindow = NULL;
  GtkWidget *main_vbox, *hbox;
  GtkWidget *widget, *label;

  if (SearchWindow)
  {
  	gtk_window_present (GTK_WINDOW (SearchWindow));
  	return;
  }
  
  if (replace)
  {
  	SearchWindow = gtk_dialog_new_with_buttons (
			_("Replace"),
			GTK_WINDOW (MainWindow),
			GTK_DIALOG_DESTROY_WITH_PARENT | GTK_DIALOG_NO_SEPARATOR,
			GTK_STOCK_CANCEL, GTK_RESPONSE_REJECT,
			_("_Replace All"), GTK_RESPONSE_ACCEPT,
			NULL);
  }
  else
  {
  	SearchWindow = gtk_dialog_new_with_buttons (
			_("Search"),
			GTK_WINDOW (MainWindow),
			GTK_DIALOG_DESTROY_WITH_PARENT | GTK_DIALOG_NO_SEPARATOR,
			GTK_STOCK_CANCEL, GTK_RESPONSE_REJECT,
			GTK_STOCK_FIND, GTK_RESPONSE_ACCEPT,
			NULL);
  }
  gtk_widget_realize (SearchWindow);
  gtk_window_set_resizable (GTK_WINDOW (SearchWindow), TRUE);
  gdk_window_set_functions (SearchWindow->window, 
	GDK_FUNC_MOVE | GDK_FUNC_MINIMIZE);
  g_signal_connect_swapped (G_OBJECT(SearchWindow), "delete-event",
			     (GtkSignalFunc) gtk_widget_destroy,
			     G_OBJECT(SearchWindow));
  g_signal_connect_swapped (G_OBJECT (SearchWindow), "destroy",
			     G_CALLBACK (g_nullify_pointer),
			     &SearchWindow);
  
  main_vbox = gtk_vbox_new (FALSE, 12);
  gtk_box_set_spacing (GTK_BOX (GTK_DIALOG (SearchWindow)->vbox), 12);
  gtk_container_set_border_width (GTK_CONTAINER (main_vbox), 6);
  gtk_container_set_border_width (GTK_CONTAINER (SearchWindow), 6);
  gtk_container_add (GTK_CONTAINER (GTK_DIALOG (SearchWindow)->vbox), main_vbox);
  
  gtk_dialog_set_default_response (GTK_DIALOG (SearchWindow), GTK_RESPONSE_ACCEPT);
  
  g_signal_connect (G_OBJECT (SearchWindow), "response", 
	G_CALLBACK (search_callback), GINT_TO_POINTER ((gint) replace));
  
  hbox = gtk_hbox_new (FALSE, 12);
  gtk_container_add (GTK_CONTAINER (main_vbox), hbox);
  
  widget = gtk_label_new_with_mnemonic ("_Search for:");
  gtk_misc_set_alignment (GTK_MISC (widget), 0, 0.5);
  gtk_container_add (GTK_CONTAINER (hbox), widget);
  label = widget;
  
  widget = gtk_entry_new ();
  gtk_container_add (GTK_CONTAINER (hbox), widget);
  g_object_set_data (G_OBJECT (SearchWindow), "text", widget);
  gtk_entry_set_activates_default (GTK_ENTRY (widget), TRUE);
  gtk_label_set_mnemonic_widget (GTK_LABEL (label), widget);
  
  if (replace)
  {
	  hbox = gtk_hbox_new (FALSE, 12);
	  gtk_container_add (GTK_CONTAINER (main_vbox), hbox);
	  
	  widget = gtk_label_new_with_mnemonic ("R_eplace with:");
	  gtk_misc_set_alignment (GTK_MISC (widget), 0, 0.5);
	  gtk_container_add (GTK_CONTAINER (hbox), widget);
	  label = widget;
	  
	  widget = gtk_entry_new ();
	  gtk_container_add (GTK_CONTAINER (hbox), widget);
	  g_object_set_data (G_OBJECT (SearchWindow), "replace_text", widget);
  	  gtk_entry_set_activates_default (GTK_ENTRY (widget), TRUE);
	  gtk_label_set_mnemonic_widget (GTK_LABEL (label), widget);
  }
  
  if (!replace)
  {
  	/* replace can't handle no case */
	  widget = gtk_check_button_new_with_mnemonic ("Match c_ase");
	  gtk_box_pack_start (GTK_BOX (main_vbox), widget, FALSE, FALSE, 0);
	  g_object_set_data (G_OBJECT (SearchWindow), "case", widget);
  }
  
  widget = gtk_check_button_new_with_mnemonic ("Match _whole word only");
  gtk_box_pack_start (GTK_BOX (main_vbox), widget, FALSE, FALSE, 0);
  g_object_set_data (G_OBJECT (SearchWindow), "word", widget);
  
  widget = gtk_check_button_new_with_mnemonic ("Search text is a re_gular expression");
  gtk_box_pack_start (GTK_BOX (main_vbox), widget, FALSE, FALSE, 0);
  g_object_set_data (G_OBJECT (SearchWindow), "regexp", widget);
  
  widget = gtk_check_button_new_with_mnemonic ("S_tart search at cursor");
  gtk_box_pack_start (GTK_BOX (main_vbox), widget, FALSE, FALSE, 0);
  g_object_set_data (G_OBJECT (SearchWindow), "from_cursor", widget);
  
  widget = gtk_check_button_new_with_mnemonic ("Search in all _open documents");
  gtk_box_pack_start (GTK_BOX (main_vbox), widget, FALSE, FALSE, 0);
  g_object_set_data (G_OBJECT (SearchWindow), "all", widget);
  
  gtk_widget_show_all (SearchWindow);
}

/* Display a Search in Files dialog window */

void search_in_files (gboolean replace)
{
  static GtkWidget *SearchFilesWindow = NULL;
  GtkWidget *main_vbox, *hbox;
  GtkWidget *widget, *label;
  GtkSizeGroup *size_group_label, *size_group_text;

  if (SearchFilesWindow)
  {
  	gtk_window_present (GTK_WINDOW (SearchFilesWindow));
  	return;
  }
  
  if (replace)
  {
  	SearchFilesWindow = gtk_dialog_new_with_buttons (
			_("Replace in Files"),
			GTK_WINDOW (MainWindow),
			GTK_DIALOG_DESTROY_WITH_PARENT | GTK_DIALOG_NO_SEPARATOR,
			GTK_STOCK_CANCEL, GTK_RESPONSE_REJECT,
			_("_Replace All"), GTK_RESPONSE_ACCEPT,
			NULL);
  }
  else
  {
	  SearchFilesWindow = gtk_dialog_new_with_buttons (
			_("Search in Files"),
			GTK_WINDOW (MainWindow),
			GTK_DIALOG_DESTROY_WITH_PARENT | GTK_DIALOG_NO_SEPARATOR,
			GTK_STOCK_CANCEL, GTK_RESPONSE_REJECT,
			GTK_STOCK_FIND, GTK_RESPONSE_ACCEPT,
			NULL);
  }
  gtk_widget_realize (SearchFilesWindow);
  gtk_window_set_resizable (GTK_WINDOW (SearchFilesWindow), TRUE);
  gdk_window_set_functions (SearchFilesWindow->window, 
	GDK_FUNC_MOVE | GDK_FUNC_MINIMIZE);
  g_signal_connect_swapped (G_OBJECT(SearchFilesWindow), "delete-event",
			     (GtkSignalFunc) gtk_widget_destroy,
			     G_OBJECT(SearchFilesWindow));
  g_signal_connect_swapped (G_OBJECT (SearchFilesWindow), "destroy",
			     G_CALLBACK (g_nullify_pointer),
			     &SearchFilesWindow);
  
  size_group_label = gtk_size_group_new (GTK_SIZE_GROUP_HORIZONTAL);
  size_group_text = gtk_size_group_new (GTK_SIZE_GROUP_HORIZONTAL);
  
  main_vbox = gtk_vbox_new (FALSE, 12);
  gtk_box_set_spacing (GTK_BOX (GTK_DIALOG (SearchFilesWindow)->vbox), 12);
  gtk_container_set_border_width (GTK_CONTAINER (main_vbox), 6);
  gtk_container_set_border_width (GTK_CONTAINER (SearchFilesWindow), 6);
  gtk_container_add (GTK_CONTAINER (
	GTK_DIALOG (SearchFilesWindow)->vbox), main_vbox);
  
  gtk_dialog_set_default_response (GTK_DIALOG (SearchFilesWindow),
	 GTK_RESPONSE_ACCEPT);
  
  g_signal_connect (G_OBJECT (SearchFilesWindow), "response",
	 G_CALLBACK (search_in_files_callback), GINT_TO_POINTER ((gint) replace));
  
  hbox = gtk_hbox_new (FALSE, 12);
  gtk_container_add (GTK_CONTAINER (main_vbox), hbox);
  
  widget = gtk_label_new_with_mnemonic ("_Search for:");
  gtk_misc_set_alignment (GTK_MISC (widget), 0, 0.5);
  gtk_container_add (GTK_CONTAINER (hbox), widget);
  gtk_size_group_add_widget (size_group_label, widget);
  label = widget;
  
  widget = gtk_entry_new ();
  gtk_container_add (GTK_CONTAINER (hbox), widget);
  g_object_set_data (G_OBJECT (SearchFilesWindow), "text", widget);
  gtk_size_group_add_widget (size_group_text, widget);
  gtk_entry_set_activates_default (GTK_ENTRY (widget), TRUE);
  gtk_label_set_mnemonic_widget (GTK_LABEL (label), widget);
  
  if (replace)
  {
	  hbox = gtk_hbox_new (FALSE, 12);
	  gtk_container_add (GTK_CONTAINER (main_vbox), hbox);
	  
	  widget = gtk_label_new_with_mnemonic ("R_eplace with:");
	  gtk_misc_set_alignment (GTK_MISC (widget), 0, 0.5);
	  gtk_container_add (GTK_CONTAINER (hbox), widget);
  	  gtk_size_group_add_widget (size_group_label, widget);
  	  label = widget;
	  
	  widget = gtk_entry_new ();
	  gtk_container_add (GTK_CONTAINER (hbox), widget);
	  g_object_set_data (G_OBJECT (SearchFilesWindow), "replace_text", widget);
  	  gtk_size_group_add_widget (size_group_text, widget);
  	  gtk_entry_set_activates_default (GTK_ENTRY (widget), TRUE);
	  gtk_label_set_mnemonic_widget (GTK_LABEL (label), widget);
  }
  
  hbox = gtk_hbox_new (FALSE, 12);
  gtk_container_add (GTK_CONTAINER (main_vbox), hbox);
  
  widget = gtk_label_new_with_mnemonic ("Search in these fi_les: ");
  gtk_misc_set_alignment (GTK_MISC (widget), 0, 0.5);
  gtk_container_add (GTK_CONTAINER (hbox), widget);
  gtk_size_group_add_widget (size_group_label, widget);
  label = widget;
  
  widget = gtk_entry_new ();
  gtk_entry_set_text (GTK_ENTRY (widget), "*");
  gtk_container_add (GTK_CONTAINER (hbox), widget);
  g_object_set_data (G_OBJECT (SearchFilesWindow), "file-pattern", widget);
  gtk_size_group_add_widget (size_group_text, widget);
  gtk_entry_set_activates_default (GTK_ENTRY (widget), TRUE);
  gtk_label_set_mnemonic_widget (GTK_LABEL (label), widget);
  
  hbox = gtk_hbox_new (FALSE, 12);
  gtk_container_add (GTK_CONTAINER (main_vbox), hbox);
  
  widget = gtk_label_new_with_mnemonic ("Search in this _directory: ");
  gtk_misc_set_alignment (GTK_MISC (widget), 0, 0.5);
  gtk_container_add (GTK_CONTAINER (hbox), widget);
  gtk_size_group_add_widget (size_group_label, widget);
  label = widget;
  
  widget = gtk_entry_new ();
  gtk_entry_set_text (GTK_ENTRY (widget), DIRECTORY);
  gtk_container_add (GTK_CONTAINER (hbox), widget);
  g_object_set_data (G_OBJECT (SearchFilesWindow), "directory", widget);
  gtk_size_group_add_widget (size_group_text, widget);
  gtk_entry_set_activates_default (GTK_ENTRY (widget), TRUE);
  gtk_label_set_mnemonic_widget (GTK_LABEL (label), widget);
  
  if (!replace)
  {
  	/* replace can't handle no case */
	  widget = gtk_check_button_new_with_mnemonic ("Match c_ase");
	  gtk_box_pack_start (GTK_BOX (main_vbox), widget, FALSE, FALSE, 0);
	  g_object_set_data (G_OBJECT (SearchFilesWindow), "case", widget);
  }
  
  widget = gtk_check_button_new_with_mnemonic ("Match _whole word only");
  gtk_box_pack_start (GTK_BOX (main_vbox), widget, FALSE, FALSE, 0);
  g_object_set_data (G_OBJECT (SearchFilesWindow), "word", widget);
  
  widget = gtk_check_button_new_with_mnemonic 
	(_("Search text is a re_gular expression"));
  gtk_box_pack_start (GTK_BOX (main_vbox), widget, FALSE, FALSE, 0);
  g_object_set_data (G_OBJECT (SearchFilesWindow), "regexp", widget);
  /*
  widget = gtk_check_button_new_with_mnemonic ("Search s_ubdirectories");
  gtk_box_pack_start (GTK_BOX (main_vbox), widget, FALSE, FALSE, 0);
  g_object_set_data (G_OBJECT (SearchFilesWindow), "subdirs", widget);
  */
  gtk_widget_show_all (SearchFilesWindow);
}

void show_cursor_on_screen (GtkTextView *view)
{
	GtkTextMark *mark;
	GtkTextBuffer *buffer;
	
	buffer = gtk_text_view_get_buffer (view);
	mark = gtk_text_buffer_get_mark (buffer, "insert");
	
	gtk_text_view_scroll_to_mark (view, mark, 0, TRUE, 0, 0.5);
}

void show_on_screen (GtkTextView *Text, GtkTextIter start)
{
	gtk_text_view_scroll_to_iter (Text, &start, 0, TRUE, 0, 0.5);
}

void set_line (GtkWidget *SpinButton)
{
  gint CurrentPage, line_to_go;
  gchar *msg;
  GtkTextIter start, end;
  GtkTextBuffer *EditorBuffer;

  if (!OpenedFilesCnt)
    {
      print_msg ("Hey, you closed all the files!");
      return;
    }
  line_to_go = gtk_spin_button_get_value_as_int (GTK_SPIN_BUTTON(SpinButton));
  CurrentPage = gtk_notebook_get_current_page (GTK_NOTEBOOK(MainNotebook));
  EditorBuffer = FPROPS(CurrentPage, Buffer);
  
  gtk_text_buffer_get_iter_at_line (EditorBuffer, &start, line_to_go - 1);
  gtk_text_buffer_get_iter_at_line (EditorBuffer, &end, line_to_go);
  
  show_on_screen (GTK_TEXT_VIEW (FPROPS(CurrentPage, Text)), start);

  if (gtk_text_iter_equal (&start, &end)) /* not enough lines */
    gtk_text_buffer_get_end_iter (EditorBuffer, &end);
  
  gtk_text_buffer_place_cursor (EditorBuffer, &start);
  gtk_text_buffer_move_mark_by_name (EditorBuffer, "selection_bound", &end);
  
  msg = g_strdup_printf ("Line %d selected...", gtk_text_iter_get_line (&start) + 1);
  print_msg (msg);
  g_free (msg);
}


void goto_line_window_not_visible (void)
{
  GotoLineIsVisible = FALSE;
}


void goto_line (void)
{
  GtkWidget *GotoLineWindow;
  GtkWidget *HBox;
  GtkWidget *Label;
  GtkWidget *Button;
  GtkObject *Size;
  GtkWidget *SpinButton;

  if (GotoLineIsVisible) return;
  GotoLineWindow = gtk_dialog_new ();
  gtk_window_set_title (GTK_WINDOW(GotoLineWindow), "Goto Line..."); 
  gtk_window_set_resizable (GTK_WINDOW(GotoLineWindow), FALSE);
  g_signal_connect (G_OBJECT(GotoLineWindow), "delete_event",
		      (GtkSignalFunc) goto_line_window_not_visible, NULL);
  g_signal_connect_swapped (G_OBJECT(GotoLineWindow), "delete_event",
			     (GtkSignalFunc) gtk_widget_destroy,
			     G_OBJECT(GotoLineWindow));
  g_signal_connect (G_OBJECT (GotoLineWindow), "destroy",
		      (GtkSignalFunc) goto_line_window_not_visible, NULL);
  g_signal_connect_swapped (G_OBJECT (GotoLineWindow), "destroy",
			     (GtkSignalFunc) gtk_widget_destroy,
			     G_OBJECT(GotoLineWindow));
  HBox = gtk_hbox_new (FALSE, 0);
  gtk_container_set_border_width (GTK_CONTAINER(HBox), 10);
  gtk_box_pack_start (GTK_BOX(GTK_DIALOG(GotoLineWindow) -> vbox),
		      HBox, FALSE, FALSE, 0);
  Label = gtk_label_new ("Line :");
  gtk_box_pack_start (GTK_BOX(HBox), Label, FALSE, FALSE, 0);
  Label = gtk_label_new ("");
  gtk_box_pack_start (GTK_BOX(HBox), Label, FALSE, FALSE, 3);
  Size = gtk_adjustment_new (1, 1, 0xffff, 1, 5, 0);
  SpinButton = gtk_spin_button_new (GTK_ADJUSTMENT(Size), 1, 0);
  gtk_entry_set_max_length (GTK_ENTRY(SpinButton), 5);
  gtk_entry_set_activates_default (GTK_ENTRY(SpinButton), TRUE);
  gtk_spin_button_set_numeric (GTK_SPIN_BUTTON(SpinButton), TRUE);
  gtk_box_pack_start (GTK_BOX(HBox), SpinButton, TRUE, TRUE, 0);
  Button = gtk_button_new_from_stock (GTK_STOCK_CANCEL);
  g_signal_connect (G_OBJECT(Button), "clicked",
		      (GtkSignalFunc) goto_line_window_not_visible, NULL);
  g_signal_connect_swapped (G_OBJECT(Button), "clicked",
			     (GtkSignalFunc) gtk_widget_destroy,
			     G_OBJECT(GotoLineWindow));
  GTK_WIDGET_SET_FLAGS (Button, GTK_CAN_DEFAULT);
  gtk_box_pack_start (GTK_BOX(GTK_DIALOG(GotoLineWindow) -> action_area),
		      Button, TRUE, TRUE, 0);
  Button = gtk_button_new_from_stock (GTK_STOCK_JUMP_TO);
  g_signal_connect_swapped (G_OBJECT(Button), "clicked",
		      (GtkSignalFunc)set_line, SpinButton);
  g_signal_connect (G_OBJECT(Button), "clicked",
		      (GtkSignalFunc) goto_line_window_not_visible, NULL);
  g_signal_connect_swapped (G_OBJECT(Button), "clicked",
			     (GtkSignalFunc) gtk_widget_destroy,
			     G_OBJECT(GotoLineWindow));
  GTK_WIDGET_SET_FLAGS (Button, GTK_CAN_DEFAULT);
  gtk_box_pack_start (GTK_BOX(GTK_DIALOG(GotoLineWindow) -> action_area),
		      Button, TRUE, TRUE, 0);
  gtk_widget_grab_default (Button);
  gtk_window_set_focus (GTK_WINDOW(GotoLineWindow), SpinButton);
  gtk_widget_show_all (GotoLineWindow);
  print_msg ("Display Goto Line window...");
  GotoLineIsVisible = TRUE;
}
