/*
 *                    BioJava development code
 *
 * This code may be freely distributed and modified under the
 * terms of the GNU Lesser General Public Licence.  This should
 * be distributed with the code.  If you do not have a copy,
 * see:
 *
 *      http://www.gnu.org/copyleft/lesser.html
 *
 * Copyright for this code is held jointly by the individual
 * authors.  These should be listed in @author doc comments.
 *
 * For more information on the BioJava project and its aims,
 * or to join the biojava-l mailing list, visit the home page
 * at:
 *
 *      http://www.biojava.org/
 *
 * Created on 28.04.2004
 * @author Andreas Prlic
 *
 */
package org.biojava.bio.structure;


/**
 * Implementation of an Atom of a PDB file.
 * currently the coordinates of an atom are represented by a doubl[3] array.
 * @author Andreas Prlic
 * @since 1.4
 * @version %I% %G%
 */
public class AtomImpl implements Atom {
    
    String name     ;
    String fullName ;
    double[] coords ;
    String pdbline  ;
    int pdbserial   ;
    
    double occupancy ;
    double tempfactor;
    
    Character altLoc ;
    Group parent;
    
    public AtomImpl () {
        name     = null        ;
        fullName = null        ;
        coords   = new double[3];
        pdbline  = ""          ;  
        occupancy  = 0.0       ;
        tempfactor = 0.0       ;
        altLoc = new Character(' ');
        parent = null;
    }
    
    /** trimmed version of atom name, e.g. "CA" 
     * @see #getName
     */
    public void   setName(String s) { name = s ;}
    
    /**
     * Gets this object's name.
     * @return a String representing the name value
     * @see #setName
     */
    public String getName()         { return name ;}
    
    /** set full name of atom e.g. " CA " .
     * @see #getFullName
     */
    public void   setFullName( String s ) { fullName =s ; }
    
    /** get full name of atom e.g. " CA ". 
     * @return a String representing the full name value
     * @see #setFullName
     */
    public String getFullName()           { return fullName; }
    
    /** set PDB atom number. 
     * @see #getPDBserial
     */
    public void setPDBserial(int i) { pdbserial = i    ; }
    
    /** get PDB atom number.
     * @see #setPDBserial
     */
    public int  getPDBserial()      { return pdbserial ; }
    
    /** the coordinates.
     * @see #getCoords 
     */    
    public void     setCoords( double[] c ) { coords = c   ; } 
    
    /** get the coordinates as a double[3] array .
     * @return an array of doubles representing the coords value
     * @see #setCoords     
     */
    public double[] getCoords()            { return coords ; }
    
    public void setX(double x) {
        coords[0] = x ;
    }
    public void setY(double y) {
        coords[1] = y ;
    }
    public void setZ(double z) {
        coords[2] = z ;
    }
    
    /** get the X coordinate. */
    public double getX() { return coords[0]; }
    
    /** get the Y coordinate. */
    public double getY() { return coords[1]; }
    
    /** get the Z coordinate. */
    public double getZ() { return coords[2]; }
    
    /** set alternate Location. 
     * @see #getAltLoc
     */
    public void setAltLoc(Character c) {
        altLoc = c ;
    }
    /** get alternate Location.
     * @return a Character object representing the alt loc value
     * @see #setAltLoc
     */
    public Character getAltLoc() {
        return altLoc ;
    }
    
    
    /** store the whole line. 
     * @see #getPDBline
     */
    public void   setPDBline(String s) { pdbline = s;}
    
    /** get the whole line .
     * @return a String representing the PDBline value
     * @see #setPDBline
     */
    public String getPDBline() { return pdbline ;}
    
    /** string representation. */
    public String toString() {
        String str = name +" "+ pdbserial + " " + coords[0] + " " + coords[1] + " " + coords[2];
        return str ;
    }
    
    public void   setOccupancy(double occu){ occupancy = occu ;} ;
    public double getOccupancy(){ return occupancy; } ;
    
    public void   setTempFactor(double temp){ tempfactor = temp ;} ;
    public double getTempFactor(){ return tempfactor; } ;
    
    /** returns and identical copy of this  object .
     * @return  and identical copy of this  object 
     */
    public Object clone() {
        AtomImpl n = new AtomImpl();
        n.setOccupancy(getOccupancy());
        n.setTempFactor(getTempFactor());
        n.setAltLoc(getAltLoc());
        double[] coords = getCoords();     
        n.setX(coords[0]);
        n.setY(coords[1]);
        n.setZ(coords[2]);
        n.setPDBserial(getPDBserial());
        n.setFullName(getFullName());
        n.setName(getName());
        return n ;
    }
    
    public void setParent(Group parent) {
        
        this.parent = parent;
    }
  
    public Group getParent() {
        return parent;
       
    }   
    
    
}
