module Eval where

import Types
import Parser(parseProg, staticCheck, parseExp)
import Utils(pluralise, arityComplaint)
import Monad(when)
import Char(toLower)          -- for ghc compatibility

newtype Interpreter a = I (State -> (a, State))
-- recall that newtype is just like data except saves some overhead :)

type State = (Prog, Int, Maybe ErrorMsg)
type VarName = String
type Env = [(VarName, Int)]

-- the plumbing of the interpreter
instance Monad Interpreter where 
    return a     
        = I (\state -> (a, state))    -- maintain state and return value

    (I comp) >>= fun
        = I (\state -> let (result, newState) = comp state
                           I comp' = fun result
                       in  
                       case newState of (_,_,Just _) -> (undefined, newState)
                                        _            -> comp' newState)
    -- we only continue with the calculations if there isn't an error
    -- so the calculation 1/0 + undef doesn't get up to seeing that undef is 
    -- undefined: the div by 0 error is immediately passed onto the final state
                   
        
    
runI :: State -> Interpreter a -> (a, State)
runI startState (I comp) = comp startState

initState :: Prog -> State
initState prog = (prog, initComputations, initErrorStatus)
    where initComputations = 0
          initErrorStatus = Nothing

{--------------------------------------------+
| Getting state information out of the monad |
+--------------------------------------------}

getProg :: Interpreter Prog
getProg = I (\state@(prog, _, _) -> (prog, state))

getOperations :: Interpreter Int
getOperations = I (\state@(_, num, _) -> (num, state))

getErrorMsg :: Interpreter (Maybe ErrorMsg)
getErrorMsg = I (\state@(_, _, msg) -> (msg, state))


{------------------------------------------+
| Modifying state information in the monad |
+------------------------------------------}

{- add 1 to the number of operations carried out -}
incOps :: Interpreter ()
incOps = I (\(prog, num, errors) -> ((), (prog, num + 1, errors)))

{- set an error message and make the result of the computation undefined, so 
   any attempt to access will result in an error -}
setErrorMsg :: ErrorMsg -> Interpreter a
setErrorMsg msg 
    = I (\(prog, num, errors) -> 
          (undefined, (prog, num, 
                       Just msg))
                       {- case errors of {Nothing -> Just msg; _ -> errors})) -}
        )
      -- we don't want to clobber existing errors, so we only set the msg if 
      -- there isn't already an error message


clearErrorMsg :: Interpreter () 
clearErrorMsg = I (\(prog, num, errors) -> ((), (prog, num, Nothing)))


                     {-----------------------------------+
                     | The Evaluation Part of the Module |
                     +-----------------------------------}

run :: IO ()
run 
    = do
      let welcome = "Welcome to funlang, the FUNctionaL progrAmmiNG environment"
          -- I bet you thought that funlang was short for Functional Language!
          len        = length welcome

      {- welcome message -}
      putStrLn (replicate len '*')
      putStrLn welcome
      putStrLn (replicate len '*')
      putStrLn ""

      {- get script name -}
      putStr   "Script: "
      filename <- getLine  
      -- Here we're not very robust: if the file can't be read/opened, just bomb
      unparsed <- readFile filename  
      let program = parseProg unparsed

      {- check for static correctness of program -}
      correct  <- staticCheck program
      case correct of 
           True -> mainLoop program   -- no static errors
           False -> putStrLn  
                        "Sorry, there were static errors so I can't continue"

exitString :: String -> Bool
exitString str = map toLower str `elem` ["quit", "bye", "exit", ":q", ":quit"]

{- this is the main read-eval-print loop that the program stays in until one
   of the exit strings (listed above) is entered -}
mainLoop :: Prog -> IO ()
mainLoop prog
    = do 
      putStr "> " 
      cmd <- getLine
    
      if exitString cmd then putStrLn "Good bye!" else
          do
          let exp = parseExp cmd 
              emptyEnv = []
              (result, state) = runI (initState prog) (eval emptyEnv exp)

          case state of 
               (_, _, Just error) -> do
                                     putStrLn "*** Runtime Error Occurred ***"
                                     putStrLn error
               (_, ops, Nothing) ->  do
                                     putStrLn (show result)
                                     putStrLn (pluralise ops "operation")

          putStrLn ""
          mainLoop prog     -- repeat the loop


{- this function is very general, but allows us to define many cases for
   eval and beval in terms of it, as it's a common theme: for an operator,
   pass in a way of evaluating the two arguments and a way of combining them,
   get the result -}
opEval :: (exp -> Interpreter a) -> (a -> a -> b) -> 
           exp -> exp -> Interpreter b   -- exp <- {Exp, Bexp} a <- {Int, Bool}
opEval evaluate op x y
    = do
      val1 <- evaluate x       -- evaluate x
      val2 <- evaluate y       -- evaluate y
      incOps                   -- increase number of operations counter
      return (val1 `op` val2)  -- return result


eval :: Env -> Exp -> Interpreter Int
eval _   (Num x)   = return x
eval env (Add x y) = opEval (eval env) (+) x y
eval env (Sub x y) = opEval (eval env) (-) x y
eval env (Mul x y) = opEval (eval env) (*) x y

{- Can't use opEval here because of div by zero problem
   Note: we don't evaluate y first as this could make for confusion in something
         like undefined(x,y) / 0, whereas the convention is the leftmost error
         in the parse tree gets reported -}
eval env (Div x y)
    = do
      numer <- eval env x 
      denom <- eval env y 
      incOps             
      if denom == 0 then setErrorMsg "Attempted division by zero!" 
                    else return (numer `div` denom)
eval env (If cond x y)
    = do
      truth <- beval env cond    -- evaluate conditional and evaluate 
      if truth then eval env x   -- the right subexpression accordingly
               else eval env y

eval env (Var varname)
    = case lookup varname env of
           Just val -> return val  -- the usual case (hopefully!)

           Nothing  -> setErrorMsg $ "Undefined variable: " ++ varname
           {- can't happen in a statically checked program, but can happen
              at the command line if you ask to just evaluate, say, x + 4 -}

{- the messy case, but we must do some error checking here -}
eval env (App fun args)
    = do
      program <- getProg    -- we have to lookup the function in the program
      case lookup fun program of
          Just (formalArgs, functionDefn) ->
              if length formalArgs == length args    -- check arity first
              then                                   -- the non-error case 
              do args' <- mapM (eval env) args       -- evaluate the arguments
                 let newEnv = zip formalArgs args'   -- set up new environment
                 eval newEnv functionDefn
              else 
                 setErrorMsg $ arityComplaint fun (length formalArgs) 
                                              (length args)
          Nothing -> setErrorMsg $ "Call to undefined function: " ++ fun


{- beval evaluates Bexps, but is otherwise like eval -}
beval :: Env -> Bexp -> Interpreter Bool
beval env (Or  x y)      = opEval (beval env) (||) x y
beval env (And x y)      = opEval (beval env) (&&) x y
beval env (Rel x comp y) = opEval (eval env) compFun x y
    where
    {- define the operation comp' in terms of the value of comp -}
    compFun = case comp of 
                   Equal     -> (==)
                   NEqual    -> (/=)
                   Less      -> (<)
                   LessEq    -> (<=)
                   Greater   -> (>)
                   GreaterEq -> (>=)
    {- recall that with the definition of opEval these turn into things
       like: 
              do
                  val1 <- eval env x  
                  val2 <- eval env y 
                  incOps        
                  return (val1 `compFun` val2)    -- e.g. val1 <= val2
       which is exactly what we want in evaluating a comparison relation -}
