module Parser(parseProg, staticCheck, parseExp) where

import Types
import Utils(findDuplicate, arityComplaint, matchingBrackets)
import ParseLib           -- Hutton and Meijer's parsing library 
import Maybe(listToMaybe)
import Monad(msum)
import Char(isSpace)      -- for ghc compatibility

{------------------+
| Static Semantics |
+------------------}

{- check the static semantics of program, display error messages and return
   True if everything's okay, False otherwise. It's recommended that it this
   function returns False, the whole compilation fails -}
staticCheck :: Prog -> IO Bool
staticCheck prog
    = do
      let result = case firstPass prog of 
                        Left synTree -> secondPass prog synTree 
                        Right error -> Just error
          indent = unlines . map ("  " ++) . lines
      case result of 
           Just error -> do
                         putStrLn "************************************"
                         putStrLn ""
                         putStrLn "Error in static semantics of program:"
                         putStrLn (indent error)
                         putStrLn "************************************"
                         return False
           Nothing    -> do
                         putStrLn "No static semantics errors in program"
                         return True



-- from a list of possible errors, return Just first, otherwise return Nothing
firstError :: [Maybe ErrorMsg] -> Maybe ErrorMsg
firstError = msum   

{- check to make sure no duplicate definitions of functions, and return
   a list of (function, arity) pairs -}
firstPass :: Prog -> Either [(FunName,Arity)] ErrorMsg
firstPass prog
    = case findDuplicate (map fst funArities) of 
           Nothing -> Left funArities  -- no duplicate definitions: good
           Just def -> Right $ "Multiple definitions of function " ++ def
    where
    funArities = [(fun, length args) | (fun, (args,_)) <- prog]


{- given a program and a list of function/arities report first error found -}
secondPass :: Prog -> [(FunName,Arity)] -> Maybe ErrorMsg
secondPass prog funArities 
    = firstError (map errorsCheck prog)
    where
    {- given one definition, see if there are any errors with it -}
    errorsCheck :: FunEqn -> Maybe ErrorMsg
    errorsCheck (fun, (params, expr))
        = fmap (\err -> "In definition for function " ++ fun ++ " --\n" ++ err) 
               (firstError [functionCall, defnVars, diffFormalVars])
          where
          -- all variables in the definition must be formal variables
          defnVars = listToMaybe
              [ "Illegal use of non-formal variable " ++ var
                | var <- getVars expr, var `notElem` params]
          

          -- formal variables must be different in binding
          diffFormalVars   
             = fmap (\var -> "Repeated var in pattern: "++ var)
                    (findDuplicate params)


          -- check to see that the function has right arity, and a 
          -- definition for it exists: just return first bad function call
          -- that we encounter in the parse tree
          functionCall = 
              firstError [ interpret f (lookup f funArities) callArity | 
                           (f, callArity) <- getFuns expr]


          -- for functionCall, neaten up the output to check for incorrect
          -- number of parameters, or call to undefined functions:
          interpret f Nothing _ = Just ("Call to undefined function " ++ f)
          interpret f (Just arity) callArity
              = if arity == callArity then Nothing
                else Just (arityComplaint f arity callArity)

{----------------------------------+
| Frontend functions to the parser |
+----------------------------------}

{- given the parser, make the frontend function that does the parser
   application to any string it's given. We also pass in progParser :: Bool
   which tells us whether we were trying to parse a Prog (if not, then we
   are trying to parser an expression -- needed for useful error msgs -}
frontend :: Show a => Parser a -> Bool -> (String -> a)
frontend parser progParser inp
    = case papply parser inp of
           [(tree, "")] -> tree
           [(_, rest)]  -> pError inp $ "Couldn't finish parsing " ++ rest
           []           -> pError inp $ "Couldn't successfully parse input"
           list@(_:_)   -> pError inp $ "Ambiguous parse:\n" ++ show list
    where
    {- provide an error message and a useful diagnosis of the problem -}
    pError :: String -> String -> a
    pError source errorMsg
        = error (errorMsg ++ "\nPossible cause(s)\n" ++ replicate 30 '*' ++ "\n" ++ helpfulMsg)
        where

        -- do our brackets match?
        bracketCount = not (matchingBrackets source)
        
        -- is last non-space character a semicolon?
        semicolon  = (/= ';') . head . dropWhile isSpace . reverse $ source
        
        -- did we mean to have a definition, and just forget?
        definition = any (== '=') source   
        helpfulMsg 
            = makeMsg [(progParser && semicolon, 
                            "Semicolon missing?"),
                       (progParser && not definition,   
                            "Do you mean not to define anything?"),
                       (bracketCount, 
                            "Unmatched brackets?")]
              {-  Note: The first two suggestions are only useful if we were
                        parsing a program rather than an expression  -}
        makeMsg poss = unlines [hint | (condition, hint) <- poss, condition]


{- take a program as a large string and return syntax tree -}
parseProg :: String -> Prog
parseProg = frontend parseProg' True  -- True since we're parsing a Prog

parseExp :: String -> Exp
parseExp = frontend parseExp' False  -- False since we're parsing Exp, not Prog



{----------------------------------+
| The parsing functions themselves |
+----------------------------------}

parseProg' :: Parser Prog
parseProg'
    = do
      junk -- clear any leading whitespace
      properProgram +++ return []
      where
      properProgram 
          = do
            prog <- parseEqn `sepby` (symbol ";")
            symbol ";"  -- consume last semicolon
            return prog
      
parseEqn :: Parser FunEqn
parseEqn
    = do 
      funName <- tokIdent             -- get the function name
      args    <- parseArgs tokIdent   -- get its arguments (which are idents)
      symbol "="
      exp     <- parseExp'             -- get the definition
      return (funName, (args, exp))


parseExp' :: Parser Exp
parseExp' 
    = do 
      term <- parseTerm
      let parseAdd = operation "+" parseExp' (Add term)
          parseSub = operation "-" parseExp' (Sub term)  
      parseAdd +++ parseSub +++ return term

parseBexp :: Parser Bexp
parseBexp
    = do
      bterm <- parseBterm
      let parseOr = operation "||" parseBexp (Or bterm)
      parseOr +++ return bterm

parseBterm :: Parser Bexp
parseBterm
    = do
      bfactor <- parseBfactor
      let parseAnd = operation "&&" parseBterm (And bfactor)
      parseAnd +++ return bfactor

parseBfactor :: Parser Bexp
parseBfactor
    = bracketed parseBexp +++ parseComparison

parseComparison :: Parser Bexp 
parseComparison
    = do
      exp1 <- parseExp'
      op   <- firstSuccess compParsers  -- take disjunct of all comparison 
      exp2 <- parseExp'                 -- parsers (return first successful one)
      return (Rel exp1 op exp2) 
      where
        {- make a parser for each operation and find first successful one -}
        compParsers = zipWith makeParser
                          ["==", "/=", "<=", "<", ">=", ">"]
                          [Equal, NEqual, LessEq, Less, GreaterEq, Greater]
        makeParser sym value = do {symbol sym; return value}

        firstSuccess = foldr1 (+++)   -- this is really just msum, except
                                      -- we use +++ rather than mplus

parseTerm :: Parser Exp
parseTerm 
    = do
      factor <- parseFactor
      let parseMult = operation "*" parseTerm (Mul factor)
          parseDiv  = operation "/" parseTerm (Div factor) 
      parseMult +++ parseDiv +++ return factor


parseFactor :: Parser Exp
parseFactor 
    = parseIfThenElse +++ parseFunOrVar +++ parseNatural +++ bracketed parseExp'
    where
    parseNatural = do n <- natural 
                      return (Num n)


{- left-factored grammar yields this parser which parses an identifier, then
   sees if it can parse a function. If so, return the App, else return a Var -}
parseFunOrVar :: Parser Exp
parseFunOrVar
    = do 
      name <- tokIdent
      let parseApp = operation "" (parseArgs parseExp') (App name)  
          -- function application is considered an invisible "" operation :)
      parseApp +++ return (Var name)


{- succeeds on a well-formed if-then-else expression -}
parseIfThenElse :: Parser Exp
parseIfThenElse 
    = do
      symbol "if" 
      bexp    <- parseBexp
      symbol "then"
      thenExp <- parseExp'
      symbol "else"
      elseExp <- parseExp'
      return (If bexp thenExp elseExp)


{- Parse function parameters like (a,b,c), () or (3+2, x - 3) This is
   paramaterised to allow parsing of formal args (with tokIdent) and also
   arguments consisting of expressions (with parseExp'). NB: unlike the
   grammar for args, this parser does not accept eps: it's more useful to have
   the callers take care of that -}
parseArgs :: Parser a -> Parser [a]
parseArgs argParser
    = bracketed (argParser `sepby` (symbol ","))  -- 0 or more parameters



                          {-------------------------+
                          | Useful Parser Utilities |
                          +-------------------------}

{- a more convenient version of the function bracket in ParseLib, only works
   for "(" and ")", which are the only brackets in the language -}
bracketed :: Parser a -> Parser a
bracketed parser = bracket (symbol "(") parser (symbol ")") 


{- token parser for getting an identifier -}
tokIdent :: Parser String
tokIdent = token ident


{- take a string representing an operation, a parser to get the content
   after the op, and a function to apply to this content. Note: because of
   left-factoring this assumes that the first argument has already been
   consumed, so the first thing expected is the op string -}
operation :: String -> Parser a -> (a -> b) -> Parser b
operation op parser fun 
    = do
      symbol op
      secondArgument <- parser
      return (fun secondArgument)
